{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

program Binary_To_Assembly_Converter;

const
    PrefixHex = '$';
    TAB = #9;
    AutoSize : boolean = false;
    Comments : boolean = false;

type
    TBytes = array[0..$FFFE] of byte;
    TWords = array[0..$7FFE] of word;
    TLongs = array[0..$3FFE] of longint;

function NextPos ( SubStr : String; S : String; StartPos : byte ) : byte;
var
    StrPtr : ^String;
    TPos   : Byte;
begin
    if (StartPos = 0) or (StartPos > Length(S)) then
        NextPos := 0
    else begin
        Dec(StartPos);
        S[StartPos] := Char(Length(S) - StartPos);
        StrPtr := @S[StartPos];
        TPos := Pos(SubStr, StrPtr^);
        if TPos > 0 then Inc(TPos, StartPos);
        NextPos := TPos;
    end;
end;

function LastPos( SubStr, S : String ) : byte;
var
    T, N : Byte;
begin
    N := 0;
    repeat
        T := N;
        N := NextPos(SubStr, S, N + 1);
    until N = 0;
    LastPos := T;
end;

function ByteHex(const B) : String;
const
    Hex : string[16] = '0123456789abcdef';
begin
    ByteHex := Hex[(Byte(B) shr 4) + 1] + Hex[(Byte(B) and $f) + 1];
end;

function  WordHEX (const W) : String;
begin
    WordHex := ByteHEX(Hi(Word(W))) + ByteHex(Lo(Word(W)));
end;

function HiWord(const L) : word;
begin
    HiWord := TWords(L)[1];
end;

function LoWord(const L) : word;
begin
    LoWord := TWords(L)[0];
end;

function  LongHEX (const L) : String;
var
    LH, LL : Word;
begin
    LH:= HiWord(L);
    LL:= LoWord(L);
    LongHex := WordHEX(LH) + WordHex(LL);
end;

function HexStr(V : LongInt; Z : Integer) : String;
begin
  case Z of
  	1 : HexStr:=ByteHex(V);
  	2 : HexStr:=WordHex(V);
  	4 : HexStr:=LongHex(V);
  else
    HexStr:='ERR!';
  end;
end;

function UCase(S : String) : String;
var
    I : integer;
begin
    for I := 1 to Length(S) do
        if (S[I] >= 'a') and (S[I] <= 'z') then
            S[I] := Chr(Ord(S[I]) - $20);
    UCase := S;
end;

function NewName (Filename, Extension : String) : String;
var
    D : String;
    B : String[16];
    N : byte;
begin
    D := FileName;
    N := LastPos('\', D);
    B := Copy(D, N + 1, 255);
    D[0] := Char(N);
    N := LastPos('.', B);
    if N = 0 then begin
        B := B + '.'
    end else begin
        B[0] := Char(N);
    end;
    NewName := D + B + Extension;
end;

function ArrayName (Filename : String) : String;
var
    D : String;
    N : byte;
begin
    D := FileName;
    N := LastPos('\', D);
    D := Copy(D, N + 1, 255);
    N := LastPos('.', D);
    ArrayName := 'bin_' + Copy(D, 1, N - 1);
end;


procedure Convert(Filename: String);
var
    Buffer : array[0..15] of byte;
var
    OutName     : String;
    FI          : File;
    FO          : Text;
    FP, FS      : LongInt;
    FE, FC, FR  : integer;
    I, IC, X, FX: integer;
    S, FZ, CS   : String;

begin
    OutName := NewName(FileName, 'INC');
    if Filename = OutName then begin
        WriteLn('destination for ', FileName, ' is same as source');
        Exit;
    end;
    Assign(FI, FileName);
    Reset(FI, 1);
    Assign(FO, OutName);
    Rewrite(FO);
    WriteLn(FO, 'const ' + '{ ', Filename, ' }');
    FS:=FileSize(FI);
    FZ:='byte';
    FX:=1;
    if AutoSize then begin
      if FS mod 2 = 0 then begin
        FZ:='word';
        FX:=2;
      end;
      if FS mod 4 = 0 then begin
        FZ:='longint';
        FX:=4;
      end;
    end;
    WriteLn(FO, '  ', ArrayName(FileName), ' : array [0..', FS - 1, '] of ', FZ, ' = (');
    { WriteLn(FO); }
    repeat
        FP := FilePos(FI);
        BlockRead(FI, Buffer, Sizeof(Buffer), FC);
        if FC = 0 then Break;
        I := 0;
        while FC > 0 do begin
            X := I;
            IC := 0;
			S := '  ' + PrefixHex + HexStr(Buffer[I], FX);
			Inc(I, FX);
			Inc(IC, FX);
			Dec(FC, FX);
			while FC >= FX do begin
				S := S + ',' + PrefixHex + HexStr(Buffer[I],FX);
				Inc(I, FX);
				Inc(IC, FX);
				Dec(FC, FX);
			end;
			if FilePos(FI) < FS then
			  CS := ','
			else
			  CS := ' ';
            if Comments then begin
                S := S + CS;
                while Length(S) < 50 do
                    S := S + ' ';
                S := S + '{' + WordHEX(FP) + ' ';
                Inc(FP, IC);
                for IC := 0 to IC - 1 do
                    if (Buffer[X + IC] < $20) or (Buffer[X + IC] > $7e)
                    or (Buffer[X + IC] = $7b) or (Buffer[X + IC] = $7d) then
                        S := S + '.'
                    else
                        S := S + Chr(Buffer[X + IC]);
                S:= S + '}';
            end else if CS <> ' ' then
              S:=S + CS;
            WriteLn(FO, S);
        end;
    until False;
    { WriteLn(FO); }
    WriteLn(FO, '  );');
    Close(FI);
    Close(FO);
    WriteLn('created ', OutName);
end;

procedure Help;
begin
    WriteLn('usage BIN2PAS: FILENAME');
    WriteLn('converts a binary file to pascal language include');
    WriteLn;
    WriteLn('   /A      Automatic data sizing (byte, word, longint)');
    WriteLn('   /C      Add comments');
    WriteLn;
    Halt(0);
end;

var
    I : integer;
    S : String;
begin
    if ParamCount = 0 then
        Help
    else begin
        for I := 1 to ParamCount do begin
            S := UCase(ParamStr(I));
            if S = '/A' then AutoSize := True
            else if S = '/C' then Comments := True
            else if S[1] = '/' then Help;
        end;
        for I := 1 to ParamCount do begin
            S := UCase(ParamStr(I));
            if S[1] <> '/' then
                Convert(S);
        end;
    end;
end.