; this file is part of ethflop, a floppy drive emulator over Ethernet
; Copyright (C) 2019-2024 Mateusz Viste
;
; contains FUNCTIONS (ie. things invoked with CALL and returning with ret)
;
; MIT license


; ============================================================================
; === FIND PACKET DRIVER =====================================================
; ============================================================================
; scan int vectors 60h..80h looking for a packet drvr sig. returns found
; address in ES:BX, or set CF on failure.
FINDPKTDRVR:
mov al, 0x5F   ; initial vect number to scan - 1
.TRYNEXTVECT:
inc al
cmp al, 0x80
ja short .PKTDRVRNOTFOUND     ; don't look past int 80h
mov ah, 0x35   ; AH=35h (GET INT VECT, DOS 2+) - vect is in AL already
int 21h        ; vector is at ES:BX now
; make sure offset is less than 0xfff5 - otherwise I would read over 0xffff
; when looking for the pktdrvr signature, potentially triggerring a GPF if
; we are running under EMM386 or some other protected mode thing.
; thx to ECM for spotting this <http://svardos.org/?p=forum&thread=1725345339>
cmp bx, 0xfff5
jae short .TRYNEXTVECT
; look for the 'PKT DRVR' sig, ie. 4x WORDS: 0x4B50 0x2054 0x5244 0x5256
cmp word [es:bx+3], 0x4b50 ; 'PK'
jne short .TRYNEXTVECT
cmp word [es:bx+5], 0x2054 ; 'T '
jne short .TRYNEXTVECT
cmp word [es:bx+7], 0x5244 ; 'DR'
jne short .TRYNEXTVECT
cmp word [es:bx+9], 0x5256 ; 'VR'
jne short .TRYNEXTVECT
; FOUND!
clc                     ; clear CF (no error, valid pkt drv ptr in ES:BX)
ret
.PKTDRVRNOTFOUND:
stc                     ; CF set to indicate error
ret


; ============================================================================
; === FIND TSR ===============================================================
; ============================================================================
; looks for ethflop tsr, on success, it will return:
;   TSR's seg in ES (PSP at ES:00, entry point at ES:BX)
; on error, CF is set and ES:BX points to the current int 13h handler
; this function destroys AX, BX, CX, SI, DI, ES and FLAGS
FINDTSR:
; get int 13h vector into ES:BX
mov ax, 0x3513  ; AH=35h (GET INT VECT, DOS 2+), AL=13h (interrupt number)
int 21h         ; ES = segment, BX = offset
; is it me? (look for the 5-bytes 'EFLOP' signature)
lea di, [bx - (INTHANDLERBIOS - STR_SIG)] ; my sig is above int entry point
mov si, STR_SIG
mov cx, 5
cld                ; cmpsb will move forward
repe cmpsb         ; compares CX bytes at ES:DI and DS:SI, CX is 0 if matching
clc                ; assume success
je short .FINDTSRFOUND ; jump if sig matched
stc
.FINDTSRFOUND:
ret


; ============================================================================
; === RELEASE PKTDRVR HANDLE =================================================
; ============================================================================
; requires config block segment to be set in ES
; destroys ax and bx
PKTDRVR_RELEASE:
push es
mov ah, 3     ; AH=3 is release_type()
mov bx, [es:PKTDRVRHANDLE]
; simulate an int call
pushf
cli
call far [es:PKTDRVR]
pop es
ret


; ============================================================================
; === SEND CONTROL QUERY TO SERVER AND PRINT RESULT ==========================
; ============================================================================
; es must be set to the tsr's config block
; returns with CF set on error, clear on success
SENDSRVQUERY:
; make sure that DS = CS
push cs
pop ds
; copy hdr area from TSR
mov cx, (HDR_END - HDR_DMAC) / 2
xor bx, bx
.COPYWORD:
mov ax, [es:HDR_DMAC+bx]
mov [HDR_DMAC+bx], ax
inc bx
inc bx
loop .COPYWORD
; copy pktdrvr addr
mov ax, [es:PKTDRVR]
mov [PKTDRVR], ax
mov ax, [es:PKTDRVR+2]
mov [PKTDRVR+2], ax
; set ethertype to 'control'
mov word [HDR_ETYPE], 0xDCEF   ; 0xEFDC in net byte order
; register a handle for control ethertype
call PKTDRVRREGISTER
jnc short .PKTDRVINITOK
; pktdrv init failed
mov ah, 0x09
mov dx, STR_PKTDRVRINITFAIL
int 21h
mov ax, 0x4C05
int 21h
.PKTDRVINITOK: ; init ok, handle acquired
; set reqid to some random value
call GETCURTICK     ; tick val in BL now
mov [HDR_REQID], bl
; copy cmdline len and body to packet data
mov cx, 128       ; copy 128 words (256 bytes)
xor bx, bx
.COPYWORD2:
mov ax, [0x80+bx]
mov [PKT_DATA+bx], ax
inc bx
inc bx
loop .COPYWORD2
; send
call PKTDRVR_SEND
pushf     ; save flags so I can check later if send succeeded
push es
push ds
pop es
call PKTDRVR_RELEASE   ; es must point to my local cfg block
pop es
popf
jnc short .GOTANSWER
; err
mov ah, 0x09
mov dx, STR_SERVERUNREACHABLE
int 0x21
stc
ret
; print received answer
.GOTANSWER:
mov ah, 0x09
mov dx, PKT_DATA
int 0x21
; update TSR's FLOPID
mov ax, [PKT_AX]
mov [es:HDR_FLOPID], ax
mov ax, [PKT_BX]
mov [es:HDR_FLOPID+2], ax
mov ax, [PKT_CX]
mov [es:HDR_FLOPID+4], ax
mov ax, [PKT_DX]
mov [es:HDR_FLOPID+6], ax
; end of story
clc
ret


; ============================================================================
; === REGISTER ETHERTYPE RECEIVING ROUTINE ===================================
; ============================================================================
; requires DS:HDR_ETYPE to contain the ethertype value
; DS:PKTDRVR must contain a valid far pointer to the pktdrvr routine
; returns CF set on error, otherwise sets [PKTDRVRHANDLE] with the handle
PKTDRVRREGISTER:
push ax
push bx
push cx
push dx
push si
push di
push es
; init packet driver (register a handle)
mov ax, 0x0201           ; AH=access_type()   AL=ifclass=1(eth)
mov bx, 0xffff           ; if_type=0xffff (all)
mov dl, 0                ; if_number=0 (first interface available)
mov si, HDR_ETYPE ; DS:SI points to the ethertype val in network byte order
mov cx, 2                ; typelen (ethertype is 16 bits)
push cs                  ; ES:DI must point to the frame-receiving routine
pop es
mov di, PKTDRVR_RECV
; simulate an 'int' call
pushf
cli
call far [PKTDRVR] ; now pkthandle should be in AX, or CF set on error
mov [PKTDRVRHANDLE], ax     ; save my precious handle
pop es
pop di
pop si
pop dx
pop cx
pop bx
pop ax
ret


; ============================================================================
; === PRINT AL ON SCREEN AS TWO HEX DIGITS ===================================
; ============================================================================
PRINT_AL_HEX:
push ax
push bx
push dx
; ------
aam 0x10           ; secret knowledge: this splits AL nibbles into AH and AL
mov bx, .HEXTABLE  ; preset bx to my lookup table (for xlatb)
xlatb              ; AL = DS:[BX + unsigned AL]
xchg  ah, al       ; do the high nibble now
xlatb              ; both nibbles are translated (but swapped with each other)
; use DOS to print the characters in al and ah
mov dx, ax
mov ah, 0x02       ; DOS 1+ print DL on screen
int 21h
xchg dl, dh        ; now do the other nibble
int 21h
; ------
pop dx
pop bx
pop ax
ret
.HEXTABLE db "0123456789ABCDEF"


; ============================================================================
; === PRINT AX ON SCREEN AS FOUR HEX DIGITS ==================================
; ============================================================================
PRINT_AX_HEX:
xchg ah, al
call PRINT_AL_HEX
xchg ah, al
call PRINT_AL_HEX
ret
