{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$IFDEF SECTINT}

function AssetLoad(Name : String; Source : word; var P : PAsset) : boolean;
function AssetLoadAll(WildCard : String; Lock : boolean) : boolean;
function AssetLoadOrDie(Name : String; Source : word; var P : PAsset) : pointer;

procedure AssetLock(Asset : PAsset);
procedure AssetUnlock(Asset : PAsset);


function AssetPrint(P : PAsset) : boolean;

procedure AssetDispose(var Asset : PAsset);
procedure AssetDisposeAll;
procedure AssetIndexFree(FileName : String);
procedure AssetIndex(FileName : String);
procedure AssetIndexSys(SysPath, AssetPath : String); { EXE PATH is
automatically included as AssetPath ''. For example, to index a SUBDIR called
FONTS. SYSPATH='FONTS\' and maybe AssetPath='FNTS\' }

function NewAssetFile(FileName : String) : PAssetFile;
function NewAsset(AssetFile : PAssetFile; Name : String) : PAsset;

{$IFOPT D+}
procedure PrintAssets; { temporary debugging tool }
{$ENDIF}

{$ENDIF}

{$IFDEF SECTIMP}

type
    TResourceBlockID = record   { size 20 bytes }
        BlockType  : word;       { = 0 }
        HeadBlock  : LongInt;    { = File Position of first block in chain }
        LastSize   : Longint;    { = Last size of file when footer was written }
        Identifier : array[0..9] of byte; { = 'XBINRSRCv2' }
    end;
    TResourceBlockData = record
        BlockType : word;        { = 1 }
        BlockSize : LongInt;     { Total size of block }
    	Attr      : byte;
    	Time      : LongInt;
    	Size      : LongInt;
    end;
    TResourceBlockName = record
    	Name      : String;
    	Zero      : byte;        { = 0, Null terminator }
    end;
    TResourceBlock = record        { Maximum Sized File Block Header And Search }
        Data : TResourceBlockData;
        Name : String;
        Zero : byte;
    end;

var
    MaxOpenFiles : word;
    CurOpenFiles : word;
    AssetFiles   : PList;
    Assets       : PList;

function NewAssetFile(FileName : String) : PAssetFile;
var
    P : PAssetFile;
begin
    FileName := UCase(FileName);
    P := nil;
    if FileName <> '' then
        P := New(PAssetFile);
    if Assigned(P) then begin
        P^.Name := StrPtr(FileName);
        if not Assigned(P^.Name) then begin
            Dispose(P);
            P := nil;
        end;
    end;
    if Assigned(P) then begin
        if AddList(AssetFiles, P) = nil then begin
            Dispose(P);
            P := nil;
        end;
    end;
    if Assigned(P) then begin
        P^.Open := False;
        Assign(P^.F, PtrStr(P^.Name));
    end;
    NewAssetFile := P;
end;

function NewAsset(AssetFile : PAssetFile; Name : String) : PAsset;
var
    P : PAsset;
begin
    Name := UCase(Name);
    P := nil;
    if Assigned(AssetFile) then
        P := New(PAsset);
    if Assigned(P) then begin
        FillChar(P^, Sizeof(TAsset), 0);
        P^.Name := StrPtr(Name);
        if not Assigned(P^.Name) then begin
            Dispose(P);
            P := nil;
        end;
    end;
    if Assigned(P) then begin
        if AddList(Assets, P) = nil then begin
            Dispose(P);
            P := nil;
        end;
    end;
    if Assigned(P) then begin
        P^.Resource := AssetFile;
        P^.Kind := FormatType(Name);
        if P^.Kind = ffUnknown then
            P^.Kind := ffBinary;
    end;
    NewAsset := P;
end;

procedure AssetLock(Asset : PAsset);
begin
    if not assigned(Asset) then exit;
    if Asset^.State = rsLoaded then
        Asset^.State := rsNeeded;
end;

procedure AssetUnlock(Asset : PAsset);
begin
    if not assigned(Asset) then exit;
    if Asset^.State = rsNeeded then
        Asset^.State := rsLoaded;
end;

function AssetPrint(P : PAsset) : boolean;
var
    I : word;
begin
    AssetPrint := False;
    if Not Assigned(P) then exit;
    if not Assigned(P^.Data) then exit;
    if P^.MemSize = 0 then Exit;
    case P^.Kind of
        ffText, ffNLS, ffNLSText : begin
            for I := 0 to P^.memSize - 1 do
                Write(Chars(P^.Data^)[I]);
        end
    else
        HexDump(P^.Data^, P^.MemSize);
    end;
    AssetPrint := True;
end;

procedure AssetDispose(var Asset : PAsset);
begin

    if not Assigned(Asset) then exit;
    if not Assigned(Asset^.Data) then exit;
    {$IFOPT D+}
    if Asset^.State = rsNeeded then
        FatalError(erDispose_Locked_Asset_Error, 'locked asset ' + PtrStr(Asset^.Name));
    {$ENDIF}
    {$IFDEF LOGS} Log('Unload Asset ' + + PtrStr(Asset^.Name)); {$ENDIF}
    case Asset^.Kind of
        ffBitmapFont : Video^.FreeFont(PFont(Asset^.Data));
        ffImage      : Video^.FreeImage(PImage(Asset^.Data));
        ffSprite     : Video^.FreeSprite(PSprite(Asset^.Data));
        ffAudio      : RunError(erOperation_Not_Supported);
    else
        { ffUnknown, ffDriver, ffText, ffNLS, ffNLSText, ffBinary }
        FreeMem(Asset^.Data, Asset^.MemSize);
        Asset^.Data := nil;
    end;

    Asset^.State := rsNone;
    if not Assigned(Asset^.Resource) then begin
    { Index was previously freed, so destroy asset now it is not needed }
        {$IFDEF LOGS}
            Log('Asset: ' + PtrStr(Asset^.Name) + ' destroyed');
        {$ENDIF}
        RemoveList(Assets, Asset);
        FreeStr(Asset^.Name);
        Dispose(Asset);
        Asset := nil;
    {$IFDEF LOGS}
    end else begin
        Log('Asset: ' + PtrStr(Asset^.Name) + ' unloaded');
    {$ENDIF}
    end;
end;

procedure AssetDisposeAll;
var
    P, N : PListItem;
begin
    {$IFDEF LOGS}
        Log('Assets: dispose all');
    {$ENDIF}
    P := Assets^.Last;
    while Assigned(P) do begin
        N := P^.Prev;
        if (PAsset(P^.Data)^.State = rsLoaded) then
            AssetDispose(PAsset(P^.Data));
        P := N;
    end;
end;

procedure CloseAssetFile(P : PAssetFile);
begin
    if Not Assigned(P) then Exit;
    if P^.Open then begin
        {$IFDEF LOGS}
            Log('Asset Close Resource: ' + RelativePath(PtrStr(P^.Name)));
        {$ENDIF}
        Close(P^.F);
        P^.Open := False;
        GetError;
        Dec(CurOpenFiles);
    end;
end;

procedure CloseAssetFiles;
var
    P : PListItem;
begin
    if Not Assigned(AssetFiles) then Exit;
    P := AssetFiles^.First;
    while Assigned(P) do begin
        CloseAssetFile(PAssetFile(P^.Data));
        P := P^.Next;
    end;
    ClearError;
end;

function OpenAssetFile(P : PAssetFile) : boolean;
var
    FM : word;
begin
    OpenAssetFile := True;
    if P^.Open then Exit;
    if CurOpenFiles >= MaxOpenFiles then CloseAssetFiles;
    if NoError then begin
        {$IFDEF LOGS}
            Log('Asset Open Resource: ' + RelativePath(PtrStr(P^.Name)));
        {$ENDIF}
        FM := FileMode;
        FileMode := 0;
        Reset(P^.F, 1);
        FileMode := FM;
        if NoError then begin
            Inc(CurOpenFiles);
            P^.Open := True;
            MoveToListStart(AssetFiles, P);
        end;
    end;
    OpenAssetFile := NoError;
end;

procedure AssetIndexFree(FileName : String);
var
    AF : PAssetFile;
    P, N : PListItem;
begin
    {$IFDEF LOGS}
        Log('Asset free index of ' + RelativePath(FileName));
    {$ENDIF}
    P := AssetFiles^.First;
    while Assigned(P) and (PtrStr(PAssetFile(P^.Data)^.Name) <> FileName) do
        P := P^.Next;
    if Not Assigned(P) then exit;
    AF := PAssetFile(P^.Data);
    CloseAssetFile(AF);
    P := Assets^.First;
    while Assigned(P) do begin
        N := P^.Next;
        if PAsset(P^.Data)^.Resource = AF then begin
            PAsset(P^.Data)^.Resource := nil;
            if (PAsset(P^.Data)^.State = rsLoaded) then
                AssetDispose(PAsset(P^.Data));
        end;
        P := N;
    end;
    RemoveList(AssetFiles, AF);
    FreeStr(AF^.Name);
    Dispose(AF);
end;

procedure AssetIndexSys(SysPath, AssetPath : String);
var
    Search : TSysFindRec;
    AK : Word;
    AF : PAssetFile;
    AI : PAsset;
    E  : Word;
    {$IFDEF LOGS}
    S1, S2 : String;
    {$ENDIF}
begin
    SysPath := asPath(SysPath);
    AssetPath := asPath(AssetPath);
    {$IFDEF LOGS}
        S1 := RelativePath(SysPath);
        if S1 = '' then S1 := '.';
        S2 := AssetPath;
        if S2 = '' then S2 := '.';
        Log('Asset index path: "' + S1 + '" under "' + S2 + '"');
    {$ENDIF}
    if SysFindFirst(SysPath+ '*.*', faAnyFile, Search) then
    repeat
        if Search.Attr and (faDirectory or faVolumeID or faSystem or faHidden) <> 0 then Continue;
        if FormatType(Search.Name) = ffUnknown then continue;
        AF := NewAssetFile(SysPath + Search.Name);
        if not Assigned(AF) then Break;
        {$IFDEF LOGS}
            Log('Asset: ' + AssetPath + Search.Name);
        {$ENDIF}
        AI := NewAsset(AF, AssetPath + Search.Name);
        if not Assigned(AI) then begin
            RemoveList(AssetFiles, AF);
            Dispose(AF);
            AF := nil;
            Break;
        end;
        AI^.StandAlone := True;
        AI^.DiskSize := Search.Size;
    until not SysFindNext(Search);
    E := GetError;
    SysFindClose(Search);
    SetError(E);
end;

procedure AssetIndex(FileName : String);
var
    ID : TResourceBlockID;
    T  : TResourceBlock;

    AF : PAssetFile;
    AI : PAsset;
    P, N : PListItem;

    S  : String;
    C  : Word;
    FP : LongInt;
    E : word;
begin
    ClearError;
    P := AssetFiles^.First;
    while Assigned(P) and (PtrStr(PAssetFile(P^.Data)^.Name) <> FileName) do
        P := P^.Next;
    if Assigned(P) then exit;
    {$IFDEF LOGS}
        Log('Asset index resource: ' + RelativePath(FileName) );
    {$ENDIF}
    AF := NewAssetFile(FileName);
    if not Assigned(AF) then Exit;
    OpenAssetFile(AF); { Postpone exit to cleanup }
    if AF^.Open then with AF^ do begin
        Seek(F, FileSize(F) - SizeOf(TResourceBlockID));
        if NoError then
            BlockRead(F, ID, Sizeof(TResourceBlockID));
        if NoError then begin
            S[0] := Chr(Length(fiXBINRSRC));
            Move(ID.Identifier, S[1], Sizeof(ID.Identifier));
            if S <> fiXBINRSRC then SetError(erInvalid_File_Format);
        end;
        if NoError then
            FP := ID.HeadBlock - (ID.LastSize - FileSize(F));
        while NoError do begin
            Seek(F, FP);
            if IsError then Break;
            BlockRead(F, T, Sizeof(T), C);
            if IsError then Break;
            if (C < sizeof(TResourceBlockID)) then
                SetError(erData_Verification_Error);
            if IsError then Break;
            if T.Data.BlockType = 1 then begin
                if C < sizeof(T.Data) + 3 then
                    SetError(erData_Verification_Error);
                if IsError then Break;
                {$IFDEF LOGS}
                    Log('Asset: ' + T.Name);
                {$ENDIF}
                AI := NewAsset(AF, T.Name);
                if Not Assigned(AI) then Break;
                AI^.Location := FP + SizeOf(T.Data) + Length(T.Name) + 2;
                if T.Data.Size > $FFFF then
                    AI^.DiskSize := 0
                else
                    AI^.DiskSize := T.Data.Size;
                if AI^.DiskSize = 0 then
                    AI^.State    := rsLoaded
                else
                    AI^.State    := rsNone;
                AI^.MemSize := 0;
            end;
            FP := FP + T.Data.BlockSize;
            if (T.Data.BlockType = 0) then break;
        end;
    end;
    E := GetError;
    CloseAssetFile(AF);
    if E <> 0 then
        AssetIndexFree(FileName);
    SetError(E);
end;

function AssetFileLoad (var P : PAsset; Item : PListItem; Locked : boolean ) : boolean;
var
    MSize : word;
    E : integer;
begin
    AssetFileLoad := False;
    if not OpenAssetFile(P^.Resource) then Exit;
    {$IFDEF LOGS}
        Log('Asset Load: ' + PtrStr(P^.Name) +
            ' (' + RelativePath(PtrStr(P^.Resource^.Name)) + ')');
    {$ENDIF}
    MSize := P^.DiskSize;
    case P^.Kind of
        ffText, ffNLS, ffNLSText : begin
            Inc(MSize, 2);
        end
    end;
    if P^.Kind = ffDriver then
        GetMemAlign(P^.Data, MSize)
    else
        GetMem(P^.Data, MSize);
    if not Assigned(P^.Data) then exit;
    if MSize <> P^.DiskSize then
        FillChar(P^.Data^, MSize, 0);
    Seek(P^.Resource^.F, P^.Location);
    if NoError then
        BlockRead(P^.Resource^.F, P^.Data^, P^.DiskSize);
    if NoError then begin
        P^.MemSize := MSize;
        case P^.Kind of
            ffText, ffNLS, ffNLSText : begin
            end;
        else
            FormatAsset(P);
        end;
    end;
    if NoError then begin
        MakeListItemStart(Assets, Item);
        if Locked then
            P^.State := rsNeeded
        else begin
            P^.State := rsLoaded;
        end;
        if P^.StandAlone then begin
            E := GetError;
            CloseAssetFile(P^.Resource);
            SetError(E);
        end;
    end else begin
        FreeMem(P^.Data, P^.DiskSize);
        P^.Data := nil;
        P^.MemSize := 0;
        P^.State := rsNone;
    end;
    AssetFileLoad := NoError;
end;

function AssetLoad(Name : String; Source : word; var P : PAsset) : boolean;
var
    LI, LF, LA : PListItem;
    FI, AI : PAsset;
begin
    Name := UCase(Name);
    ClearError;
    P := nil;
    AssetLoad := False;
    if Source = asNowhere then exit;
    LI := Assets^.First;
    FI := nil;
    AI := nil;
    LF := nil;
    LA := nil;
    while Assigned(LI) do begin
        if PtrStr(PAsset(LI^.Data)^.Name) = Name then begin
            if PAsset(LI^.Data)^.StandAlone then begin
                FI := PAsset(LI^.Data);
                LF := LI;
                if Assigned(AI) or (Source = asOnlyStandAlone) then break;
            end else begin
                AI := PAsset(LI^.Data);
                LA := LI;
                if Assigned(FI) or (Source = asOnlyAssetFile) then break;
            end;
        end;
        LI := LI^.Next;
    end;
    case Source of
        asPreferAssetFile : begin
            if Assigned(AI) then begin
                P := AI;
                LI := LA;
            end else begin
                P := FI;
                LI := LF;
            end;
        end;
        asPreferStandAlone : begin
            if Assigned(FI) then begin
                P := FI;
                LI := LF;
            end else begin
                P := AI;
                LI := LA;
            end;
        end;
        asOnlyAssetFile : begin
            P := AI;
            LI := LA;
        end;
        asOnlyStandAlone : begin
            P := FI;
            LI := LF;
        end;
    end;
    if not assigned(P) then exit;
    if P^.State <> rsNone then begin
        {$IFDEF LOGS}
            Log('Asset Return: ' + PtrStr(P^.Name));
        {$ENDIF}
        P^.State := rsNeeded;
        AssetLoad := True;
        Exit;
    end;
    if not Assigned(P^.Resource) then
        SetError(erUnassigned_Pointer_Error);
    if IsError then Exit;
    AssetFileLoad(P, LI, true);
    AssetLoad := Assigned(P^.Data) and NoError;
end;

function AssetLoadAll(WildCard : String; Lock : boolean) : boolean;
var
    P, N : PListItem;
    X : boolean;
begin
    {$IFDEF LOGS} Log('Load all assets: ' + Wildcard); {$ENDIF}
    WildCard := UCase(WildCard);
    P := Assets^.First;
    X := True;
    while Assigned(P) do begin
        N := P^.Next;
        if (PAsset(P^.Data)^.State = rsNone) and
        PatternMatch(WildCard, PtrStr(PAsset(P^.Data)^.Name)) then
            X := X and AssetFileLoad(PAsset(P^.Data), P, false);
        P := N;
    end;
    {$IFDEF LOGS} Log('Load all complete'); {$ENDIF}
    AssetLoadAll := X;
end;

function AssetLoadOrDie(Name : String; Source : word; var P : PAsset) : pointer;
begin
    if not AssetLoad(Name, Source, P) then begin
        if GetError <> 0 then
            FatalError(GetError, Name)
        else
            FatalError(erMissing_Asset_Error, Name);
    end;
    if not Assigned(P^.Data) then
        FatalError(erInvalid_Asset_Error, Name);
    AssetLoadOrDie := P^.Data;
end;

procedure DoneAssets;
begin
    CloseAssetFiles;
end;

procedure InitAssets;
begin
    AssetFiles := NewList;
    Assets := NewList;
    if not Assigned(Assets) then
        FatalError(erInsufficient_memory, 'asset initialization');
    AssetIndexSys(FilePath(ExeName), '');
    {$IFOPT D+}
        { Disable debug option to die on any and every error }
        DieOnError := False;
    {$ENDIF}
    AssetIndex(ExeName);
    if GetError = erInvalid_File_Format then
        ClearError; { Ignore startup EXE has no Assets Errors }
    {$IFOPT D+}
        DieOnError := True;
    {$ENDIF}
end;

{$IFOPT D+}
procedure PrintAssets;
var
    LR : PAssetFile;
    P : PListItem;
    I : integer;
begin
    LR := nil;
    P := Assets^.First;
    I := 0;
    While Assigned(P) do begin
        if LR <> PAsset(P^.Data)^.Resource then begin
            LR := PAsset(P^.Data)^.Resource;
            if Assigned(LR) then begin
                Write('(asset file: ', PtrStr(LR^.Name), ', ');
                if LR^.Open then
                    Write('open')
                else
                    Write('closed');
                WriteLn(')');
            end else
                WriteLn('(memory only assets)');
        end;
        With PAsset(P^.Data)^ do begin
            Write('#', IntStr(I), ' ');
            if StandAlone then Write('FILE:');
            Write(PtrStr(Name), ', ');
            Case Kind of
                ffUnknown       : Write('Unknown');
                ffDriver        : Write('Driver');
                ffBitmapFont    : Write('Font');
                ffImage         : Write('Image');
                ffSprite        : Write('Sprite');
                ffAudio         : Write('Audio');
                ffText          : Write('Plain Text');
                ffNLS           : Write('NLS');
                ffNLSText       : Write('NLS Text');
                ffBinary        : Write('Binary Data');
            else
                Write('INVALID');
            end;
            Write(', ');
            case State of
                rsNone   : Write('avail');
                rsLoaded : Write('loaded');
                rsNeeded : Write('LOCKED');
            end;
            Write(', ', DiskSize);
            if (State <> rsNone) and (MemSize <> DiskSize) then
                Write('+', MemSize - DiskSize );
            Write(' byte(s) at ', Location);
            WriteLn;
        end;
        P := P^.Next;
        Inc(I);
    end;
end;
{$ENDIF}

{$ENDIF}