{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$IFDEF SECTINT}

procedure Terminate(Code : integer);

function NoError : boolean;
function IsError : boolean;
function GetError : word;
procedure SetError(ErrorCode : word);
procedure SetIsError(ErrorCode : word);
procedure ClearError;
procedure FatalError(Code : integer; Message : String);

procedure GetMemAlign(var P : Pointer; Size : word); { GetMem Segment Aligned }
procedure GetIntVec ( IntNo : byte; var Vec : pointer );
procedure SetIntVec ( IntNo : byte; Vec : pointer );
function GetEnvCount: integer;
function GetEnvParam(AIndex : integer): String;
function GetEnv(Key : String): String;

function Identical(const A, B; Size : word) : boolean;
function CompareBytes(const A, B; Count : word) : integer;
function CompareWords(const A, B; Count : word) : integer;

function ByteHEX ( B : byte ) : String;
function WordHEX ( W : word ) : String;
function ByteBIN( B : Byte ) : String;
function WordBIN ( W : Word ) : String;
procedure HexDump(const Addr; Size : word);

function  RTrim ( Str : String ) : String;
function  LTrim ( Str : String ) : String;
function  Trim ( Str : String ) : String;
function  StrInt  ( S : String ) : LongInt;
function  IntStr  ( L : LongInt ) : String;
function  IntSign ( L : LongInt ) : String;
function  BoolStr (B : boolean) : String;
function  ZPad ( I : LongInt; W : word ) : String;
function  ZSign( I : LongInt; W : word ) : String;
function  RPad ( S : String; W : word ) : String;
function  LPad ( S : String; W : word ) : String;

function  AsciiZStr(const AsciiZ) : String;
function  UCase ( Str : String ) : String;
function  LCase ( Str : String ) : String;

function  StrPtr(const S : String) : PString;
function  PtrStr(const P : PString) : String;
procedure FreeStr(var P : PString);

function NextPos ( SubStr : String; S : String; StartPos : byte ) : byte;
function ReplaceStr ( Str : String; O, N : String ) : String;
function PopStr(Delim : String; var S : String) : String;
function PullStr(Delim : String; var S, SubStr : String) : boolean;
function FormatStr(S, Data : String) : String;

function RawNLS(ID : String) : String;
function FormatNLS(ID, Data : String) : String;

function NewList : PList;
function AddList(List : PList; Data : Pointer) : PListItem;
function PushList(List : PList; Data : Pointer) : PListItem;
function PopList(List : PList) : Pointer;
function PullList(List : PList) : Pointer;
function InList(List : PList; Data : Pointer) : PListItem;
function MoveToListStart(List : PList; Data : Pointer) : PListItem;
function MoveToListEnd(List : PList; Data : Pointer) : PListItem;
procedure RemoveList(List : PList; Data : Pointer);

procedure IncPtr(var P : Pointer; Count : word);
procedure DecPtr(var P : Pointer; Count : word);
function  PtrHex(P : Pointer) : Str12;

function FileBase(FileName : String) : string; { PATH\EXENAME.  }
function FileRoot(FileName : String) : string; { EXENAME.  }
function FilePath(FileName : String) : string; { PATH\ }
function RelativePath(FileName : String) : string; { relative to EXE, all or nothing }
function FileExt(FileName : String) : string;  { EXT without DOT}
function FileExists(FileName : String) : boolean;
function FileLoad(FileName : String; var P : Pointer; var Size : word) : boolean;
function FileSave(FileName : String; P : Pointer; Size : word) : boolean;
function FileAppend(FileName : String; P : Pointer; Size : word) : boolean;
function FileRead(FileName : String; Position : LongInt; Size : word) : pointer;
function FileSizeOf(FileName : String) : longint;

function TailDelim(S, Delim : String) : String;
{ if S <> '', make sure it ends in Delim }

procedure SysGetDate(var Year, Month, Day, DayOfWeek: Word);
procedure SysGetTime(var Hour, Minute, Second, Sec100: Word);

function SysFindFirst(PathName : String; Attr : Word; var SearchRec : TSysFindRec ) : boolean;
function SysFindNext(var SearchRec : TSysFindRec) : boolean;
function SysFindClose(var SearchRec : TSysFindRec) : boolean;

procedure Idle;
procedure Delay(ms : longint);
{ Hard, approximate delay in milliseconds with roughly 55ms precision.
  In general, this procedure should be avoided }

function PatternMatch( WildCard, Str : String ) : boolean;

procedure SetDefaultLanguage(P : Pointer);
function GetDefaultLanguage : pointer;
procedure SetUserLanguage(P : Pointer);
function GetUserLanguage : pointer;

function DateFmtStr (Format : String) : String;
function TimeFmtStr (Format : String) : String;
function TimeStamp(Format:byte) : String;

function asPath(Name : String) : String;

function PrintText(P : PAsset) : boolean;
procedure GetVersionInfo(var P : Pointer; var Count : Word; Width : word);
procedure PrintVersion;
procedure PrintHelp;

function Pause(ms : LongInt) : boolean;
function Wait(ms : LongInt; Manditory : boolean ) : boolean;

{$IFDEF LOGS}
const
    Logging : boolean = False;

procedure LogStart( Message : String);
procedure LogStop;
procedure LogReset;
procedure Log(Message:String);
{$ENDIF}

function SetVRTInterval(Interval : word) : word; { returns previous Interval }
{ VRT rate of 1 is a 1:1 of standard timer, ~18.2/second
  2 is 2:1 at ~36.4/second
  3 is 3:1 at ~54.6/second
  4 is 4:1 at ~72.8/second
  Anything faster would be excessive. So, other requests are completely ignored.
}
function GetVRTInterval : word;

procedure InitNLS;
{ This is done automatically during program initialization from the attached
asset pool. If no translation assets are attached it is performed again
during Ignite. I don't really ever see a need for the game/program to call it
manually. But, here it is anyway. }

{$ENDIF}

{$IFDEF SECTIMP}

const
    NLSTryAgain : boolean = True; { becomes false when InitNLS is able to load NLS }

{$IFOPT D+}
    DieOnError : boolean = true;
{$ENDIF}
    NLSData    : TNLSData = (
        TRUEStr:'True';
        FALSEStr:'False';
        Month: (
            Long:('', 'January', 'February', 'March', 'April', 'May', 'June',
            'July', 'August', 'September', 'October', 'November', 'December');
            Short:('', 'Jan','Feb','Mar','Apr','May','Jun', 'Jul','Aug','Sep',
            'Oct','Nov','Dec')
        );
        Day : (
            Long:('','Sunday','Monday','Tuesday','Wednesday','Thursday',
            'Friday','Saturday');
            Short:('','Sun','Mon','Tue','Wed','Thu','Fri','Sat')
        );
        AM:'am';
        PM:'pm';
        SYS: (
            DATE : '%0-%1-%2';
            TIME : '%0:%1:%2';
            DATETIME : '%0 %1'
        );
        LOG: (
            DATE : '%0-%1-%2';
            TIME : '%0%1%2';
            DATETIME : '%1'
        );
        Error: (
            DriverInit  : 'unable to initialize %0 driver';
            DriverFind  : 'unable to locate %0 driver';
            DriverMode  : 'opening graphics driver %0 mode %1';
            FontInit    : 'unable to load %0 font';
            FontFind    : 'unable to find %0 font'
        )
    );

procedure Terminate(Code : integer);
begin
    if Assigned(Video) and (Video <> VideoNULL) then Video^.Close;
    {$IFDEF LOGS}
        Log('Terminate: ' + IntStr(Code));
    {$ENDIF}
   Halt(Code);
end;

procedure Idle; assembler;
asm
    mov  ax, [CPUIdle]
    test ax, ax
    jz   @NoIdle
    hlt
@NoIdle:
end;

procedure Delay(ms : longint); { far; } external;

function Ext_GetError : word; far; external;
procedure Ext_SetError(ErrorCode : word); far; external;

function NoError : boolean;
begin
    NoError := GetError = 0;
end;

function IsError : boolean;
begin
    IsError := GetError <> 0;
end;

function GetError : word;
var
    E : Integer;
begin
    E := IOResult;
    if E <> 0 then Ext_SetError(E);
    {$IFOPT D+}
        E := Ext_GetError;
        if DieOnError and (E <> erNo_Error) then
            FatalError(E, 'Debug Termination');
        GetError := E;
    {$ELSE}
        GetError := Ext_GetError;
    {$ENDIF}
end;

procedure SetError(ErrorCode : word);
begin
    GetError;
    Ext_SetError(ErrorCode);
    {$IFOPT D+}
        if DieOnError and (ErrorCode <> erNo_Error) then
            FatalError(ErrorCode, 'Debug Termination');
    {$ENDIF}
end;

procedure SetIsError(ErrorCode : word);
begin
    if ErrorCode <> 0 then SetError(ErrorCode);
end;

procedure ClearError;
begin
    SetError(erNo_Error);
end;

procedure FatalError(Code : integer; Message : String);
begin
    ErrorMessage := RawNLS('ERROR.' + IntStr(Code));
    if (ErrorMessage <> '') and (Message <> '') then
        ErrorMessage := ErrorMessage + ', ';
    ErrorMessage := ErrorMessage + Message;
    {$IFDEF LOGS}
        Log('Fatal Error: ' + IntStr(Code) + ' , ' + ErrorMessage);
    {$ENDIF}
    Terminate(Code);
end;

procedure GetMemAlign(var P : Pointer; Size : word);
{ maybe a better way to ensure pointer is segment aligned. but meh, its
  only required on driver loading and this will work fine. }
type
    PChain = ^TChain;
    TChain = record
        Next : PChain;
        Fill : word;
    end;
var
    Chain, T : PChain;
begin
    Chain := nil;
    GetMem(P, Size);
    while Assigned(P) and (Ofs(P^) <> 0) do begin
        FreeMem(P, Size);
        P := nil;
        T := New(PChain);
        if Not Assigned(T) then break;
        T^.Next := Chain;
        Chain := T;
        GetMem(P, Size);
    end;
    while Assigned(Chain) do begin
        T := Chain^.Next;
        Dispose(Chain);
        Chain := T;
    end;
end;

procedure GetIntVec ( IntNo : byte; var Vec : pointer ); assembler;
asm
      CLI
      MOV  AH, $35
      MOV  AL, IntNo
      INT  $21
      MOV  DX, ES
      LES  DI, Vec
      MOV  AX, BX
      STOSW
      MOV  AX, DX
      STOSW
      STI
end;

procedure SetIntVec ( IntNo : byte; Vec : pointer ); assembler;
asm
      CLI
      PUSH DS
      MOV  AH, $25
      MOV  AL, IntNo
      LDS  DX, Vec
      INT  $21
      POP  DS
      STI
end;

function GetEnvCount: integer; assembler;
asm
		PUSH DS
		MOV  DS, PrefixSeg
        MOV  DI, 0
        MOV  DS, [$2C]
        XOR  CX, CX
   @ReadLoop:
    	MOV	 AL, [DI]
    	CMP	 AL, 0
		JE	 @ReadDone
		INC  CX
	@ReadString:
    	MOV	 AL, [DI]
		INC  DI
    	CMP	 AL, 0
		JNE	 @ReadString
		JMP @ReadLoop
	@ReadDone:
	@Done:
		MOV	 AX, CX
		POP	 DS
end;

function GetEnvParam(AIndex : integer): String; assembler;
asm
		PUSH DS
		PUSH ES
		MOV  DX, AIndex
        LES  BX, @Result
        PUSH BX
		MOV  DS, PrefixSeg
        MOV  DI, 0
        MOV  DS, [$2C]
        XOR  CX, CX
        XOR  AH, AH
        CMP  DX, 0
        JE   @ReadDone
   @ReadLoop:
    	MOV	 AL, [DI]
    	CMP	 AL, 0
		JE	 @ReadDone
		INC  CX
		CMP  CX, DX
		JE	 @ReadString
	@ReadSkip:
    	MOV	 AL, [DI]
		INC  DI
    	CMP	 AL, 0
		JNE	 @ReadSkip
		JMP  @ReadLoop
	@ReadString:
    	MOV	 AL, [DI]
		INC  DI
    	CMP	 AL, 0
		JE	 @ReadDone
		INC	AH
		INC BX
		MOV  ES:[BX], AL
		CMP	 AH, $FF
		JE	 @ReadDone
		JMP  @ReadString
	@ReadDone:
		POP  BX
		MOV  ES:[BX], AH
		POP  ES
		POP	 DS
end;

function GetEnv(Key : String): String;
var
	I : integer;
	S : String;
begin
	GetEnv:='';
	Key := UCase(Key) + '=';
	for I := 1 to GetEnvCount do begin
		S := GetEnvParam(I);
		if Pos(Key, UCase(S)) = 1 then begin
			GetEnv := Copy(S, Length(Key) + 1, Length(S));
			Break;
		end;
	end;
end;

function Identical(const A, B; Size : word) : boolean;
var
    I : word;
begin
    Identical := False;
    if Size = 0 then exit;
    for I := 0 to Size - 1 do
        if Bytes(A)[I] <> Bytes(B)[I] then Exit;
    Identical := True;
end;

function CompareBytes(const A, B; Count : word) : integer;
var
    I : word;
begin
    CompareBytes := 0;
    if Count = 0 then exit;
    for I := 0 to Count - 1 do
        if Bytes(A)[I] < Bytes(B)[I] then begin
            CompareBytes := -1;
            Exit;
        end else if Bytes(A)[I] > Bytes(B)[I] then begin
            CompareBytes := 1;
            Exit;
        end;
end;

function CompareWords(const A, B; Count : word) : integer;
var
    I : word;
begin
    CompareWords := 0;
    if Count = 0 then exit;
    for I := 0 to Count - 1 do
        if Words(A)[I] < Words(B)[I] then begin
            CompareWords := -1;
            Exit;
        end else if Words(A)[I] > Words(B)[I] then begin
            CompareWords := 1;
            Exit;
        end;
end;

function ByteHEX ( B : byte ) : String; {far;} external;

function WordHEX ( W : word ) : String;
begin
    WordHEX := ByteHEX(Hi(W)) + ByteHex(Lo(W));
end;

function ByteBIN( B : Byte ) : String;
var
    I : byte;
    S : String;
begin
    S := '';
    for I := 7 downto 0 do
        if B and (1 shl I) <> 0 then
            S := S + '1'
        else
            S := S + '0';
    ByteBIN := S;
end;

function WordBIN ( W : Word ) : String;
begin
    WordBIN := ByteBIN(Hi(W)) + ByteBIN(lo(W));
end;

procedure HexDump(const Addr; Size : word);
var
    R, C : word;
    S : String;
begin
    R := 0;
    while R < Size do begin
        C := 0;
        S := '  ' + ByteHEX(Hi(R)) + ByteHEX(lo(R)) + '  |  ';
        while C < 15 do begin
            if R + C >= Size then break;
            S := S + ByteHEX(Bytes(Addr)[R + C]) + ' ';
            Inc(C);
        end;
        While Length(S) < 60 do
            S := S + ' ';
        C := 0;
        while C < 15 do begin
            if R + C >= Size then break;
            if (Bytes(Addr)[R + C] < 32) then
                S := S + '.'
            else
                S := S + Chars(Addr)[R + C];
            Inc(C);
        end;
        WriteLn(S);
        Inc(R, 16);
    end;
end;

function RTrim ( Str : String ) : String;
begin
    asm
        CLD
        XOR		BL, BL
        LEA     SI, Str
        LES     DI, @Result
        PUSH 	DI
        SEGSS   LODSB
        STOSB
        XOR     AH, AH
        XCHG    AX, CX
        JCXZ    @4
    @1:
        SEGSS   LODSB
        INC     AH
        CMP 	AL, 20h
        JE		@2
        CMP 	AL, 09h
        JE		@2
        MOV		BL, AH
    @2:
        STOSB
        @3:
        LOOP    @1
    @4:
        POP 	DI
        MOV 	ES:[DI], BL
    end;
end;

function LTrim ( Str : String ) : String;
begin
    asm
        CLD
        MOV 	BL, False
        LEA     SI, Str
        LES     DI, @Result
        PUSH 	DI
        SEGSS   LODSB
        STOSB
        XOR     AH, AH
        XCHG    AX, CX
        JCXZ    @5
    @1:
        SEGSS   LODSB
        CMP 	BL, True
        JE 		@3
        CMP 	AL, 20h
        JE		@4
        CMP 	AL, 09h
        JE		@4
    @2:
        MOV		BL, True
    @3:
        INC     AH
        STOSB
    @4:
        LOOP    @1
    @5:
        POP 	DI
        MOV 	ES:[DI], AH
    end;
end;

function Trim ( Str : String ) : String;
begin
    asm
        CLD
        MOV 	BL, False
        LEA     SI, Str
        LES     DI, @Result
        PUSH 	DI
        SEGSS   LODSB
        STOSB
        XOR     AH, AH
        XCHG    AX, CX
        JCXZ    @5
    @1:
        SEGSS   LODSB
        CMP 	BL, True
        JE 		@3
        CMP 	AL, 20h
        JE		@4
        CMP 	AL, 09h
        JE		@4
    @2:
        MOV		BL, True
    @3:
        INC     AH
        STOSB
    @4:
        LOOP    @1
    @5:
        POP 	DI
        XOR 	BH, BH
        MOV 	BL, AH
    @6:
        CMP		BX, 0h
        JE		@8
        MOV		AL, ES:[DI + BX]
        CMP		AL, 20h
        JE		@7
        CMP		AL, 09h
        JNE		@8
    @7:
        DEC		BL
        DEC		AH
        JMP		@6
    @8:
        MOV 	ES:[DI], AH
    end;
end;

function StrInt  ( S : String ) : LongInt;
var
    I : LongInt;
    E : integer;
begin
    Val(S, I, E);
    if E <> 0 then
        StrInt := 0
    else
        StrInt := I;
end;

function IntStr  ( L : LongInt ) : String;
var
    S : String;
begin
    Str ( L, S );
    IntStr := S;
end;
function IntSign( L : LongInt ) : String;
var
    S : String;
begin
    Str ( L, S );
    if L = ABS(L) then
        IntSign := '+' + S
    else
        IntSign := S;
end;

function BoolStr (B : boolean) : String;
begin
    if B then
        BoolStr := NLSData.TRUEStr
    else
        BoolStr := NLSData.FALSEStr;
end;

function ZPad ( I : LongInt; W : word ) : String;
var
    S, N : String;
begin
    N := IntStr(Abs(I));
    if I <> Abs(I) then Dec(W);
    if Length(N) < W then begin
        if W >= Sizeof(S) then W := Sizeof(S) - 1;
        W := W - Length(N);
        FillChar(S[1], W, '0');
        S[0] := Char(W);
        N := S + N;
    end;
    if I <> Abs(I) then
        N := '-' + N;
    ZPad := N;
end;
function ZSign ( I : LongInt; W : word ) : String;
var
    S, N : String;
begin
    N := IntStr(Abs(I));
    Dec(W);
    if Length(N) < W then begin
        if W >= Sizeof(S) then W := Sizeof(S) - 1;
        W := W - Length(N);
        FillChar(S[1], W, '0');
        S[0] := Char(W);
        N := S + N;
    end;
    if I = Abs(I) then
        ZSign := '+' + N
    else
        ZSign := '-' + N;
end;

function RPad ( S : String; W : word ) : String;
var
    N : String;
begin
    if Length(S) < W then begin
        if W >= Sizeof(S) then W := Sizeof(S) - 1;
        W := W - Length(S);
        FillChar(N[1], W, #32);
        N[0] := Char(W);
        RPad := S + N;
    end else
        RPad := S;
end;

function LPad (  S : String; W : word ) : String;
var
    N : String;
begin
    if Length(S) < W then begin
        if W >= Sizeof(S) then W := Sizeof(S) - 1;
        W := W - Length(S);
        FillChar(N[1], W, #32);
        N[0] := Char(W);
        LPad := N + S;
    end else
        LPad := S;
end;

function AsciiZStr(const AsciiZ) : String;
type
	AsciiZChars = array[0..$FF] of char;
var
	I : integer;
	S : String;
begin
	AsciiZStr := '';
	I := 0;
	while (I <= 254) and (AsciiZChars(AsciiZ)[I] <> #0) do Inc(I);
	if I > 0 then begin
		Move(AsciiZ, S[1], I);
		S[0] := Chr(I);
		AsciiZStr := S;
	end;
end;

function UCase ( Str : String ) : String;
begin
    asm
        CLD
        LEA     SI, Str
        LES     DI, @Result
        SEGSS   LODSB
        STOSB
        XOR     AH, AH
        XCHG    AX, CX
        JCXZ    @3
    @1:
        SEGSS   LODSB
        CMP     AL, 'a'
        JB      @2
        CMP     AL, 'z'
        JA      @2
        SUB     AL, 20h
    @2:
        STOSB
        LOOP    @1
    @3:
    end;
end;

function LCase ( Str : String ) : String;
begin
    asm
        CLD
        LEA     SI, Str
        LES     DI, @Result
        SEGSS   LODSB
        STOSB
        XOR     AH, AH
        XCHG    AX, CX
        JCXZ    @3
    @1:
        SEGSS   LODSB
        CMP     AL, 'A'
        JB      @2
        CMP     AL, 'Z'
        JA      @2
        ADD     AL, 20h
    @2:
        STOSB
        LOOP    @1
    @3:
    end;
end;

function StrPtr(const S : String) : PString;
var
	P : PString;
begin
    {$IFDEF MEMCHECK}
	MemCheck(Length(S) + 1);
	{$ENDIF}
	GetMem(P, Length(S) + 1);
	StrPtr := P;
    if Not Assigned(P) then exit;
	Move(S, P^, Length(S) + 1);
end;

function PtrStr(const P : PString) : String;
var
	S : String;
begin
	if Assigned(P) then
		Move(P^, S, Length(P^) + 1)
	else
		S := '';
	PtrStr := S;
end;

procedure FreeStr(var P : PString);
begin
	if Assigned(P) then
		FreeMem(P, Length(P^) + 1);
	P := nil;
end;

function NextPos ( SubStr : String; S : String; StartPos : byte ) : byte;
var
    StrPtr : ^String;
    TPos   : Byte;
begin
    if (StartPos = 0) or (StartPos > Length(S)) then
        NextPos := 0
    else begin
        Dec(StartPos);
        S[StartPos] := Char(Length(S) - StartPos);
        StrPtr := @S[StartPos];
        TPos := Pos(SubStr, StrPtr^);
        if TPos > 0 then Inc(TPos, StartPos);
        NextPos := TPos;
    end;
end;

function LastPos( SubStr, S : String ) : byte;
var
    T, N : Byte;
begin
    N := 0;
    repeat
        T := N;
        N := NextPos(SubStr, S, N + 1);
    until N = 0;
    LastPos := T;
end;

function ReplaceStr ( Str : String; O, N : String ) : String;
var
    P : integer;
begin
    P := Pos(O, Str);
    while P > 0 do begin
        Delete(Str, P, Length(O));
        Insert(N, Str, P);
        P :=NextPos(O, Str, P + Length(N));
    end;
    ReplaceStr := Str;
end;

function PopStr(Delim : String; var S : String) : String;
var
	P : integer;
begin
	P := Pos(Delim, S);
	if P = 0 then begin
		PopStr := S;
		S := '';
	end else begin
		PopStr := Copy(S, 1, P - 1);
		Delete(S, 1, P + Length(Delim) - 1);
	end;
end;

function PullStr(Delim : String; var S, SubStr : String) : boolean;
begin
    PullStr := S <> '';
    SubStr := PopStr(Delim, S);
end;

function FormatStr(S, Data : String) : String;
var
    T : String;
    I : integer;
begin
    I := 0;
    while Data <> '' do begin
        T := PopStr(NLSDelim,Data);
        S := ReplaceStr(S, NlsVarTag + IntStr(I), T);
        Inc(I);
    end;
    FormatStr := S;
end;


function RawNLS(ID : String) : String; {far;} external;
function FormatNLS(ID, Data : String) : String;
var
    S, T : String;
    I : integer;
begin
    S := RawNLS(ID);
    I := 0;
    while Data <> '' do begin
        T := PopStr(NLSDelim,Data);
        S := ReplaceStr(S, NlsVarTag + IntStr(I), T);
        Inc(I);
    end;
    FormatNLS := S;
end;


function NewList : PList;
var
    P : PList;
begin
    P := New(PList);
    if Assigned(P) then begin
        P^.First := nil;
        P^.Last := nil;
        P^.Count := 0;
    end else
        RunError(erInsufficient_Memory);
    NewList := P;
end;

function AddList(List : PList; Data : Pointer) : PListItem;
var
    Item : PListItem;
begin
    AddList := nil;
    if not Assigned(List) then Exit;
    Item := New(PListItem);
    if not Assigned(Item) then Exit;
    Inc(List^.Count);
    Item^.Data := Data;
    Item^.Next := nil;
    Item^.Prev := List^.Last;
    if Assigned(List^.Last) then
        List^.Last^.Next := Item
    else
        List^.First := Item;
    List^.Last := Item;
    AddList := Item;
end;

function PushList(List : PList; Data : Pointer) : PListItem;
var
    Item : PListItem;
begin
    PushList := nil;
    if not Assigned(List) then Exit;
    Item := New(PListItem);
    if not Assigned(Item) then Exit;
    Inc(List^.Count);
    Item^.Data := Data;
    Item^.Prev := nil;
    Item^.Next := List^.First;
    if Assigned(List^.First) then
        List^.First^.Prev := Item
    else
        List^.Last := Item;
    List^.First := Item;
    PushList := Item;
end;

function PopList(List : PList) : Pointer;
var
    P : PListItem;
begin
    PopList := nil;
    if not (Assigned(List) and Assigned(List^.First)) then Exit;
    PopList := List^.First^.Data;
    P := List^.First;
    List^.First := P^.Next;
    if Assigned(List^.First) then
        List^.First^.Prev := nil
    else
        List^.Last := List^.First;
    Dispose(P);
end;

function PullList(List : PList) : Pointer;
var
    P : PListItem;
begin
    PullList := nil;
    if not (Assigned(List) and Assigned(List^.Last)) then Exit;
    PullList := List^.Last^.Data;
    P := List^.Last;
    List^.Last := P^.Prev;
    if Assigned(List^.Last) then
        List^.Last^.Next := nil
    else
        List^.First := List^.Last;
    Dispose(P);
end;

function InList(List : PList; Data : Pointer) : PListItem;
var
    P : PListItem;
begin
    InList := nil;
    if Not Assigned(List) then exit;
    P := List^.First;
    while Assigned(P) and (P^.Data <> Data) do
        P := P^.Next;
    InList := P;
end;

procedure DetachListItem(List : PList; Item : PListItem);
begin
    { only called by list functions }
    {$IFOPT D+}
    if Assigned(List) and Assigned(Item) then begin
    {$ENDIF}
        if Assigned(Item^.Prev) then
            Item^.Prev^.Next := Item^.Next
        else
            List^.First := Item^.Next;
        if Assigned(Item^.Next) then
            Item^.Next^.Prev := Item^.Prev
        else
            List^.Last := Item^.Prev;
    {$IFOPT D+}
    end else
        FatalError(erInvalid_Pointer_Operation, 'DetachListItem');
    {$ENDIF}
end;

procedure MakeListItemStart(List : PList; Item : PListItem);
begin
    if Assigned(List) and Assigned(Item) then begin
        DetachListItem(List, Item);
        Item^.Prev := nil;
        Item^.Next := List^.First;
        if Assigned(List^.First) then
            List^.First^.Prev := Item
        else
            List^.Last := Item;
        List^.First := Item;
    end;
end;

procedure MakeListItemEnd(List : PList; Item : PListItem);
begin
    if Assigned(List) and Assigned(Item) then begin
        DetachListItem(List, Item);
        Item^.Next := nil;
        Item^.Prev := List^.Last;
        if Assigned(List^.Last) then
            List^.Last^.Next := Item
        else
            List^.First := Item;
        List^.Last := Item;
    end;
end;

procedure RemoveListItem(List : PList; Item : PListItem);
begin
    if Assigned(Item) then begin
        DetachListItem(List, Item);
        Dispose(Item);
    end;
end;

procedure RemoveList(List : PList; Data : Pointer);
begin
    RemoveListItem(List, InList(List, Data));
end;

function MoveToListStart(List : PList; Data : Pointer) : PListItem;
begin
    MakeListItemStart(List, InList(List, Data));
end;

function MoveToListEnd(List : PList; Data : Pointer) : PListItem;
begin
    MakeListItemEnd(List, InList(List, Data));
end;

procedure IncPtr(var P : Pointer; Count : word);
begin
    P := Ptr(Seg(P^), Ofs(P^) + Count);
end;

procedure DecPtr(var P : Pointer; Count : word);
begin
    P := Ptr(Seg(P^), Ofs(P^) - Count);
end;

function  PtrHex(P : Pointer) : Str12;
begin
    PtrHex := '[' + WordHex(Seg(P^)) + ':' +  WordHex(Ofs(P^)) + ']'
end;

function FileBase(FileName : String) : string;
var
    B : String[16];
    N : byte;
begin
    N := LastPos(PathDelim, FileName);
    B := Copy(FileName, N + 1, 255);
    FileName[0] := Char(N);
    N := LastPos('.', B);
    if N = 0 then begin
        B := B + '.';
    end else begin
        B[0] := Char(N);
    end;
    FileBase :=  FileName + B;
end;

function FileRoot(FileName : String) : string;
var
    N : byte;
begin
    N := LastPos(PathDelim, FileName);
    FileName := Copy(FileName, N + 1, 255);
    N := LastPos('.', FileName);
    if N = 0 then
        Filename := FileName + '.'
    else
        FileName[0] := Char(N);
    FileRoot :=  FileName;
end;

function FilePath(FileName : String) : string;
var
    N : byte;
begin
    N := LastPos(PathDelim, FileName);
    FileName[0] := Char(N);
    FilePath := FileName;
end;

function RelativePath(FileName : String) : string;
var
    E : String;
begin
    E := Filepath(ExeName);
    if Copy(Filename, 1, Length(E)) = E then
        Delete(FileName, 1, Length(E));
    RelativePath := FileName;
end;

function FileExt(FileName : String) : string;
begin
    FileExt := UCase(Copy(FileName, Length(FileBase(FileName))+1, 255));
end;

function FileExists(FileName : String) : boolean;
var
    FM : Word;
    FH : File;
    FE : Integer;
begin
    FileExists := False;
    if FileName = '' then exit;
    FM := FileMode;
    FileMode := 0;
    Assign(FH, FileName);
    Reset(FH, 1);
    FE := IOResult;
    Close(FH);
    FE := IOResult or FE;
    FileMode := FM;
    FileExists := FE = 0;
end;

function FileLoad(FileName : String; var P : Pointer; var Size : word) : boolean;
var
    FM : Word;
    FH : File;
    FS : LongInt;
begin
    ClearError;
    P := nil;
    FileLoad := false;
    Size := 0;
    FS := 0;
    if FileName = '' then exit;
    FM := FileMode;
    FileMode := 0;
    Assign(FH, FileName);
    Reset(FH, 1);
    if NoError then begin
        FS := FileSize(FH);
        if FS > $FFF0 then SetError(erStructure_Too_Large);
    end;
    if NoError then begin
        Size := FS;
        if FileExt(FileName) = 'DRV' then
            GetMemAlign(P, Size)
        else
            GetMem(P, Size);
        if Not Assigned(P) then SetError(erInsufficient_Memory);
    end;
    if NoError then BlockRead(FH, P^, Size);
    Close(FH);
    if IsError and Assigned(P) then begin
        FreeMem(P, Size);
        P := nil;
        Size := 0;
    end;
    FileMode := FM;
    FileLoad := Assigned(P);
end;

function FileSave(FileName : String; P : Pointer; Size : word) : boolean;
var
    FH : File;
begin
    ClearError;
    FileSave := False;
    if FileName = '' then exit;
    Assign(FH, FileName);
    Rewrite(FH, 1);
    if NoError then BlockWrite(FH, P^, Size);
    Close(FH);
    FileSave := NoError;
end;

function FileAppend(FileName : String; P : Pointer; Size : word) : boolean;
var
    FH : File;
begin
    ClearError;
    FileAppend := False;
    if FileName = '' then exit;
    Assign(FH, FileName);
    Reset(FH, 1);
    if NoError then Seek(FH, FileSize(FH));
    if NoError then BlockWrite(FH, P^, Size);
    Close(FH);
    FileAppend := NoError;
end;

function FileRead(FileName : String; Position : LongInt; Size : word) : pointer;
var
    FM : Word;
    FH : File;
    P : Pointer;
begin
    ClearError;
    P := nil;
    FileRead := nil;
    if FileName = '' then exit;
    FM := FileMode;
    FileMode := 0;
    Assign(FH, FileName);
    Reset(FH, 1);
    if (Position <> 0) and NoError then
        Seek(FH, Position);
    if NoError then begin
        {$IFDEF MEMCHECK}
        MemCheck(Size);
        {$ENDIF}
        GetMem(P, Size);
        if Not Assigned(P) then SetError(erInsufficient_Memory);
    end;
    if NoError then BlockRead(FH, P^, Size);
    Close(FH);
    if IsError and Assigned(P) then begin
        FreeMem(P, Size);
        P := nil;
    end;
    FileMode := FM;
    FileRead := P;
end;

function FileSizeOf(FileName : String) : longint;
var
    FM : Word;
    FH : File;
    FS : LongInt;
begin
    ClearError;
    FS := -1;
    FileSizeOf := FS;
    if FileName = '' then exit;
    FM := FileMode;
    FileMode := 0;
    Assign(FH, FileName);
    Reset(FH, 1);
    if NoError then begin
        FS := FileSize(FH);
        if FS > $FFF0 then SetError(erInsufficient_Memory);
    end;
    Close(FH);
    FileMode := FM;
    if IsError then
        FileSizeOf := -1
    else
        FileSizeOf := FS;
end;


function TailDelim(S, Delim : String) : String;
begin
    if (S <> '') and (Copy(S, Length(S) - Length(Delim) + 1, Length(Delim)) <> Delim) then
        S := S + Delim;
    TailDelim := S;
end;

procedure SysGetDate(var Year, Month, Day, DayOfWeek: Word); assembler;
{ Day of Week is only returned in DOS 3.00 or better }
asm
      MOV  AH, 2Ah
      INT  21h
      LES  DI, Year
      MOV  ES:[DI], CX
      XOR  AH, AH
      LES  DI, DayOfWeek
      CLD
      STOSW
      LES  DI, Month
      MOV  AL, DH
      STOSW
      LES  DI, Day
      MOV  AL, DL
      STOSW
end;

procedure SysGetTime(var Hour, Minute, Second, Sec100: Word); assembler;
asm
    MOV  AH, 2Ch
    INT  21h
    XOR  AH, AH
    MOV  AL, CH
    { DOSBox Bug Fix / could be 28hr of the day :-( }
@1:
    CMP   AL, 24
    JBE   @2
    SUB   AL, 24
    JMP   @1
@2:
    { End Fix }
    CLD
    LES  DI, Hour
    STOSW
    MOV  AL, CL
    LES  DI, Minute
    STOSW
    MOV  AL, DH
    LES  DI, Second
    STOSW
    MOV  AL, DL
    LES  DI, Sec100
    STOSW
end;

function SysFindFirst(PathName : String; Attr : Word; var SearchRec : TSysFindRec ) : boolean;
var
    DosError : integer;
begin
    PathName := PathName + #0;
    asm
        MOV		AH, 1Ah
        PUSH 	DS
        LDS		DX, SearchRec
        INT     21h
        POP		DS
        MOV     CX, attr
        PUSH 	DS
        LEA     DX, PathName
        INC		DX
        PUSH 	ES
        POP		DS
        MOV     AH, 4Eh
        INT     21h
        JC      @NotFound
        POP		DS

        { Make into Pascal String }
        MOV		CX, 12
        MOV		AH, CL
        LES		DI, SearchRec
        ADD		DI, 30 + 12
    @Looped:
        CMP		CL, 0
        JE		@SetLen
        MOV		AL, ES:[DI - 1]
        MOV		ES:[DI], AL
        DEC		CL
        DEC		DI
        CMP		AL, 0
        JNE		@Looped
        MOV		AH, CL
        JMP		@Looped
    @SetLen:
        LES		DI, SearchRec
        MOV		ES:[DI + 30], AH
        MOV 	DosError, 0
        JMP 	@Done
    @NotFound:
        XOR		AL, AL
        LES		DI, SearchRec
        MOV		ES:[DI + 30], AL
        POP		DS
        MOV		DosError, 12h
    @Done:
    end;
    if DosError = 18 then
        ClearError
    else
        SetError(DosError);
    SysFindFirst := DosError = 0;
end;

function SysFindNext(var SearchRec : TSysFindRec) : boolean;
var
    DosError : integer;
begin
    asm
        MOV     AH, 4Fh
        PUSH 	DS
        LDS		DX, SearchRec
        INT     21h
        JC      @NotFound
        POP		DS

        { Make into Pascal String }
        MOV		CX, 12
        MOV		AH, CL
        LES		DI, SearchRec
        ADD		DI, 30 + 12
    @Looped:
        CMP		CL, 0
        JE		@SetLen
        MOV		AL, ES:[DI - 1]
        MOV		ES:[DI], AL
        DEC		CL
        DEC		DI
        CMP		AL, 0
        JNE		@Looped
        MOV		AH, CL
        JMP		@Looped
    @SetLen:
        LES		DI, SearchRec
        MOV		ES:[DI + 30], AH
        MOV 	DosError, 0
        JMP 	@Done
    @NotFound:
        XOR		AL, AL
        LES		DI, SearchRec
        MOV		ES:[DI + 30], AL
        POP		DS
        MOV		DosError, 12h
    @Done:
    end;
    if DosError = 18 then
        ClearError
    else
        SetError(DosError);
    SysFindNext := DosError = 0;
end;

function SysFindClose(var SearchRec : TSysFindRec) : boolean;
begin
    SysFindClose := True;
    ClearError;
end;


function PatternMatchSub(WildCard, Str :String) : boolean;
var
    I : Integer;
begin
    PatternMatchSub:= False;
    if Length(WildCard) <> Length(Str) then exit;
    for I := 1 to Length(WildCard) do
        if (WildCard[I] <> Str[I]) and (WildCard[I] <> '?') then exit;
    PatternMatchSub := True;
end;

function PatternMatchPos(SubStr, Str : String) : integer;
var
    I : integer;
begin
    PatternMatchPos := 0;
    for I := 1 to Length(Str) - Length(SubStr) + 1 do
        if PatternMatchSub(SubStr, Copy(Str, I, Length(SubStr))) then begin
            PatternMatchPos := I;
            Break;
        end;
end;

function PatternMatch(WildCard, Str : String) : boolean;
{ meh... far from perfect, but good enough for now. :-) }
var
    PW, PS : integer;
    X : integer;
begin
    X := 0;
    PatternMatch := True;
    if WildCard = Str then Exit;
    if WildCard = '' then begin
        PatternMatch := False;
        Exit;
    end;
    repeat
        Inc(X);
        if WildCard[1] = '*' then begin
            while (WildCard<> '') and (WildCard[1] = '*') do Delete(WildCard, 1,1);
            if WildCard = '' then Exit;
            PW := Pos('*', WildCard);
            if PW < 1 then PW := Length(WildCard)+ 1;
            PS := PatternMatchPos(Copy(WildCard, 1, PW -1), Str);
            if PS < 1 then Break;
            Delete(Str, 1, PS - 1);
        end;
        if PatternMatchSub(WildCard, Str) then Exit;
        PW := Pos('*', WildCard) - 1;
        if PW < 1 then PW := Length(WildCard) + 1;
        if not PatternMatchSub(Copy(WildCard,1, PW), Copy(Str, 1, PW)) then Break;
        Delete(WildCard, 1, PW);
        Delete(Str, 1, PW);
    until (WildCard = '') or (Str = '') or (X = 1000);
    PatternMatch := ((WildCard = '*') or (WildCard = '')) and (Str = '');
end;

function DateFmtStr (Format : String) : String;
var
    Year, Month, Day, DayOfWeek: Word;
begin
    SysGetDate(Year, Month,Day,DayOfWeek);
    Inc(DayOfWeek);
    DateFmtStr := FormatStr(Format,
        { 0 }    ZPad(Year, 4) + NLSDelim +
        { 1 }    ZPad(Month, 2) + NLSDelim +
        { 2 }    ZPad(Day, 2) + NLSDelim +
        { 3 }    ZPad(DayOfWeek, 1) + NLSDelim +
        { 4 }    NLSData.Month.Short[Month] + NLSDelim +
        { 5 }    NLSData.Day.Short[DayOfWeek] + NLSDelim +
        { 6 }    NLSData.Month.Long[Month] + NLSDelim +
        { 7 }    NLSData.Day.Long[DayOfWeek] + NLSDelim +
        { 8 }    ZPad( Year mod 100, 2 ) + NLSDelim
        );
end;

function TimeFmtStr (Format : String) : String;
var
    Hour, Minute, Second, Sec100: Word;
    HourX : word;
    HourS, AMPM : String;
begin
    SysGetTime(Hour,Minute,Second,Sec100);
    if Hour > 11 then
        AMPM := NLSData.PM
    else
        AMPM := NLSData.AM;
    HourX := Hour;
    if HourX > 12 then Dec(HourX, 12);
    if HourX = 0 then HourX := 12;
    HourS := IntStr(HourX);
    if Length(HourS) < 2 then HourS := #32 + HourS;
    TimeFmtStr := FormatStr(Format,
        { 0 }    Zpad(Hour, 2) + NLSDelim +
        { 1 }    Zpad(Minute, 2) + NLSDelim +
        { 2 }    Zpad(Second, 2) + NLSDelim +
        { 3 }    Zpad(Sec100, 2) + NLSDelim +
        { 4 }    HourS + NLSDelim +
        { 5 }    AMPM + NLSDelim
        );
end;

function TimeStamp(Format:byte) : String;
begin
    case Format of
        1 : TimeStamp := FormatStr(NLSData.Log.DATETIME,
            DateFmtStr(NLSData.Log.Date) + NLSDelim +
            TimeFmtStr(NLSData.Log.Time) + NLSDelim);
    else
        TimeStamp := FormatStr(NLSData.Sys.DATETIME,
            DateFmtStr(NLSData.Sys.Date) + NLSDelim +
            TimeFmtStr(NLSData.Sys.Time) + NLSDelim);
    end;
end;

function asPath(Name : String) : String;
begin
    asPath := UCase(TailDelim(Trim(Name), PathDelim));
end;

{$IFDEF LOGS}
const
  LogLine : longint = 0;
  LogInit : boolean = false;

var
    LogFile : Text;

procedure LogReset;
var
    E : integer;
begin
    Assign(LogFile, FileBase(ExeName) + 'LOG');
    Rewrite(LogFile);
    E := IOResult;
    if not E <> 0 then begin
        WriteLn(LogFile, WordHex(LogLine), ' ', MaxAvail, ' created ', FileRoot(ExeName),
        'LOG at ', TimeStamp(0));
        Inc(LogLine);
        E := IOResult;
        Close(LogFile);
        E := IOResult;
        LogInit := true;
    end;
end;

procedure Log(Message:String);
var
    E : integer;
begin
    if Not Logging then exit;
    if Not LogInit then LogReset;
    Assign(LogFile, FileBase(ExeName) + 'LOG');
    Append(LogFile);
    E := IOResult;
    if not E <> 0 then begin
        WriteLn(LogFile, WordHex(LogLine), ' ', MaxAvail, ' ', Message);
        Inc(LogLine);
        E := IOResult;
        Close(LogFile);
        E := IOResult;
    end;
end;

procedure LogStart( Message : String);
var
    E : integer;
begin
    if Not LogInit then LogReset;
    Assign(LogFile, FileBase(ExeName) + 'LOG');
    Append(LogFile);
    E := IOResult;
    if not E <> 0 then begin
        WriteLn(LogFile, ' ');
        WriteLn(LogFile, MaxAvail, '/', MemAvail, ' ##### Start Logging ' + Message);
        Inc(LogLine);
        E := IOResult;
        Close(LogFile);
        E := IOResult;
    end;
    Logging := True;
end;

procedure LogStop;
var
    E : integer;
begin
    if Not Logging then exit;
    Logging := False;
    Assign(LogFile, FileBase(ExeName) + 'LOG');
    Append(LogFile);
    E := IOResult;
    if not E <> 0 then begin
        WriteLn(LogFile, MaxAvail, '/', MemAvail, ' ##### Stop Logging');
        WriteLn(LogFile, ' ');
        Inc(LogLine);
        E := IOResult;
        Close(LogFile);
        E := IOResult;
    end;
end;

{$ENDIF}

function PrintText(P : PAsset) : boolean;
var
    I : word;
    N : boolean;
begin
    PrintText := False;
    if Not Assigned(P) then exit;
    if not Assigned(P^.Data) then exit;
    if P^.MemSize = 0 then Exit;
    for I := 0 to P^.MemSize - 1 do begin
        Write(Chars(P^.Data^)[I]);
        if (Bytes(P^.Data^)[I] <> 10) and (Bytes(P^.Data^)[I] <> 0) then
            N := (Bytes(P^.Data^)[I] = 13);
    end;
    if not N then WriteLn;
    PrintText := True;
end;

const
    VersionShown : boolean = false;

procedure GetVersionInfo(var P : Pointer; var Count : Word; Width : word);
var
    PA : PAsset;
    Pass, Index : word;
    L, T : String;
    procedure AddStr(Str : String);
    begin
        if Pass = 0 then
            Inc(Count)
        else begin
            Pointers(P^)[Index] := StrPtr(Str);
            Inc(Index);
        end
    end;
    procedure AddLn(Str : String);
    var
        X, S : String;
        I : integer;
    begin
        While Length(Str) > Width do begin
            X := '';
            for I := Width downto 1 do
                if Str[I] = #32 then begin
                    X := Copy(Str, 1, I);
                    Str := Trim(Copy(Str, I + 1, 255));
                    Break;
                end;
            if X = '' then Break;
            AddStr(Trim(X));
        end;
        AddStr(Str);
    end;
    function AddText(A : PAsset) : boolean;
    var
        I : word;
        S : String;
        L : word;
    begin
        AddText := false;
        if Not (Assigned(A) and Assigned(A^.Data)) then exit;
        I := 0;
        While I < A^.MemSize do begin
            L := 0;
            While (L < 250) and (I+L < A^.MemSize) and
            (Bytes(A^.Data^)[I+L] <> 13) and (Bytes(A^.Data^)[I+L] <> 10) do begin
                S[L + 1] := Chr(Bytes(A^.Data^)[I+L]);
                Inc(L);
            end;
            S[0] := Chr(L);
            AddLn(S);
            Inc(I,L);
            if Bytes(A^.Data^)[I] = 13 then Inc(I);
            if Bytes(A^.Data^)[I] = 10 then Inc(I);
        end;
        Addtext := I > 0;
    end;

begin
    P := nil;
    Index := 0;
    Count := 0;
    for Pass := 0 to 1 do begin
        if Pass = 1 then begin
            GetMem(P, Count * sizeof(Pointer));
            if Not Assigned(P) then Exit;
            FillChar(P^,  Count * sizeof(Pointer), 0);
        end;
        if Not VersionShown then begin
            AddLn(FormatNLS('VERSION', AppInfo.Title + NLSDelim +
                AppInfo.Version {$IFOPT D+} + ' DEBUG' {$ENDIF}));
            AddLn(RawNLS('LICENSE'));
            AddLn(FormatNLS('COPYRIGHT', AppInfo.Author + NLSDelim + AppInfo.Year));
            AddLn(RawNLS('RIGHTS'));
            AddLn('');
            T := RawNLS('TRANSLATOR');
            L := RawNLS('LANGUAGE');
            if (T <> '') and (L <> '') then begin
                AddLn(FormatNLS('THANKS', T + NLSDelim + L));
                AddLn('');
            end;
        end;
        {$IFDEF MSGTEXT}
            if AssetLoad('GOODBYE.' + Language, asOnlyAssetFile , PA) then
                AddText(PA)
            else
            if AssetLoad('GOODBYE.EN', asOnlyAssetFile , PA) then
                AddText(PA);
            if Assigned(PA) then GoodbyeMessage := true;
        {$ELSE}
            T := 'Debug build, farewell text omitted';
            AddLn(T);
            {$IFDEF LOGS} if Pass = 1 then Log(T); {$ENDIF}
        {$ENDIF}
    end;
end;

procedure PrintVersion;
var
    P : Pointer;
    C : word;
    I : integer;
begin
    if NLSTryAgain then Exit;
    GetVersionInfo(P, C, 79);
    if Assigned(P) then begin
        for I := 0 to C - 1 do begin
            WriteLn(PtrStr(Pointers(P^)[I]));
            FreeStr(PString(Pointers(P^)[I]));
        end;
        FreeMem(P, C * Sizeof(Pointer));
    end;
end;

procedure SetDefaultLanguage(P : Pointer); external;
function GetDefaultLanguage : pointer; external;
procedure SetUserLanguage(P : Pointer); external;
function GetUserLanguage : pointer; external;

function DefaultNLS(Key, Default : String) : String;
begin
    Key := RawNLS(Key);
    if Key = '' then
        Key := Default;
    DefaultNLS := Key;
end;

procedure InitNLS;
var
    P : PAsset;
    I : integer;
    S : Str12;
begin
    if not NLSTryAgain then Exit;
    {$IFDEF LOGS}
        Log('InitNLS');
    {$ENDIF}
    if AssetLoad(FileRoot(ExeName) + 'EN', asDefault, P) then begin
        SetDefaultLanguage(P^.Data);
        {$IFDEF LOGS}
            Log('Default NLS:' + FileRoot(ExeName) + 'EN' + ' ' + PtrHex(P));
        {$ENDIF}
        NLSTryAgain := False;
    end;
    if Language <> 'EN' then
        if AssetLoad(FileRoot(ExeName) + Language, asDefault, P) then begin
            SetUserLanguage(P^.Data);
            {$IFDEF LOGS}
                Log('User NLS:' + FileRoot(ExeName) + Language + ' ' + PtrHex(P));
            {$ENDIF}
            NLSTryAgain := False;
        end;
    if NLSTryAgain then begin
        {$IFDEF LOGS}
            Log('NLS assets not found, try again later.');
        {$ENDIF}
        Exit;
    end;

    NLSData.TRUEStr  := DefaultNLS('TRUE', NLSData.TRUEStr);
    NLSData.FalseStr := DefaultNLS('FALSE', NLSData.FALSEStr);

    With NLSData.Sys do begin
        Date     := DefaultNLS('SYS.DATE', Date);
        Time     := DefaultNLS('SYS.TIME', Time);
        DateTime := DefaultNLS('SYS.DATETIME', DateTime);
    end;
    With NLSData.Log do begin
        Date     := DefaultNLS('LOG.DATE', Date);
        Time     := DefaultNLS('LOG.TIME', Time);
        DateTime := DefaultNLS('LOG.DATETIME', DateTime);
    end;
    NLSData.AM  := DefaultNLS('TIME.AM', NLSData.AM);
    NLSData.PM  := DefaultNLS('TIME.PM', NLSData.AM);
    for I := 1 to 12 do begin
        S := UCase(NLSData.Month.Short[I]);
        NLSData.Month.Short[I] := DefaultNLS('MONTH.SHORT.' + S,
            NLSData.Month.Short[I]);
        NLSData.Month.Long[I] := DefaultNLS('MONTH.LONG.' + S,
            NLSData.Month.Long[I]);
    end;
    for I := 1 to 7 do begin
        S := NLSData.Day.Short[I];
        NLSData.Day.Short[I] := DefaultNLS('DAY.SHORT.' + S,
            NLSData.Day.Short[I]);
        NLSData.Day.Long[I] := DefaultNLS('DAY.LONG.' + S,
            NLSData.Day.Long[I]);
    end;
    with NLSData.Error do begin
        DriverFind  := DefaultNLS('ERROR.DRIVER.FIND',  DriverFind);
        DriverInit  := DefaultNLS('ERROR.DRIVER.INIT',  DriverInit);
        DriverMode  := DefaultNLS('ERROR.DRIVER.MODE',  DriverMode);
        FontFind    := DefaultNLS('ERROR.FONT.FIND',    FontFind);
        FontInit    := DefaultNLS('ERROR.FONT.INIT',    FontInit);
    end;
end;

procedure PrintHelp;
var
    PA : PAsset;
    T : String;
begin
    InitNLS;
    if AssetLoad('HELPTXT.' + Language, asDefault, PA) then
        PrintText(PA)
    else
    if AssetLoad('HELPTXT.EN', asDefault, PA) then
        PrintText(PA)
    else
        WriteLn(RawNLS('ERROR.NOHELP'));
    WriteLn;
    VersionShown := True;
    PrintVersion;
    Terminate(0);
end;

function Pause(ms : LongInt) : boolean;
begin
    Pause := Wait(ms, false);
end;

function Wait(ms : LongInt; Manditory : boolean ) : boolean;
var
    Event : TEvent;
    LT : LongInt;
    Forever, NeedUpdate : boolean;
begin
    LT := TimerTick;
    PurgeEvents;
    Forever := (ms = 0) and (not Manditory);
    repeat
        While (not GetEvent(Event)) do begin
            if TimerTick <> LT then break;
            Idle;
        end;
        NeedUpdate := False;
        if TimerTick <> LT then begin
            if Assigned(Video) and Assigned(MousePtr) then begin
                Video^.SpriteNext(MousePtr);
                NeedUpdate := True;
            end;
            LT := TimerTick - LT;
            if LT < 1 then LT := 1; { once per rollover, don't care! }
            ms := ms - LT * 55;
            LT := TimerTick;
        end;
        if Event.Kind and evMouse <> 0 then begin
            if Assigned(Video) and Assigned(MousePtr) then begin
                MouseMove(Event.Position);
                NeedUpdate := True;
            end;
            if (Event.Kind and evMouseClick = evMouseClick) and
            (not Manditory) then Break;
        end;
        if (Event.Kind = evKeyPress) and (not Manditory) then Break;
        if NeedUpdate then Video^.Update;
    until (ms < 1) and (not Forever);
    PurgeEvents;
    if Assigned(Video) and Assigned(MousePtr) then begin
        Video^.SpriteUndraw(MousePtr);
    end;
    Wait := ms < 1;
end;

function DRECompress(PIn, POut : Pointer; Size : LongInt ) : LongInt; { Simplified RLE scheme }
var
   RC : byte;
   CC : byte;
   SI, DI : word;
   MI : LongInt;
begin
    if not (Assigned(PIn) and Assigned(POut)) then exit;
    {$IFDEF LOGS}
        Log('DRE compress ' + IntStr(Size) +  ' bytes' );
    {$ENDIF}
    SI := 1;
    DI := 0;
    MI := Size;
    MI := MI * 9 div 10 - 10;
    RC := 1;
    CC := Bytes(PIn^)[SI];
    while (SI < Size) do begin
        if (CC <> Bytes(PIn^)[SI]) or (RC = 254) then begin
            Bytes(POut^)[DI] := RC;
            Inc(DI);
            Bytes(POut^)[DI] := CC;
            Inc(DI);
            RC := 1;
            CC := Bytes(PIn^)[SI];
            if DI >= MI then break;
        end else
            Inc(RC);
        Inc(SI);
    end;
    Bytes(POut^)[DI] := RC;
    Inc(DI);
    Bytes(POut^)[DI] := CC;
    Inc(DI);
    Bytes(POut^)[DI] := 0; { terminate Data }
    Inc(DI);
    if DI >= MI then begin
        {$IFDEF LOGS}
            Log('DRE compression should not be used, threshold of ' +
                IntStr(MI) +  ' bytes reached' );
        {$ENDIF}
        DRECompress := -1;
    end else begin
        {$IFDEF LOGS}
            Log('DRE compression reduced to ' + IntStr(DI) + ' bytes' );
        {$ENDIF}
        DRECompress := DI;
    end;
end;

function DREUncompress(PIn, POut : Pointer; Size : LongInt ) : LongInt; { Simplified RLE Decompression }
var
   RC : byte;
   CC : byte;
   SI, DI : word;
begin
    if not (Assigned(PIn) and Assigned(POut)) then exit;
    {$IFDEF LOGS}
        Log('DRE Uncompress ' + IntStr(Size) +  ' bytes' );
    {$ENDIF}
    SI := 0;
    DI := 0;
    while (SI < Size) and (DI < Size) do begin
        RC := Bytes(PIn^)[SI];
        if RC = 0 then Break;
        Inc(SI);
        CC := Bytes(PIn^)[SI];
        Inc(SI);
        if DI + RC > Size then RC := Size - DI;
        FillChar(Bytes(POut^)[DI], RC, CC);
        Inc(DI, RC);
    end;
    {$IFDEF LOGS}
        Log('DRE uncompressed to ' + IntStr(DI) + ' bytes' );
    {$ENDIF}
    DREUncompress := DI
end;

function SetVRTInterval(Interval : word) : word; {far; } external;
function GetVRTInterval : word; external;
function GetVRTimer : pointer; far; external;

{$ENDIF}