{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$IFDEF SECTINT}

{ Mostly for loading and saving of device independent bitmap data }

function NewDIBitMap(Width, Height : word) : PDIBitmap;
function NewDIBitMapFixedSize(Width, Height : word; Size, Method : word) : PDIBitmap;

procedure FreeDIBitMap(var DIBitMap : PDIBitMap);
function DIBitMapSizeOf (DIBitMap : PDIBitMap) : word;
function DIBitMapSizeData (DIBitMap : PDIBitMap) : word;
function ImageToDIBitMap (Image : PImage) : PDIBitMap;
function DIBitMapToImage (DIBitMap : PDIBitMap) : PImage;

procedure DIBitMapImplode (var DIBitMap : PDIBitMap; Method : word);
procedure DIBitMapExplode (var DIBitMap : PDIBitMap);

{$ENDIF}

{$IFDEF SECTIMP}
const
    DIBitMapHeaderSize = Sizeof(TDIBitMap) - Sizeof(TImageDataBytes);

function NewDIBitMap(Width, Height : word) : PDIBitmap;
var
    P : PDIBitmap;
    LW : word;
begin
    {$IFDEF OPTIMIZED}
    Error, Device independet DIBitMap only supports up to 256 colors
    {$ENDIF}
    P := nil;
    LW := Width and $fffc;
    if LW <> Width then Inc(LW, 4);
    {$IFDEF LOGS} Log('New DIB: ' + IntStr(Width) +
        '(' + IntStr(LW) + ')'+ 'x' + IntStr(Height) + ' ' +
        IntStr(LW * Height + DIBitMapHeaderSize) + ' bytes ');
    {$ENDIF}
    GetMem(P, LW * Height + DIBitMapHeaderSize);
    {$IFDEF LOGS} Log('DIB: ' + PtrHex(P)); {$ENDIF}
    if Assigned(P) then begin
        FillChar(P^, DIBitMapHeaderSize, 0);
        P^.Width := Width;
        P^.Height := Height;
        P^.ByteWidth := LW;
        P^.DataSize := LW * Height;
    end;
    NewDIBitMap := P;
end;

function NewDIBitMapFixedSize(Width, Height : word; Size, Method : word) : PDIBitmap;
var
    P : PDIBitmap;
    LW : word;
begin
    {$IFDEF OPTIMIZED}
    Error, Device independet DIBitMap only supports up to 256 colors
    {$ENDIF}
    P := nil;
    {$IFDEF LOGS} Log('New Fixed DIB: ' + IntStr(Width) + 'x' + IntStr(Height) + ' ' +
        IntStr(Size + DIBitMapHeaderSize) + ' bytes' ); {$ENDIF}
    LW := Width and $fffc;
    if LW <> Width then Inc(LW, 4);
    GetMem(P, Size + DIBitMapHeaderSize);
    {$IFDEF LOGS} Log('DIB: ' + PtrHex(P)); {$ENDIF}
    if Assigned(P) then begin
        FillChar(P^, DIBitMapHeaderSize, 0);
        P^.Width := Width;
        P^.Height := Height;
        P^.ByteWidth := LW;
        P^.DataSize := Size;
        P^.Compression := Method;
    end;
    NewDIBitMapFixedSize := P;
end;

procedure FreeDIBitMap(var DIBitMap : PDIBitMap);
begin
    if Not Assigned(DIBitMap) then exit;
    {$IFDEF LOGS} Log('Free DIB: ' +  IntStr(DIBitMapSizeOf(DIBitMap)) ); {$ENDIF}
    FreeMem(DIBitMap, DIBitMapHeaderSize + DIBitMap^.DataSize);
    DIBitMap := nil;
end;

function DIBitMapSizeOf (DIBitMap : PDIBitMap) : word;
begin
    {$IFDEF OPTIMIZED}
    Error, Device independet DIBitMap only supports up to 256 colors
    {$ENDIF}
    if Not Assigned(DIBitMap) then
        DIBitMapSizeOf := 0
    else
        DIBitMapSizeOf := DIBitMap^.DataSize + DIBitMapHeaderSize;
end;

function DIBitMapSizeData (DIBitMap : PDIBitMap) : word;
begin
    {$IFDEF OPTIMIZED}
    Error, Device independet DIBitMap only supports up to 256 colors
    {$ENDIF}
    if Not Assigned(DIBitMap) then
        DIBitMapSizeData := 0
    else
        DIBitMapSizeData := DIBitMap^.DataSize;
end;

function ImageToDIBitMap (Image : PImage) : PDIBitMap;
var
    I : PImage;
    P : PDIBitMap;
    X, Y : integer;
begin
    {$IFDEF OPTIMIZED}
    Error, Device independet DIBitMap only supports up to 256 colors
    {$ENDIF}
    P := nil;
    I := Image;
    if Assigned(I) and (I^.Compression <> 0) then begin
        I := Video^.CloneImage(Image);
        if Assigned(I) then
            Video^.ImageExplode(I);
    end;
    if assigned(I) and (I^.Compression = 0) then
        P := NewDIBitMap(I^.Width, I^.Height);
    if Assigned(P) then begin
        for Y := 0 to P^.Height - 1 do begin
            for X := 0 to P^.Width - 1 do begin
                P^.ImageData[X + Y * P^.ByteWidth] := Video^.ImageGetPixel(I, X, Y);
            end;
            for X := P^.Width to P^.ByteWidth - 1 do begin
                P^.ImageData[X + Y * P^.ByteWidth] := 0;
            end;
        end;
    end;
    if Assigned(I) and (I <> Image) then
        Video^.FreeImage(I);
    ImageToDIBitMap := P;
end;

function DIBitMapToImage (DIBitMap : PDIBitMap) : PImage;
var
    P : PImage;
    X, Y : integer;
begin
    {$IFDEF OPTIMIZED}
    Error, Device independet DIBitMap only supports up to 256 colors
    {$ENDIF}
    P := nil;
    if assigned(DIBitMap) then
        P := Video^.NewImage(DIBitMap^.Width,DIBitMap^.Height);
    if Assigned(P) then begin
        for Y := 0 to P^.Height - 1 do
            for X := 0 to P^.Width - 1 do begin
                Video^.ImagePutPixel(P, X, Y,
                    DIBitMap^.ImageData[X + Y * DIBitMap^.ByteWidth]);
            end;
    end;
    DIBitMapToImage := P;
end;

procedure DIBitMapRLE(var DIB : PDIBitMap);
var
   B : PDIBitMap;
   SY, SX : word;
   SC, CC, SB : byte;
   LI : word;
   LT : LongInt;
begin
    if not Assigned(DIB) then exit;
    if DIB^.Compression <> icaUncompressed then exit;
    LT := DIB^.DataSize;
    LT := LT * 9 div 10 - 10;
    {$IFDEF LOGS}
        Log('DIBitMap compress ' + PtrHex(DIB) +
            ' with ' + IntStr(LT) + ' threshold for ' +
            IntStr(DIB^.DataSize) +  ' bytes' );
    {$ENDIF}
    { exit; }
    B := NewDIBitMap(DIB^.Width, DIB^.Height);
    if not Assigned(B) then Exit;
    LI := 0;
    SB := 0;
    for SY := 0 to DIB^.Height - 1 do begin
        SX := 0;
        while SX < DIB^.ByteWidth do begin
            CC := DIB^.ImageData[SX + SY * DIB^.ByteWidth];
            Inc(SX);
            if SB = 0 then SC := CC;
            if (SC <> CC) or (SB = 254) then begin
                if LI >= LT then Break;
                B^.ImageData[LI] := SB;
                Inc(LI);
                B^.ImageData[LI] := SC;
                Inc(LI);
                SB := 1;
                SC := CC;
            end else
                Inc(SB);
        end;
        { Per some specs/ Lines should wrap. But, the Preview on OS X needed
          each line to terminate with or it wouldn't display them properly }
        B^.ImageData[LI] := SB; { write pending RLE data }
        Inc(LI);
        B^.ImageData[LI] := SC;
        Inc(LI);
        SB := 0;
        B^.ImageData[LI] := 0; { terminate line }
        Inc(LI);
        B^.ImageData[LI] := 0;
        Inc(LI);
    end;
    B^.ImageData[LI] := 0; { terminate bitmap }
    Inc(LI);
    B^.ImageData[LI] := 1;
    Inc(LI);
    if LI >= LT then begin
        {$IFDEF LOGS}
            Log('DIBitMap ' + PtrHex(DIB) + ' marked can not compress' );
        {$ENDIF}
            DIB^.Compression := icaCanNotCompress;
    end else begin
        {$IFDEF LOGS}
            Log('DIBitMap ' + PtrHex(DIB) + ' compressed to ' + IntStr(LI) + ' bytes' );
        {$ENDIF}
        FreeDIBitMap(DIB);
        GetMem(DIB, LI + DIBitMapHeaderSize);
        if assigned(DIB) then begin
            Move(B^, DIB^,  LI + DIBitMapHeaderSize);
            DIB^.Compression := icaRunLine;
            DIB^.DataSize := LI;
        end;
    end;
    FreeDIBitMap(B);
end;

procedure DIBitMapUnRLE(var DIB : PDIBitmap);
var
   B : PDIBitMap;
   SY, SX : word;
   OB : boolean;
   LI, SB, SC : word;
begin
    if not Assigned(DIB) then exit;
    if DIB^.Compression = icaCanNotCompress then
        DIB^.Compression := icaUncompressed;
    if DIB^.Compression = icaUncompressed then exit;
    {$IFDEF LOGS}
        Log('DIBitMap ' + PtrHex(DIB) + ' uncompress ' + IntStr(DIB^.DataSize) + ' bytes');
    {$ENDIF}
    B := NewDIBitMap(DIB^.Width, DIB^.Height);
    if not Assigned(B) then Exit;
    FillChar(B^.ImageData, B^.DataSize, 0);
    LI := 0;
    SX := 0;
    SY := 0;
    while (LI < DIB^.DataSize) and (SY < B^.Height) do begin
        SB := DIB^.ImageData[LI];
        Inc(LI);
        if SB = 0 then begin
            SB := DIB^.ImageData[LI];
            Inc(LI);
            case SB of
                0 : begin { End of Line }
                    SX := 0;
                    Inc(SY);
                end;
                1 : begin { End of BitMap }
                    Break;
                end;
                2 : begin { Delta offset }
                    Inc(SX, DIB^.ImageData[LI]);
                    Inc(LI);
                    Inc(SY, DIB^.ImageData[LI]);
                    Inc(LI);
                    while SX > B^.Width do begin
                        Dec(SX, B^.Width);
                        Inc(SY);
                    end;
                end;
                3..$ff : begin { absolute mode }
                    OB := SB and 1 <> 0;
                    while (SB > 0) and (LI < DIB^.DataSize) do begin
                        SC := DIB^.ImageData[LI];
                        Inc(LI);
                        if SX < B^.Width then begin
                            B^.ImageData[B^.ByteWidth * SY + SX] := SC;
                        Inc(SX);
                        end;
                        Dec(SB);
                    end;
                    if OB then begin
                        SC := DIB^.ImageData[LI];
                        { SC should be 0; absolute runs are supposed to be
                          word aligned }
                        if SC = 0 then Inc(LI);
                    end;
                end;
            end;
        end else begin
            SC := DIB^.ImageData[LI];
            Inc(LI);
            while SB > 0 do begin
                if SX < B^.Width then begin
                    B^.ImageData[B^.ByteWidth * SY + SX] := SC;
                    Inc(SX);
                end;
                Dec(SB);
            end;
        end;
    end;
    {$IFDEF LOGS}
        Log('DIBitMap ' + PtrHex(DIB) + ' uncompressed to ' +
        IntStr(B^.DataSize) + ' bytes' );
    {$ENDIF}
    FreeDIBitMap(DIB);
    DIB := B;
end;

procedure DIBitMapDRE(var DIB : PDIBitmap); { Simplified RLE scheme }
var
   B : PDIBitmap;
   Sz : LongInt;
begin
    if not Assigned(DIB) then exit;
    if DIB^.Compression <> icaUncompressed then exit;
    B := NewDIBitMap(DIB^.Width, DIB^.Height);
    if not Assigned(B) then Exit;
    Sz := DRECompress(@DIB^.ImageData, @B^.ImageData, B^.DataSize);
    if Sz <= 0 then begin
        {$IFDEF LOGS}
            Log('DIB should not be compressed' );
        {$ENDIF}
        DIB^.Compression := icaCanNotCompress;
    end else begin
        {$IFDEF LOGS}
            Log('DIB ' + PtrHex(DIB) + ' compressed from ' + IntStr(B^.DataSize) +
            ' to ' + PtrHex(B) + ' ' + IntStr(Sz) + ' bytes' );
        {$ENDIF}
        FreeDIBitMap(DIB);
        GetMem(DIB, Sz + DIBitMapHeaderSize);
        if assigned(DIB) then begin
            Move(B^, DIB^,  Sz + DIBitMapHeaderSize);
            DIB^.Compression := icaDataRun;
            DIB^.DataSize := Sz;
        end;
    end;
    FreeDIBitMap(B);
end;

procedure DIBitMapUnDRE(var DIB : PDIBitmap); { Simplified RLE Decompression }
var
   B : PDIBitmap;
   Sz : LongInt;
begin
    If Not Assigned(DIB) then Exit;
    if (DIB^.Compression = icaCanNotCompress) then
        DIB^.Compression := icaUncompressed;
    if (DIB^.Compression = icaUncompressed) then exit;
    B := NewDIBitMap(DIB^.Width, DIB^.Height);
    if not Assigned(B) then Exit;
    Sz := DREUncompress(@DIB^.ImageData, @B^.ImageData, B^.DataSize);
    {$IFDEF LOGS}
        Log('DIB ' + PtrHex(DIB) + ' uncompressed to ' +
            PtrHex(B) + ' ' + IntStr(Sz) + ' bytes' );
    {$ENDIF}
    FreeDIBitMap(DIB);
    DIB := B;
end;

procedure DIBitMapImplode (var DIBitMap : PDIBitMap; Method : word);
begin
    if not assigned(DIBitMap) then exit;
    if Method = icaRunLine4 then
        Method := icaUncompressed; { RLE-4 not supported at present }
    if Method = DIBitMap^.Compression then exit;
    if DIBitMap^.Compression <> icaUncompressed then
        DIBitMapExplode(DIBitMap);
    if DIBitMap^.Compression = icaUncompressed then
        case Method of
            icaRunLine : DIBitMapRLE(DIBitMap);
            icaDataRun : DIBitMapDRE(DIBitMap);
        end;
end;

procedure DIBitMapExplode (var DIBitMap : PDIBitMap);
begin
    if not assigned(DIBitMap) then exit;
    case DIBitMap^.Compression of
        icaUncompressed : begin end; { ignore }
        icaRunLine : DIBitMapUnRLE(DIBitMap);
        icaDataRun : DIBitMapUnDRE(DIBitMap);
        icaCanNotCompress : DIBitMapUnRLE(DIBitMap); { just to flip comp off }
    else
        SetError(erOperation_Not_Supported);
    end;
end;

{$ENDIF}
