; BSD 3-Clause License
; Copyright (c) 2021, Jerome Shidel
; All rights reserved.

; NASM 2.14rc0 for DOS

use16

cpu 386

%include "TPASCAL.INC"

%idefine NotOptimal

%idefine MaxWidth       320
%idefine MaxHeight      200
%idefine MaxColorDepth  8
%idefine MaxColors      256
%idefine MaxProfiles    1
%idefine MemSize        MaxWidth * MaxHeight
%idefine Updating       Partial         ; Partial, Complete, Always

%include "DRVSTRUC.INC"

Header:
    istruc THeader
        at THeader.Platform,        db 7,'INFERNO'
        at THeader.Class,           db 5,'VIDEO'
        at THeader.Name,            db 8,'VGA-i386'
        at THeader.Version,         dw 1            ; version 1
        at THeader.VersionCompat,   dw 0            ; Inferno pre-release devel
        at THeader.CPU,             dw 3            ; 386
        at THeader.Flags,           dw 0
        at THeader.MinimumMemory,   dd MemSize
        at THeader.Width,           dw MaxWidth
        at THeader.Height,          dw MaxHeight
        at THeader.ColorDepth,      dw MaxColorDepth
        at THeader.Colors,          dw MaxColors
        at THeader.ColorProfiles,   dw MaxProfiles
    iend

Core:
    istruc TCore ; Absoultely required functions
        at TCore.GetCopyright,      dw GetCopyright
        at TCore.GetLicense,        dw GetLicense
        at TCore.GetModes,          dw GetModes
        at TCore.OpenVideo,         dw OpenVideo
        at TCore.CloseVideo,        dw CloseVideo
        at TCore.SetSync,           dw SetSync
        at TCore.SetBuffered,       dw SetBuffered
        at TCore.UpdateVideo,       dw UpdateVideo
        at TCore.UpdateForce,       dw UpdateForce
        at TCore.GetPalettes,       dw GetPalettes
        at TCore.SetPalettes,       dw SetPalettes
        at TCore.GetProfile,        dw GetProfile
        at TCore.GetPixel,          dw GetPixel
        at TCore.PutPixel,          dw PutPixel
        at TCore.ImageSize,         dw ImageSize
        at TCore.ImageGetPixel,     dw ImageGetPixel
        at TCore.ImagePutPixel,     dw ImagePutPixel
    iend

Performance:
    istruc TPerformance
        at TPerformance.Fill,               dw Fill
        at TPerformance.Region,             dw Region
        at TPerformance.ImageFill,          dw ImageFill
        at TPerformance.ImageRegion,        dw ImageRegion
        at TPerformance.GetImage,           dw GetImage
        at TPerformance.PutImage,           dw PutImage
        at TPerformance.PutImageMode,       dw PutImageMode
        at TPerformance.PutMaskMode,        dw PutMaskMode
        at TPerformance.PutBits,            dw PutBits
        at TPerformance.ShiftRegion,        dw ShiftRegion
    iend

HighLevel:
    istruc THighLevel ; These are provided by the graphics subsystem
    iend

Optional:
    istruc TOptional
        at TOptional.ExtendedData,        dw GetExtendedData ; debugging phase/data
    iend

    dd  0xffffffff  ; end of functions list

Copyright:
    db  'Copyright (C) 2021, Jerome Shidel',13
    db  'All rights reserved.',0
License:
    db  'BSD 3-Clause License',0

ModeTable: ; Best mode first
    dw  0x0013, 320, 200, 8
    ; dw  0x0012, 640, 480, 4 ; Not supported in this driver
    dw  0 ; end of mode list

; internal data
%ifidni Updating, partial
    PartialUpdates:
        dw      1
    UpdateStart:
        dw      0
    UpdateEnd:
        dw      0
%else
    UpdateFlag:
        dw 0
%endif

ExtData: ; extended data, can be any size, not used by Inferno (except to
         ; do devel debugging). Could be used by a custom driver to pass
         ; non-standard data to an app. recommend first word be size of data
         ; being returned. The GetExtendedData returns a pointer to this
         ; data or nil.
    times 16 dw 0

Syncing:
    dw      1
PrevMode:
    dw      0
Frames:
    dd      0
Buffering:
    dw      0
Stage:
    dd      0
MemBlock:
    dd      0
Screen:
    dw      0x0000,0xa000

PaletteBuffer:
    times MaxColors * 3     db 0

Profile_0:
DefaultPalette:
    %include "VGA.PAL"

%imacro Update 1-*
    %ifnidni Updating, partial
        %ifidni %1, none
            mov [UpdateFlag], word 0
        %else
            mov [UpdateFlag], word 1
        %endif
    %elifidni %1, none
        mov [UpdateStart], word MaxHeight - 1
        mov [UpdateEnd], word 0
    %elifidni %1, maybe
        ; for now only PrepareVideo uses this
    %elifidni %1, all
        mov [UpdateStart], word 0
        mov [UpdateEnd], word MaxHeight - 1
    %elifidni %1, line
        cmp     %2, [UpdateStart]
        ja      %%start_ok
        mov     [UpdateStart], %2
        %%start_ok:
        cmp     %2, [UpdateEnd]
        jb      %%end_ok
        mov     [UpdateEnd], %2
        %%end_ok:
    %elifidni %1, rows
        cmp     %2, [UpdateStart]
        ja      %%first_start_ok
        mov     [UpdateStart], %2
        %%first_start_ok:
        cmp     %2, [UpdateEnd]
        jb      %%first_end_ok
        mov     [UpdateEnd], %2
        %%first_end_ok:
        cmp     %3, [UpdateStart]
        ja      %%second_start_ok
        mov     [UpdateStart], %3
        %%second_start_ok:
        cmp     %3, [UpdateEnd]
        jb      %%second_end_ok
        mov     [UpdateEnd], %3
        %%second_end_ok:
    %else
        %error %1 is an unknown update requirement
    %endif
%endmacro

%imacro VerticalSync 0
    mov     ax, [cs:Syncing]
    test    ax, ax
    jz      %%NoSync
    mov     dx,  0x03da
    mov     ah, 0x08
%%WaitSyncA:
    in      al, dx
    test    al, ah
    jz      %%WaitSyncA
%%WaitSyncB:
    in      al, dx
    test    al, ah
    jnz     %%WaitSyncB
%%NoSync:
%endmacro

ResetInternals:
    Update  None
    xor     eax, eax
    mov     [Frames], eax
    ; Save Current Video Mode
    mov     ah, 0x0f
    int     0x10
    xor     ah, ah
    mov     [PrevMode], ax
    xor     ax, ax
    mov     [Header + THeader.Font], ax
    mov     [Header + THeader.Font + 2], ax
    mov     [Header + THeader.Monospace], ax
    mov     [Header + THeader.Direction], word 6    ; dmRight - Left to Right
ret

; Core Functions
fpascal GetCopyright, pointer
; function : pointer;
; We just returning a pointer. Declaring as fpascal to avoid PUSH/POP DS
    push    cs
    pop     dx
    mov     ax, Copyright      ; DX:AX
epascal

fpascal GetLicense, pointer
; function : pointer;
; We just returning a pointer. Declaring as fpascal to avoid PUSH/POP DS
    push    cs
    pop     dx
    mov     ax, License      ; DX:AX
epascal

fpascal GetModes, pointer
; function : pointer;
; We just returning a pointer. Declaring as fpascal to avoid PUSH/POP DS
    push    cs
    pop     dx
    mov     ax, ModeTable       ; DX:AX
epascal

fpascal OpenVideo, word, integer
; function (Mode : word) : integer;
    mov     ax, 1
    mov     [cs:Buffering], ax
    mov     eax, [cs:MemBlock]
    mov     [cs:Stage], eax
    mov     ax, [cs:Stage + 2]
    test    ax, ax
    jnz     .MemOk
    pcall   [cs:HighLevel + THighLevel.MemAlloc], word MemSize, word True
    test     dx, dx
    jz      .MemError
    mov     [cs:Stage], ax
    mov     [cs:Stage + 2], dx
    mov     [cs:MemBlock], ax
    mov     [cs:MemBlock + 2], dx
.MemOk:
    push    ds
    push    cs
    pop     ds
    call    ResetInternals
    mov     ax, PARAM_1
    cmp     ax, 0x0013
    jne     .ModeError
    int     0x10    ; AH = 0, set video mode
    Update  All
    ; Set Default Color Palette
    push    ds
    pop     es
    mov     dx, DefaultPalette
    mov     ax, 0x1012
    xor     bx, bx
    mov     cx, 0x100
    int     0x10
    ; clear Screen & stage / zero fill
    mov     ecx, MemSize / 4
    xor     eax, eax
    cld
    push    ecx
    les     di, [Stage]
    rep     stosd
    pop     ecx
    les     di, [Screen]
    rep     stosd
    pop     ds
    mov     RESULT, 0   ; No Error
    jmp     .Done
.ModeError:
    pop     ds
    mov     RESULT, 1   ; Invalid Function
    jmp     .Done
.MemError:
    mov     RESULT, 8   ; Out of Memory/Not Assigned
    jmp     .Done
.Done:
epascal

ppascal CloseVideo
    mov     eax, [cs:MemBlock]
    mov     [cs:Stage], eax
    mov     ax, [cs:Stage + 2]
    test    ax, ax
    jz      .MemNotSet
    pcall   [cs:HighLevel + THighLevel.MemRelease], cs, word Stage, word MemSize
    xor     eax, eax
    mov     [cs:Stage], eax
    mov     [cs:MemBlock], eax
.MemNotSet:
    push    ds
    push    cs
    pop     ds
    ; Check Video Mode
    mov     ah, 0x0f
    int     0x10
    xor     ah, ah
    mov     bx, [PrevMode]
    cmp     ax, bx
    je      .Done
    ; Reset To Text Mode
    mov     ax, [PrevMode]
    xor     ah, ah
    int     0x10
.Done:
    call    ResetInternals
    pop     ds
epascal

pdriver SetSync, boolean
; procedure (Enabled : boolean);
    mov ax, PARAM_1
    xor ah, ah
    mov [cs:Syncing], ax
edriver

pdriver SetBuffered, boolean
; procedure (Enabled : boolean);
    mov     ax, PARAM_1
    cmp     ax, False
    je      .NotBuffered
    mov     eax, [cs:MemBlock]
    mov     bx, 1
    jmp     .SetStage
.NotBuffered:
    mov     eax, [cs:Screen]
    xor     bx, bx
.SetStage:
    mov     [cs:Stage],eax
    mov     [cs:Buffering], bx
edriver

pdriver UpdateVideo
; procedure;
; *driver saves/restores previous DS
%ifidni Updating, partial
    mov     ax, [Buffering]
    test    ax, ax
    jz      .SkipUpdate
    mov     bx, [UpdateStart]
    mov     dx, [UpdateEnd]
    cmp     dx, MaxHeight
    jb      .EndOk
    mov     dx, MaxHeight - 1
.EndOk:
    cmp     bx, dx
    ja      .SkipUpdate
    Update  None
    mov     eax, [Frames]
    inc     eax
    mov     [Frames], eax
    les     di, [Screen]
    lds     si, [Stage]
    mov     ax, [PartialUpdates]
    test    ax, ax
    jz      .FullUpdate
    push    dx
    mov     ax, MaxWidth
    mul     bx
    add     di, ax
    add     si, ax
    pop     dx
    sub     dx, bx
    inc     dx
    mov     ax, MaxWidth
    mul     dx
    shr     ax, 2   ; divide by 4
    cwd
    mov     ecx, eax
    jmp     .DoUpdate
.FullUpdate:
%else
    %ifnidni Updating, Always
        mov     ax, [UpdateFlag]
        test    ax,ax
        jz      .SkipUpdate
        Update  None
    %endif
    mov     eax, [Frames]
    inc     eax
    mov     [Frames], eax
    les     di, [Screen]
    lds     si, [Stage]
%endif
    mov     ecx, MemSize / 4
.DoUpdate:
    push    eax
    VerticalSync
    pop eax
    cld
    rep     movsd
.SkipUpdate:
edriver

pdriver UpdateForce
    Update All
edriver

pdriver GetPalettes, pointer
; procedure (var Palettes : TPalettes);
    les     dx, PARAM_1
    mov     ax, 0x1017
    xor     bx, bx
    mov     cx, 0x100
    int     0x10
    ; multiply palette values by 4
    cld
    mov     di, dx
    mov     cl, 2
    add     dx, MaxColors * 3
.UpConvert:
    mov     al, [es:di]
    shl     ax, cl
    stosb
    cmp     di, dx
    jne     .UpConvert
edriver

pdriver SetPalettes, pointer
; procedure (var Palettes : TPalettes);
    ; copy then divide palette values by 2
    cld
    lds     si, PARAM_1
    push    ds
    pop     es
    mov     di, PaletteBuffer
    push    di
    mov     dx, di
    mov     cl, 2
    add     dx, MaxColors * 3
.DownConvert:
    lodsb
    shr     ax, cl
    stosb
    cmp     di, dx
    jne     .DownConvert
    VerticalSync
    pop     dx ; was di
    mov     ax, 0x1012
    xor     bx, bx
    mov     cx, 0x100
    int     0x10
edriver

fdriver GetProfile, word, pointer, boolean
; function (Profile : word; var Palettes : TPalettes): boolean;
; We just returning a pointer. Declaring as fpascal to avoid PUSH/POP DS
    mov     ax, PARAM_1
    cmp     ax, 1
    ja      .DoesNotExist ; 0=default, 1=profile 0, only one profile for now
    ; copy default palette
    cld
    les     di, PARAM_2
    mov     si, Profile_0
    mov     ecx, 192    ; MaxColors * 3 / 4
    push    di
    rep     movsd
    mov     dx, di
    pop     di
    ; UpConvert
    mov     cl, 2
.UpConvert:
    mov     al, [es:di]
    shl     ax, cl
    stosb
    cmp     di, dx
    jne     .UpConvert
    mov     RESULT, TRUE
    jmp     .Done
.DoesNotExist:
    mov     RESULT, FALSE
.Done:
edriver

fdriver GetPixel, integer, integer, word
; function (x, y : integer) : word;
    les     di, [Stage]
    mov     dx, MaxHeight
    mov     ax, PARAM_2
    cmp     ax, dx
    jnb     .NoGet
    mov     cx, MaxWidth
    mov     bx, PARAM_1
    cmp     bx, cx
    jnb     .NoGet
    mul     cx
    add     di, ax
    add     di, bx
    mov     ax, [es:di]
    jmp     .Done
.NoGet:
    xor     ax, ax
.Done:
edriver

pdriver PutPixel, integer, integer, word
; procedure (x, y : integer; Color : TColor);
    les     di, [Stage]
    mov     dx, MaxHeight
    mov     ax, PARAM_2
    cmp     ax, dx
    jnb     .NoDraw
    mov     cx, MaxWidth
    mov     bx, PARAM_1
    cmp     bx, cx
    jnb     .NoDraw
    Update  Line, ax
    mul     cx
    add     di, ax
    add     di, bx
    mov     ax, PARAM_3
    stosb
.NoDraw:
edriver

fdriver ImageSize, word, word, word
; function (Width, Height : word) : word;
    mov     ax, PARAM_1
    mov     cx, PARAM_2
    mul     cx
    cmp     dx, 0
    jne     .TooBig
    mov     cx, ax
    and     cx, 0x0003
    jz      .EvenFour   ;
    and     ax, 0xfffc
    add     ax, 4
    jo      .TooBig
.EvenFour:
    add     ax, ImageHeadSize  ; add size of image control header
    jno     .Done
.TooBig:
    mov     ax, 0
.Done:
edriver

%imacro ImageHeader 2
; out ax=byte width, bx=image width, dx=image height, cx=data byte size
    les     di, %2
    %ifidni %1, var
        mov     bx, [es:di + 2]
        push    bx
        mov     di, [es:di]
        pop     es
    %endif
    clc
    mov     al, [es:di+6] ; compression
    cmp     al, 0       ; 0=uncompressed
    je      .Uncompressed
    cmp     al, 255     ; 255=not compressible
    je      .Uncompressed
    stc
.Uncompressed:
    pushf
    mov     bx, [es:di] ; width
    mov     dx, [es:di+2] ; height
    mov     ax, [es:di+4] ; LineWidth
    mov     cx, [es:di+8] ; DataSize
    add     di, ImageHeadSize    ; point to first pixel
    popf
%endmacro

fdriver ImageGetPixel, pointer, integer, integer, word
; function (Image : PImage; X, Y : integer) : word;
    ImageHeader pointer, PARAM_1
    jc      .Done
    mov     bx, PARAM_2 ; x
    mov     cx, PARAM_3 ; y
    cmp     bx, ax
    jnb     .Outside
    cmp     cx, dx
    jnb     .Outside
    mul     cx
    add     bx, ax
    xor     ah, ah
    mov     al, [es:di+bx]
    jmp     .Done
.Outside:
    xor     ax, ax
.Done:
edriver

pdriver ImagePutPixel, pointer, integer, integer, word
; procedure (var Image : PImage; X, Y : integer; Color : word);
    ImageHeader var, PARAM_1
    jc      .Done
    mov     bx, PARAM_2 ; x
    mov     cx, PARAM_3 ; y
    cmp     bx, ax
    jnb     .Done    ; outside image
    cmp     cx, dx
    jnb     .Done    ; Outside image
    mul     cx
    add     bx, ax
    mov     ax, PARAM_4
    mov     [es:di+bx], al
.Done:
edriver

pdriver Fill, word
; procedure (Color : TColor);
    les     di, [Stage]
    mov     ecx, MemSize / 4
    mov     bx, PARAM_1
    mov     bh, bl
    mov     ax, bx
    shl     eax, 16
    mov     ax, bx
    cld
    rep     stosd
    Update  All
edriver

CalcRegionFunc:
; input, ax=x1, dx=y1, cx=x2, bx=y2
; output, es:di = stage offset, cx = width, bx = height, dx=row increment, ax=?
; cld & clc if ok, stc if not ok
    cmp     ax, cx
    ja      .Error  ; treat integer like word, no < 0 pixels positions
    cmp     dx, bx
    ja      .Error
    ; Should probably also test that it is not outside of Stage area
    Update  Rows, dx, bx
    sub     cx, ax
    inc     cx
    sub     bx, dx
    inc     bx
    les     di, [Stage]
    ; ax=x, dx=y, cx=width, bx=height, di=stage
    push    ax
    push    cx
    mov     ax, dx
    mov     cx, MaxWidth
    push    cx
    mul     cx
    pop     dx
    add     di, ax
    pop     cx
    pop     ax
    add     di, ax
    sub     dx, cx
    clc
    ret
.Error:
    stc
    ret

%imacro CalcRegion 4
    mov     ax, %1
    mov     dx, %2
    mov     cx, %3
    mov     bx, %4
    call    CalcRegionFunc
%endmacro

pdriver Region, integer, integer, integer, integer, word
; procedure (x1, y1, x2, y2 : integer; Color : TColor);
    CalcRegion PARAM_1, PARAM_2, PARAM_3, PARAM_4
    jc      .Done
    mov     ax, PARAM_5
    ; es:di = stage offset, cx = width, bx = height, dx=row increment, al=color
    cld
.DrawLoop:
    push    cx
    rep     stosb
    pop     cx
    add     di, dx
    dec     bx
    jnz     .DrawLoop
.Done:
edriver

pdriver ImageFill, pointer, word
; procedure (var Image : PImage; Color : TColor);
    ImageHeader var, PARAM_1
    jc      .Done
    mov     ax, PARAM_5
    cld
    rep     stosb
.Done:
edriver

pdriver ImageRegion, pointer, integer, integer, integer, integer, word
; procedure (var Image : PImage; x1, y1, x2, y2 : integer; Color : TColor);
    ImageHeader var, PARAM_1
    jc      .Done
    push    di
    mov     si, bx
    mov     di, dx
    ; STACK DI=image, si=image width, di=image height
    mov     ax, PARAM_2
    mov     dx, PARAM_3
    mov     cx, PARAM_4
    mov     bx, PARAM_5
    cmp     ax, cx
    ja      .Error  ; treat integer like word, no < 0 pixels positions
    cmp     dx, bx
    ja      .Error
    ; x1 < x2 and y1 < y2, passed
    cmp     ax, si
    ja      .Error
    cmp     cx, si
    ja      .Error
    cmp     dx, di
    ja      .Error
    cmp     bx, di
    ja      .Error
    ; All points inside image, passed
    sub     cx, ax
    inc     cx
    sub     bx, dx
    inc     bx
    pop     di
    ; fill ax=x, dx=y, cx=width, bx=height, si=image width, di=image origin
    push    ax
    push    cx
    mov     ax, dx
    mov     cx, si
    push    cx
    mul     cx
    pop     dx
    add     di, ax
    pop     cx
    pop     ax
    add     di, ax
    mov     ax, PARAM_6
    sub     dx, cx
    ; di=image offset, cx=width, bx=height, dx=row increment, al=color
    cld
.DrawLoop:
    push    cx
    rep     stosb
    pop     cx
    add     di, dx
    dec     bx
    jnz     .DrawLoop
    jmp     .Done
.Error:
    pop     di
.Done:
edriver

pdriver ShiftRegion, integer, integer, integer, integer, word, word, word
; it would be faster to do diagonal shifts instead of using double straight shifts
; but it was giving me a headache. So, maybe I'll do that later. For now,
; broke the standard directions into sub calls, so things like UpLeft will just
; do a up shift then a left shift instead of all at once.
;procedure (x1, y1, x2, y2 : integer; Direction: TDirection; Count : Word; Fill : TFillColor);
    mov     ax, [STACK_1 + 2]   ; x1
    mov     dx, [STACK_2 + 2]   ; y1
    mov     cx, [STACK_3 + 2]   ; x2
    mov     bx, [STACK_4 + 2]   ; y2
    call    CalcRegionFunc
    ; es:di=stage, cx=width, bx=height, dx=row increment, ax=?, cld, stc=error
    jc      .Done
    mov     ax, [STACK_5 + 2] ; direction
    test    ax, ax
    jz      .Done
    test    ax, 1
    jz      .EvenDirections  ; up, down, left, right
    cmp     ax, 3
    jb      .ShiftDownLeft
    je      .ShiftDownRight
    cmp     ax, 7
    je      .ShiftUpLeft
    cmp     ax, 9
    je      .ShiftUpRight
    jmp     .Done
.EvenDirections:
    cmp     ax, 4
    jb      .ShiftDown
    je      .ShiftLeft
    cmp     ax, 6
    je      .ShiftRight

.ShiftUp:
    call    .RegionShiftUp
    jmp     .Done

.RegionShiftUp:  ; es:di=stage, cx=width, bx=height, dx=row increment, cld
    mov     si, di
    mov     ax, [STACK_7 + 0] ; fill color
    push    ax
    mov     ax, [STACK_6 + 2] ; count
    push    ax
    push    es
    pop     ds
    cmp     ax, bx ; count >= height
    jae     .FillAll
    sub     bx, ax ; lines needing shifted
    push    dx
    mov     dx, MaxWidth
    mul     dx
    add     si, ax
    pop     dx
    cld
.ShiftUpLoop:
    push    cx
    rep     movsb
    pop     cx
    add     di, dx
    add     si, dx
    dec     bx
    jnz     .ShiftUpLoop
    jmp     .ShiftUpFill
.FillAll:
    cld
    ; ax=fill color & bx=count on the stack, make count entire screen
    mov     ax,bx
    pop     bx
    push    ax      ; replace count with height
.ShiftUpFill:
    pop     bx ; was ax=count
    pop     ax ; was ax=fill
.ShiftUpFillLoop:
    push    cx
    rep     stosb
    pop     cx
    add     di, dx
    dec     bx
    jnz     .ShiftUpFillLoop
    ret

.ShiftDown:
    call    .RegionShiftDown
    jmp     .Done

.RegionShiftDown:      ; es:di=stage, cx=width, bx=height, dx=row increment
    mov     ax, [STACK_7 + 0]   ; fill color
    push    ax                  ; push fill color
    mov     ax, [STACK_6 + 2]   ; count
    push    ax                  ; push count
    push    es
    pop     ds
    cmp     ax, bx              ; count >= height
    jae     .FillAll

    push    ax                  ; count
    sub     bx, ax              ; rows to shift
    push    dx
    mov     dx, MaxWidth
    mov     ax, bx
    dec     ax
    mul     dx
    add     di, ax
    add     di, cx              ; di=last pixel/row of source
    dec     di
    pop     dx
    pop     ax                  ; ax=count
    mov     si, di              ; si source start
    push    dx
    mov     dx, MaxWidth
    mul     dx
    add     di, ax              ; di=last pixel/row of dest, dest start
    pop     dx
    std
.ShiftDownLoop:
    push    cx
    rep     movsb
    pop     cx
    sub     di, dx
    sub     si, dx
    dec     bx
    jnz     .ShiftDownLoop
    pop     bx                  ; was ax, bx=count
    pop     ax                  ; ax=fill color
.ShiftDownFillLoop:
    push    cx
    rep     stosb
    pop     cx
    sub     di, dx
    dec     bx
    jnz     .ShiftDownFillLoop
    ret

.ShiftLeft:
    call    .RegionShiftLeft
    jmp     .Done

.RegionShiftLeft: ; es:di=stage, cx=width, bx=height, dx=row increment
    mov     si, di
    mov     ax, [STACK_7 + 0]       ; fill color
    push    ax
    mov     ax, [STACK_6 + 2]       ; count
    push    ax
    push    es
    pop     ds
    cmp     ax, cx                  ; count >= width
    jae     .FillAll
    sub     cx, ax                  ; pixels needing shifted
    add     si, ax                  ; si= source pixel
    add     dx, ax                  ; row width
    pop     ax                      ; discard old count
    pop     ax                      ; was ax=fill
    cld
.ShiftLeftLoop:
    push    cx
    rep     movsb
    push    di
    mov     cx, si
    sub     cx, di
    rep     stosb
    pop     di
    pop     cx
    add     di, dx
    add     si, dx
    dec     bx
    jnz     .ShiftLeftLoop
    ret

.ShiftRight:
    call    .RegionShiftRight
    jmp     .Done
.RegionShiftRight: ; es:di=stage, cx=width, bx=height, dx=row increment
    mov     ax, [STACK_7 + 0]   ; fill color
    push    ax                  ; push fill color
    mov     ax, [STACK_6 + 2]   ; count
    push    ax                  ; push count
    push    es
    pop     ds
    cmp     ax, cx              ; count >= height
    jae     .FillAll
    push    ax                  ; count
    push    dx
    mov     dx, MaxWidth
    mov     ax, bx
    dec     ax
    mul     dx
    add     di, ax
    add     di, cx              ; di=last pixel/row of source
    dec     di
    pop     dx
    pop     ax                  ; ax=count
    mov     si, di              ; si source start
    sub     si, ax
    pop     ax                  ; old count
    sub     cx, ax
    add     dx, ax
    pop     ax                  ; was ax=fill
    std
.ShiftRightLoop:
    push    cx
    rep     movsb
    push    di
    mov     cx, di
    sub     cx, si
    rep     stosb
    pop     di
    pop     cx
    sub     di, dx
    sub     si, dx
    dec     bx
    jnz     .ShiftRightLoop
    ret

.ShiftUpLeft: ; es:di=stage, cx=width, bx=height, dx=row increment, ax=?
    pusha
    call    .RegionShiftUp
    popa
    call    .RegionShiftLeft
    jmp     .Done

.ShiftUpRight:
    pusha
    call    .RegionShiftUp
    popa
    call    .RegionShiftRight
    jmp     .Done

.ShiftDownLeft:
    pusha
    call    .RegionShiftDown
    popa
    call    .RegionShiftLeft
    jmp     .Done

.ShiftDownRight:
    pusha
    call    .RegionShiftDown
    popa
    call    .RegionShiftRight

.Done:

edriver


fdriver GetExtendedData, pointer
    push    cs
    pop     dx
    mov     ax, ExtData
edriver


pdriver GetImage, pointer, integer, integer
; procedure (var Image : PImage; X, Y : integer);

    push    ds
    ImageHeader var, PARAM_1
    ; es:di=image ax=byte width, bx=image width, dx=image height, cx=data size
    push    bx
    push    dx
    push    ax
    push    cx
    jc      .Done

    %define PixelWidth  [ss:bp-6]
    %define PixelHeight [ss:bp-8]
    %define ByteWidth   [ss:bp-10]
    %define ByteCount   [ss:bp-12]

    mov     dx, PARAM_3
    push    dx
    mov     ax, MaxWidth
    mul     dx
    mov     bx, PARAM_2
    add     ax, bx ; ax is stage pixel offset
    lds     si, [Stage]
    add     si, ax
    pop     dx
    mov     cx, PixelWidth
    cld
.CopyImage:
    push    di
    push    si
    push    cx
    push    bx
.CopyRow:
    xor     ax, ax
    cmp     dx, ax
    jl      .NoPixel
    cmp     bx, ax
    jl      .NoPixel
    cmp     bx, MaxWidth
    jge     .NoPixel
    cmp     dx, MaxHeight
    jge     .NoPixel
    mov     al, [si]
.NoPixel:
    inc     si
    stosb
    inc     bx
    loop    .CopyRow
    pop     bx
    pop     cx
    pop     si
    pop     di
    mov     ax, PixelHeight
    dec     ax
    jz      .DoneCopy
    mov     PixelHeight, ax
    add     si, MaxWidth
    add     di, ByteWidth
    inc     dx
    jmp     .CopyImage

.DoneCopy:

    %undef ScreenRow
    %undef ByteWidth
    %undef PixelWidth
    %undef ByteCount
    %undef PixelHeight

    .Done:
    pop     dx
    pop     cx
    pop     bx
    pop     ax
    pop     ds
edriver

pdriver PutImage, pointer, integer, integer
; procedure (Image : PImage; X, Y : integer);
    push    ds
    ImageHeader const, PARAM_1
    ; es:di=image ax=byte width, bx=image width, dx=image height, cx=data size
    push    bx
    push    dx
    push    ax
    push    cx
    pushf

    %define PixelWidth  [ss:bp-6]
    %define PixelHeight [ss:bp-8]
    %define ByteWidth   [ss:bp-10]
    %define ByteCount   [ss:bp-12]

    mov     dx, PARAM_3
    push    dx
    mov     ax, MaxWidth
    mul     dx
    pop     dx

    mov     bx, PARAM_2
    add     ax, bx; ax is stage pixel offset

    push    ax
    mov     ax, dx
    cmp     ax, MaxHeight
    jge     .DoneCopy
    cmp     ax, 0
    jge     .StartOK
    xor     ax, ax
.StartOK:
    push    ax
    mov     ax, dx
    add     ax, PixelHeight
    dec     ax
    cmp     ax, 0
    jl      .NoCopy
    cmp     ax, MaxHeight
    jl      .EndOK
    mov     ax, MaxHeight - 1
    jmp     .EndOK
.NoCopy:
    pop     ax
    jmp     .DoneCopy
.EndOK:
    Update  Line, ax
    pop     ax
    Update  Line, ax

    pop     ax
    push    di
    pop     si
    push    es
    pop     ds

    les     di, [cs:Stage]
    add     di, ax
    mov     cx, PixelWidth
    popf
    cld
    pushf
    jc      .CompressedCopyStart
.CopyImage:
    cmp     dx, MaxHeight
    jge     .DoneCopy
    cmp     dx, 0
    jl      .SkipRow
    push    di
    push    si
    push    cx
    push    bx
.CopyRow:
    cmp     bx, MaxWidth
    jae     .NoPixel
    movsb
    inc     bx
    loop    .CopyRow
    jmp     .RowFinished
.NoPixel:
    inc     di
    inc     si
    inc     bx
    loop    .CopyRow
.RowFinished:
    pop     bx
    pop     cx
    pop     si
    pop     di
.SkipRow:
    mov     ax, PixelHeight
    dec     ax
    jz      .DoneCopy
    mov     PixelHeight, ax
    add     si, ByteWidth
    add     di, MaxWidth
    inc     dx
    jmp     .CopyImage

.CompressedCopyStart:
    lodsw
.CompressedCopyImage:
    cmp     dx, MaxHeight
    jge     .DoneCopy
    push    di
    push    cx
    push    bx
.CompressedCopyRow:
    cmp     dx, 0
    jl      .CompressedNextPixel
    cmp     bx, MaxWidth
    jae     .CompressedNextPixel
    mov     [es:di], ah
.CompressedNextPixel:
    inc     di
    inc     bx
    dec     al
    jnz      .CompressedContinue
    lodsw
.CompressedContinue:
    loop    .CompressedCopyRow
    jmp     .CompressedRowFinished

.CompressedRowFinished:
    pop     bx
    pop     cx
    pop     di
.CompressedSkipRow:
    push    ax
    mov     ax, PixelHeight
    dec     ax
    mov     PixelHeight, ax
    pop     ax
    jz      .DoneCopy
    add     di, MaxWidth
    inc     dx
    jmp     .CompressedCopyImage
.DoneCopy:

    %undef ScreenRow
    %undef ByteWidth
    %undef PixelWidth
    %undef ByteCount
    %undef PixelHeight

.Done:
    popf
    pop     dx
    pop     cx
    pop     bx
    pop     ax
    pop     ds
edriver

pdriver PutImageMode, pointer, integer, integer, word
; procedure (Image : PImage; X, Y : integer);
    push    ds
    ImageHeader const, PARAM_1
    ; es:di=image ax=byte width, bx=image width, dx=image height, cx=data size
    push    bx
    push    dx
    push    ax
    push    cx
    push    dx
    pushf

    %define PixelWidth  [ss:bp-6]
    %define PixelHeight [ss:bp-8]
    %define ByteWidth   [ss:bp-10]
    %define ByteCount   [ss:bp-12]
    %define ScreenRow   [ss:bp-14]

    mov     dx, PARAM_3
    mov     ScreenRow, dx
    push    dx
    mov     ax, MaxWidth
    mul     dx
    pop     dx

    mov     bx, PARAM_2
    add     ax, bx; ax is stage pixel offset

    push    ax
    mov     ax, dx
    cmp     ax, MaxHeight
    jge     .DoneCopy
    cmp     ax, 0
    jge     .StartOK
    xor     ax, ax
.StartOK:
    push    ax
    mov     ax, dx
    add     ax, PixelHeight
    dec     ax
    cmp     ax, 0
    jl      .NoCopy
    cmp     ax, MaxHeight
    jl      .EndOK
    mov     ax, MaxHeight - 1
    jmp     .EndOK
.NoCopy:
    pop     ax
    jmp     .DoneCopy
.EndOK:
    Update  Line, ax
    pop     ax
    Update  Line, ax

    pop     ax
    push    di
    pop     si
    push    es
    pop     ds

    les     di, [cs:Stage]
    add     di, ax
    mov     cx, PixelWidth
    popf
    cld
    pushf
    jc      .CompressedCopyStart

    ; Set Uncompressed Mode Jump
    mov     dx, .CopyXOR
    cmp     word PARAM_4, 4
    je      .CopyImage
    mov     dx, .CopyOR
    cmp     word PARAM_4, 3
    je      .CopyImage
    mov     dx, .CopyNOT
    cmp     word PARAM_4, 2
    je      .CopyImage
    mov     dx, .CopyAND
    cmp     word PARAM_4, 1
    je      .CopyImage
    mov     dx, .CopyCopy

.CopyImage:
    cmp     word ScreenRow, MaxHeight
    jge     .DoneCopy
    cmp     word ScreenRow, 0
    jl      .SkipRow
    push    di
    push    si
    push    cx
    push    bx
.CopyRow:
    cmp     bx, MaxWidth
    jae     .NoPixel
    lodsb
    jmp     dx
.CopyCopy:
    stosb
    jmp     .CopyContinue
.CopyXOR:
    xor     [es:di], al
    inc     di
    jmp     .CopyContinue
.CopyOR:
    or      [es:di], al
    inc     di
    jmp     .CopyContinue
.CopyAND:
    and     [es:di], al
    inc     di
    jmp     .CopyContinue
.CopyNOT:
    not     al
    and     [es:di], al
    inc     di
    jmp     .CopyContinue
.CopyContinue:
    inc     bx
    loop    .CopyRow
    jmp     .RowFinished
.NoPixel:
    inc     di
    inc     si
    inc     bx
    loop    .CopyRow
.RowFinished:
    pop     bx
    pop     cx
    pop     si
    pop     di
.SkipRow:
    mov     ax, PixelHeight
    dec     ax
    jz      .DoneCopy
    mov     PixelHeight, ax
    add     si, ByteWidth
    add     di, MaxWidth
    inc     word ScreenRow
    jmp     .CopyImage

.CompressedCopyStart:
    mov     dx, .CompXOR
    cmp     word PARAM_4, 4
    je      .CompFirstByte
    mov     dx, .CompOR
    cmp     word PARAM_4, 3
    je      .CompFirstByte
    mov     dx, .CompNOT
    cmp     word PARAM_4, 2
    je      .CompFirstByte
    mov     dx, .CompAND
    cmp     word PARAM_4, 1
    je      .CompFirstByte
    mov     dx, .CompCopy
.CompFirstByte:
    lodsw
.CompressedCopyImage:
    cmp     word ScreenRow, MaxHeight
    jge     .DoneCopy
    push    di
    push    cx
    push    bx
.CompressedCopyRow:
    cmp     word ScreenRow, 0
    jl      .CompressedNextPixel
    cmp     bx, MaxWidth
    jae     .CompressedNextPixel
    jmp     dx
.CompCopy:
    mov     [es:di], ah
    jmp     .CompressedNextPixel
.CompXOR:
    xor     [es:di], ah
    jmp     .CompressedNextPixel
.CompOR:
    or      [es:di], ah
    jmp     .CompressedNextPixel
.CompAND:
    and     [es:di], ah
    jmp     .CompressedNextPixel
.CompNOT:
    push    ax
    not     ah
    and     [es:di], ah
    pop     ax
    jmp     .CompressedNextPixel
.CompressedNextPixel:
    inc     di
    inc     bx
    dec     al
    jnz      .CompressedContinue
    lodsw
.CompressedContinue:
    loop    .CompressedCopyRow
    jmp     .CompressedRowFinished

.CompressedRowFinished:
    pop     bx
    pop     cx
    pop     di
.CompressedSkipRow:
    push    ax
    mov     ax, PixelHeight
    dec     ax
    mov     PixelHeight, ax
    pop     ax
    jz      .DoneCopy
    add     di, MaxWidth
    inc     word ScreenRow
    jmp     .CompressedCopyImage
.DoneCopy:

    %undef ScreenRow
    %undef ByteWidth
    %undef PixelWidth
    %undef ByteCount
    %undef PixelHeight

.Done:
    popf
    pop     dx
    pop     dx
    pop     cx
    pop     bx
    pop     ax
    pop     ds
edriver

pdriver PutMaskMode, pointer, integer, integer, word
; procedure (Image : PImage; X, Y : integer);
    push    ds
    ImageHeader const, PARAM_1
    ; es:di=image ax=byte width, bx=image width, dx=image height, cx=data size
    push    bx
    push    dx
    push    ax
    push    cx
    push    dx
    push    dx
    push    dx
    pushf

    %define PixelWidth  [ss:bp-6]
    %define PixelHeight [ss:bp-8]
    %define ByteWidth   [ss:bp-10]
    %define ByteCount   [ss:bp-12]
    %define ScreenRow   [ss:bp-14]
    %define ScreenCol   [ss:bp-16]
    %define StartCol    [ss:bp-18]

    mov     dx, PARAM_3
    mov     ScreenRow, dx
    push    dx
    mov     ax, MaxWidth
    mul     dx
    pop     dx

    mov     bx, PARAM_2
    mov     StartCol, bx
    add     ax, bx; ax is stage pixel offset

    push    ax
    mov     ax, dx
    cmp     ax, MaxHeight
    jge     .DoneCopy
    cmp     ax, 0
    jge     .StartOK
    xor     ax, ax
.StartOK:
    push    ax
    mov     ax, dx
    add     ax, PixelHeight
    dec     ax
    cmp     ax, 0
    jl      .NoCopy
    cmp     ax, MaxHeight
    jl      .EndOK
    mov     ax, MaxHeight - 1
    jmp     .EndOK
.NoCopy:
    pop     ax
    jmp     .DoneCopy
.EndOK:
    Update  Line, ax
    pop     ax
    Update  Line, ax
    pop     ax

    push    di
    pop     si
    push    es
    pop     ds

    les     di, [cs:Stage]
    add     di, ax
    mov     cx, PixelWidth
    popf
    cld
    push    ax

    jc      .CompressedCopyStart

    ; Set Uncompressed Mode Jump
    mov     dx, .CopyXOR
    cmp     word PARAM_4, 4
    je      .CopyImage
    mov     dx, .CopyOR
    cmp     word PARAM_4, 3
    je      .CopyImage
    mov     dx, .CopyNOT
    cmp     word PARAM_4, 2
    je      .CopyImage
    mov     dx, .CopyAND
    cmp     word PARAM_4, 1
    je      .CopyImage
    mov     dx, .CopyCopy

.CopyImage:
    cmp     word ScreenRow, MaxHeight
    jge     .DoneCopy
    cmp     word ScreenRow, 0
    jl      .SkipRow
    push    di
    push    si
    push    cx
    mov     ax, StartCol
    mov     ScreenCol, ax
    mov     bh, 8
    mov     bl, [si]
.CopyRow:
    cmp     word ScreenCol, MaxWidth
    jae     .NoPixel
    xor     al, al
    rcl     bl, 1
    jnc     .CopyNoBit
    not     al
.CopyNoBit:
    dec     bh
    jz      .CopyNextByte
    jmp     dx
.CopyNextByte:
    mov     bh, 8
    inc     si
    mov     bl, [si]
    jmp     dx
.CopyCopy:
    stosb
    jmp     .CopyContinue
.CopyXOR:
    xor     [es:di], al
    inc     di
    jmp     .CopyContinue
.CopyOR:
    or      [es:di], al
    inc     di
    jmp     .CopyContinue
.CopyAND:
    and     [es:di], al
    inc     di
    jmp     .CopyContinue
.CopyNOT:
    not     al
    and     [es:di], al
    inc     di
    jmp     .CopyContinue
.CopyContinue:
    inc     word ScreenCol
    loop    .CopyRow
    jmp     .RowFinished
.NoPixel:
    inc     di
    loop    .CopyRow
.RowFinished:
    pop     cx
    pop     si
    pop     di
.SkipRow:
    mov     ax, PixelHeight
    dec     ax
    jz      .DoneCopy
    mov     PixelHeight, ax
    add     si, ByteWidth
    add     di, MaxWidth
    inc     word ScreenRow
    jmp     .CopyImage

.CompressedCopyStart:
    mov     dx, .CompXOR
    cmp     word PARAM_4, 4
    je      .CompFirstByte
    mov     dx, .CompOR
    cmp     word PARAM_4, 3
    je      .CompFirstByte
    mov     dx, .CompNOT
    cmp     word PARAM_4, 2
    je      .CompFirstByte
    mov     dx, .CompAND
    cmp     word PARAM_4, 1
    je      .CompFirstByte
    mov     dx, .CompCopy
.CompFirstByte:
    mov     bh, 8
    mov     ax,[si]
.CompressedCopyImage:
    cmp     word ScreenRow, MaxHeight
    jge     .DoneCopy
    push    di
    push    cx
    push    ax
    mov     ax, StartCol
    mov     ScreenCol, ax
    pop     ax
.CompressedCopyRow:
    test    bh, bh
    jnz     .CompBitOK
    mov     bh, 8
    mov     ah,[si+1]
    dec     al
    jnz     .CompBitOK
    add     si, 2
    mov     ax, [si]
.CompBitOK:
    xor     bl, bl
    rcl     ah, 1
    jnc     .CompNoBit
    not     bl
.CompNoBit:
    cmp     word ScreenRow, 0
    jl      .CompNextPixel
    cmp     word ScreenCol, MaxWidth
    jae     .CompNextPixel

    jmp     dx
.CompCopy:
    mov     [es:di], bl
    jmp     .CompNextPixel
.CompXOR:
    xor     [es:di], bl
    jmp     .CompNextPixel
.CompOR:
    or      [es:di], bl
    jmp     .CompNextPixel
.CompAND:
    and     [es:di], bl
    jmp     .CompNextPixel
.CompNOT:
    push    bx
    not     bl
    and     [es:di], bl
    pop     bx

.CompNextPixel:
    inc     di
    inc     word ScreenCol
    dec     bh
    loop    .CompressedCopyRow

.CompressedRowFinished:
    pop     cx
    pop     di
    push    ax
    mov     ax, PixelHeight
    dec     ax
    mov     PixelHeight, ax
    pop     ax
    jz      .DoneCopy
    add     di, MaxWidth
    inc     word ScreenRow
    xor     bh,bh
    jmp     .CompressedCopyImage
.DoneCopy:

    %undef StartCol
    %undef ScreenCol
    %undef ScreenRow
    %undef ByteWidth
    %undef PixelWidth
    %undef ByteCount
    %undef PixelHeight
    %undef BitWidth

.Done:
    pop     ax
    pop     ax
    pop     ax
    pop     ax
    pop     ax
    pop     ax
    pop     ax
    pop     ax
    pop     ds
edriver

pdriver PutBits, pointer, integer, integer, byte, byte, word
; procedure (Bits : Pointer; x, y, BytesW, BytesH : integer; Color : TColor);
    mov     ax, PARAM_3
    mov     bx, PARAM_5
    add     bx, ax
    cmp     ax, MaxHeight
    jge     .NoDraw
    jb      .StartOK
    xor     ax, ax
.StartOK:
    cmp     bx, MaxHeight
    jge     .AdjustEnd
    ja      .NoDraw
    jmp     .SetUpdate
.AdjustEnd:
    mov     bx, MaxHeight - 1
.SetUpdate:
    Update  Line, bx
    Update  Line, ax
    push    ds
    les     di, [Stage]
    mov     dx, PARAM_3
    mov     ax, MaxWidth
    mul     dx
    add     di, ax
    add     di, PARAM_2
    lds     si, PARAM_1
    mov     al, PARAM_6
    mov     cl, PARAM_4
    xor     ch, ch
.Row:
    cmp     word PARAM_3, MaxHeight
    jge     .Done
    ja      .RowDone
    push    si
    push    di
    push    cx
    mov     dx, PARAM_2
.NextByte:
    mov     bh, 8
    mov     bl, [si]
    inc     si
.ByteLoop:
    cmp     dx, MaxWidth
    jg      .SkipRestLine
    ja      .SkipPixel
.DrawPixel:
    test    bl, 0x80
    jz      .SkipPixel
    mov     [es:di], al
.SkipPixel:
    inc     di
    inc     dx
    dec     bh
    jz      .ByteDone
    shl     bl, 1
    jmp     .ByteLoop
.ByteDone:
    loop    .NextByte
.SkipRestLine:
    pop     cx
    pop     di
    pop     si
.RowDone:
    dec     byte PARAM_5
    jz      .Done
    add     si, cx
    add     di, MaxWidth
    inc     word PARAM_3
    jmp     .Row
.Done:
    pop     ds
.NoDraw:
edriver
