{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$I INFERNO.DEF}
unit FFmtBMP; { Simple BMP (Win3x) Graphic Image File Format  }

interface

uses Inferno;

var
    BMP_Handler : PFormatHandler;

implementation

{ All of this is slow and inefficient, but quick and very easy to write! }

type
    WinBMPFileHeader = record { 14-bytes, Win 2.x + }
        FileType : Word;        { Should be 0x4d42 "BM" }
        FileSize : LongInt;     { DWORD }
        Reserved : LongInt;     { 2 Words, should be 0 }
        ImageOfs : LongInt;     { DWORD, start of image data }
    end;

    WinBMPBitmapHeader = record { 40 bytes, Win 3.x + }
        HeadSize        : LongInt;  { DWORD, size of this header }
	    Width           : LongInt;  { Image Width }
	    Height          : LongInt;  { Image Height }
	    Planes          : word;     { Color Planes, should be 1 }
	    BitsPerPixel    : word;     { 1, 4, 8, or 24. Were always using 8 }
	    Compression     : LongInt;  { DWORD, types of compression }
	    SizeOfBitmap    : LongInt;  { DWORD, byte size of bitmap }
	    XPPM, YPPM      : LongInt;  { Horizontal/Vertical pixels/meter }
    	Colors          : LongInt;  { total number of different colors }
    	ColorsMin       : LongInt;  { number of important colors }
    end;
    WinBMPHeader = record
        FileHeader : WinBMPFileHeader;
        BitMapHeader : WinBMPBitmapHeader
    end;

    WinBMPColor = record
        Blue, Green, Red, Reserved : byte; { Backwards ;-( }
    end;

    WinBMPPalette = array[0..255] of WinBMPColor;
    WinBMPColorUsage = array[0..255] of word;
    WinBMPColorMap = array[0..255] of integer;

procedure SetColorUsage(Image : PImage; var Usage : WinBMPColorUsage );
var
    X, Y : integer;
begin
    FillChar(Usage, Sizeof(Usage), 0);
    for Y := 0 to Image^.Height - 1 do
        for X := 0 to Image^.Width - 1 do
            Inc(Usage[Video^.ImageGetPixel(Image, X,Y)]);
end;

procedure ColorRemap(Image : PImage; var Palette : WinBMPPalette; var Count, Min : integer);
var
    Usage : WinBMPColorUsage;
    Map, Table : WinBMPColorMap;
    RGB : TRGBPalettes;
    I, M, C : integer;
    X, Y : integer;
begin
    Video^.GetPalettes(RGB);
    SetColorUsage(Image, Usage);
    FillChar(Map, Sizeof(Map), 0);
    FillChar(Table, Sizeof(Table), 0);
    FillChar(Palette, Sizeof(Palette), 0);
    Count := 0;
    Min := 0;
    repeat
        M := -1;
        C := 0;
        for I := 0 to 255 do
            if Usage[I] > C then begin
                M := I;
                C := Usage[I];
            end;
        if M > -1 then begin
            if Usage[M] > 16 then Inc(Min);
            Map[Count] := M;
            Usage[M] := 0;
            Inc(Count);
        end;
    until M = -1;
    if Min < 15 then Min := Count;
    for I := 0 to Count - 1 do begin
        Table[Map[I]] := I;
        Palette[I].Red   := RGB[Map[I]].Red;
        Palette[I].Green := RGB[Map[I]].Green;
        Palette[I].Blue  := RGB[Map[I]].Blue;
    end;
    for Y := 0 to Image^.Height - 1 do
        for X := 0 to Image^.Width - 1 do
             Video^.ImagePutPixel(Image,X,Y, Table[Video^.ImageGetPixel(Image,X,Y)]);
end;

function BMP_Save(FileName : String; P : Pointer; Size : LongInt) : integer; far;
var
    FH : WinBMPFileHeader;
    BH : WinBMPBitmapHeader;
    BP : WinBMPPalette;
    CC, CM : integer;
    DIB : PDIBitMap;
    TI : PImage;
    Image : PImage;
begin
    Image := P;
    FileName := FileBase(FileName) + 'BMP';
    ClearError;
    FillChar(FH, Sizeof(FH), 0);
    FillChar(BH, Sizeof(BH), 0);
    TI := Video^.CloneImage(Image);
    if not Assigned(TI) then begin
        SetError(erInsufficient_Memory);
        BMP_Save := GetError;
        exit;
    end;
    ColorRemap(TI, BP, CC, CM);
    Video^.ImageFlip(TI);
    DIB := ImageToDIBitMap(TI);
    Video^.FreeImage(TI);
    if not Assigned(DIB) then begin
        SetError(erInsufficient_Memory);
        BMP_Save := GetError;
        exit;
    end;
    DIBitMapImplode(DIB, icaRunLine);
    if DIB^.Compression = icaCanNotCompress then
        DIBitMapExplode(DIB);

    With BH do begin
        HeadSize        := Sizeof(BH);
	    Width           := DIB^.Width;
	    Height          := DIB^.Height;
	    Planes          := 1;
	    BitsPerPixel    := 8;
	    Compression     := DIB^.Compression;
	    SizeOfBitmap    := DIBitMapSizeData(DIB);
	    XPPM            := 12000;   { just guessing }
	    YPPM            := 12000;  { same  }
    	Colors          := CC;
    	ColorsMin       := CM;
    end;

    with FH do begin
        FileType := $4d42;
        ImageOfs := Sizeof(FH) + Sizeof(BH) + CC * Sizeof(WinBMPColor);
        FileSize := ImageOfs + BH.SizeOfBitmap;
    end;

    if NoError then FileSave(FileName, @FH, Sizeof(FH));
    if NoError then FileAppend(FileName, @BH, Sizeof(BH));
    if NoError then FileAppend(FileName, @BP, CC * Sizeof(WinBMPColor));
    if NoError then FileAppend(FileName, @DIB^.ImageData, BH.SizeOfBitmap);
    FreeDIBitMap(DIB);
    BMP_Save := GetError;
end;

function BMP_Check(FileName : String) : boolean; far;
var
    H : ^WinBMPHeader;
    F : boolean;
begin
    F := False;
    ClearError;
    H := FileRead(FileName, 0, Sizeof(WinBMPHeader));
    if Assigned(H) then begin
        F := (H^.FileHeader.FileType = $4d42) and
             (H^.FileHeader.FileSize = FileSizeOf(FileName)) and
             (H^.FileHeader.Reserved = 0) and

             (H^.BitMapHeader.HeadSize = 40) and
             (H^.BitMapHeader.Planes   = 1) and
             (H^.BitMapHeader.BitsPerPixel = 8) and
             ((H^.BitMapHeader.Compression = 0) or (H^.BitMapHeader.Compression = 1))
             and

             (H^.FileHeader.ImageOfs = Sizeof(WinBMPHeader) +
                Sizeof(WinBMPColor) * H^.BitMapHeader.Colors);

        FreeMem(H, Sizeof(WinBMPHeader));
    end;
    BMP_Check := NoError and F;
    ClearError;
end;

function BMP_Process(var P : Pointer; var Size : word) : integer; far;
var
    DP : Pointer;
    Sz : LongInt;
    Image : PImage;
    DIB : PDIBitMap;
    WP, SP : TRGBPalettes;
    I  : integer;
begin
    ClearError;
    Image := nil;
    DIB := nil;
    DP := P;
    with WinBMPHeader(P^) do begin
        Sz := Video^.ImageSize(BitMapHeader.Width, BitMapHeader.Height);
        if not ((FileHeader.FileType = $4d42) and
             (FileHeader.FileSize = Size) and
             (FileHeader.Reserved = 0) and

             (BitMapHeader.HeadSize = 40) and
             (BitMapHeader.Planes   = 1) and
             (BitMapHeader.BitsPerPixel = 8) and
             ((BitMapHeader.Compression = 0) or (BitMapHeader.Compression = 1))
             and

             (FileHeader.ImageOfs = Sizeof(WinBMPHeader) +
                Sizeof(WinBMPColor) * BitMapHeader.Colors)) then Sz := 0;

        if (Sz < $f000) and (Sz > 1) then begin
            DIB := NewDIBitMapFixedSize(BitMapHeader.Width, BitMapHeader.Height,
                BitMapHeader.SizeOfBitmap, BitMapHeader.Compression);
            { Image := Video^.NewImage(BitMapHeader.Width, BitMapHeader.Height); }
        end;
        if Assigned(DIB) then begin
            DP := P;
            IncPtr(DP,Sizeof(WinBMPHeader));
            FillChar(WP, Sizeof(WP), 0);
            for I := 0 to BitMapHeader.Colors - 1 do begin
                WP[I].Red := WinBMPPalette(DP^)[I].Red;
                WP[I].Green := WinBMPPalette(DP^)[I].Green;
                WP[I].Blue := WinBMPPalette(DP^)[I].Blue;
            end;
            DP := P;
            IncPtr(DP, FileHeader.ImageOfs);
            Move(PBytes(DP)^, DIB^.ImageData, DIB^.DataSize);
            DIBitMapExplode(DIB);
            Image := DIBitMapToImage(DIB);
            FreeDIBitMap(DIB);
            if Assigned(Image) then begin
                Video^.ImageFlip(Image);
                Video^.GetPalettes(SP);
                Video^.ImageRemap(Image, SP, WP, FormatPaletteMode);
                if (FormatPaletteMode = ipmComplete) or (FormatPaletteMode = ipmOverride) then
                    Video^.SetPalettes(WP);
                if ImageCompress then
                    Video^.ImageImplode(Image);
            end;
        end;
    end;

    if Assigned(P) then begin
        FreeMem(P, Size);
        P := nil;
    end;
    P := Image;
    if Assigned(Image) then
        Size := Video^.ImageSizeOf(Image)
    else
        Size := 0;
    BMP_Process := GetError;
end;

function BMP_Load(FileName : String; var P : Pointer; var Size : LongInt) : integer; far;
var
    FS : Word;
begin
    if FileLoad(FileName, P, FS) then begin
        if BMP_Process(P, FS) = 0 then
            Size := FS;
    end;
    if IsError then
        Size := 0;
    BMP_Load := GetError;
end;


begin
    BMP_Handler := New(PFormatHandler);
    with BMP_Handler^ do begin
        Kind   := ffImage;
        UID    := 'IMGBMP';
        Compat := 0;
        Exts   := 'BMP';
        Check  := BMP_Check;
        Process:= BMP_Process;
        Save   := BMP_Save;
        Load   := BMP_Load;
    end;
    RegisterFileFormat(BMP_Handler);
end.