{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$IFDEF SECTINT}

type
    TFormatCheck  = function (FileName : String) : boolean;
    TFormatSave   = function (FileName : String; P : Pointer; Size : LongInt) : integer;
    TFormatLoad   = function (FileName : String; var P : Pointer; var Size : LongInt) : integer;
    TFormatProcess= function (var P : Pointer; var Size : word) : integer;

    PFormatHandler = ^TFormatHandler;
    TFormatHandler = record
        Kind   : word;
        UID    : Str8;
        Compat : word;
        Exts   : string;
        Check  : TFormatCheck;
        Process: TFormatProcess;
        Save   : TFormatSave;
        Load   : TFormatLoad;
    end;

var
    FormatPaletteMode : word;

{ Other than needing to save Images and the like to disk. Just let AssetLoad
deal with loading and converting files into usable formats. }
function FormatSuggest ( FileName : String ) : PFormatHandler;
function FormatSupports( Kind : Word ) : String;
function FormatIndex(Kind, Index : word) : PFormatHandler;

function FormatHandler(FileName : String; Verify : boolean) : PFormatHandler;
function FormatType(FileName : String) : integer;

{ For most format/file types, Size provided to the FormatSave function does
not matter. It figures it out based on then Data type supported by that
type of file. In other words, Saving when an image as a bitmap, you could
say 1 byte. The format handler will look at the image and save the correct
number of bytes. I don't foresee any output formats supported by FormatSave
ever actually needing the Size field. I'll probably either remove it. Or,
have it return the Size of written data. }
function FormatSave(FileName : String; P : Pointer; Size : LongInt) : boolean;
function FormatLoad(FileName : String; var P : Pointer; var Size : LongInt) : boolean;

procedure RegisterFileFormat(Handler : PFormatHandler);
procedure UnregisterFileFormat(Handler : PFormatHandler);

{$ENDIF}

{$IFDEF SECTIMP}
var
    Formats : PList;

function FormatSuggest ( FileName : String ) : PFormatHandler;
var
    P : PListItem;
    E : String;
begin
    P := Formats^.First;
    E := ';' + UCase(FileExt(FileName)) + ';';
    while Assigned(P) do begin
        if Pos(E, ';' + PFormatHandler(P^.Data)^.Exts + ';') > 0 then break;
        P := P^.Next;
    end;
    if Assigned(P) then
        FormatSuggest := PFormatHandler(P^.Data)
    else
        FormatSuggest := nil;
end;

function FormatSupports(Kind : word) : String;
var
    P : PListItem;
    S : String;
begin
    S := '';
    P := Formats^.First;
    while Assigned(P) do begin
        if PFormatHandler(P^.Data)^.Kind = Kind then begin
            if S <> '' then S := S + ';';
            S := S + PFormatHandler(P^.Data)^.UID;
        end;
        P := P^.Next;
    end;
    FormatSupports := S;
end;

function FormatIndex(Kind, Index : word) : PFormatHandler;
var
    P : PListItem;
begin
    P := Formats^.First;
    while Assigned(P) do begin
        if PFormatHandler(P^.Data)^.Kind = Kind then begin
            if Index = 0 then Break;
            Dec(Index);
        end;
        P := P^.Next;
    end;
    if Assigned(P) then
        FormatIndex := PFormatHandler(P^.Data)
    else
        FormatIndex := nil;
end;

function FormatSave(FileName : String; P : Pointer; Size : LongInt) : boolean;
var
    Handler : PFormatHandler;
begin
    FormatSave := false;
    Handler := FormatSuggest(FileName);
    if not Assigned(Handler) then begin
        {$IFDEF LOGS} Log('Format Save: ' + FileName + ' (NONE)'); {$ENDIF}
        SetError(erInvalid_File_Format);
        exit;
    end;
    if not Assigned(Handler^.Save) then begin
        {$IFDEF LOGS} Log('Format Save: ' + FileName + ' (NULL)'); {$ENDIF}
        SetError(erOperation_Not_Supported);
        exit;
    end;
    {$IFDEF LOGS} Log('Format Save: ' + FileName + ' (' + Handler^.UID + ')'); {$ENDIF}
    SetError(Handler^.Save(FileName, P, Size));
    FormatSave := NoError;
end;

function FormatHandler(FileName : String; Verify : boolean) : PFormatHandler;
var
    P : PListItem;
    E : String;
begin
    if UCase(FileExt(FileName)) = '' then begin
        FormatHandler := nil;
        exit;
    end;
    P := Formats^.First;
    E := ';' + UCase(FileExt(FileName)) + ';';
    while Assigned(P) do begin
        if (Pos(E, ';' + PFormatHandler(P^.Data)^.Exts + ';') > 0) then begin
            if not Verify then Break;
            if Assigned(PFormatHandler(P^.Data)^.Check) and (PFormatHandler(P^.Data)^.Check(FileName)) then
                Break;
        end;
        P := P^.Next;
    end;
    if Assigned(P) then begin
        FormatHandler := PFormatHandler(P^.Data);
        {$IFDEF LOGS} Log('Format Handler: ' + FileName + ' (' +
        PFormatHandler(P^.Data)^.UID + ')'); {$ENDIF}
    end else begin
        (*
            {$IFDEF LOGS} Log('Format Handler: ' + FileName + ' (NULL)'); {$ENDIF}
        *)
        FormatHandler := nil;
    end;
end;

function FormatType(FileName : String) : integer;
var
    H : PFormatHandler;
begin
    H := FormatHandler(FileName, False);
    if Assigned(H) then begin
        FormatType := H^.Kind;
    end else begin
        FormatType := ffUnknown;
        if (FileExt(UCase(FileName)) = 'TXT')  or (UCase(FileName) = 'LICENSE') then
            FormatType := ffText;
        if (FileExt(UCase(FileName)) <> 'EN') and (FileExt(UCase(FileName)) <> Language) then
            exit;
        if (FileRoot(UCase(FileName)) = FileRoot(Ucase(ExeName))) and (FilePath(FileName) = '') then
            FormatType := ffNLS
        else
            FormatType := ffNLSText;
    end;
end;

function FormatLoad(FileName : String; var P : Pointer; var Size : LongInt) : boolean;
var
    Handler : PFormatHandler;
begin
    FormatLoad := false;
    Handler := FormatHandler(FileName, true);
    if not Assigned(Handler) then begin
        {$IFDEF LOGS} Log('Format Load: ' + FileName + ' (NONE)'); {$ENDIF}
        SetError(erInvalid_File_Format);
        Exit;
    end;
    if not Assigned(Handler^.Load) then begin
        {$IFDEF LOGS} Log('Format Load: ' + FileName + ' (NULL)'); {$ENDIF}
        SetError(erOperation_Not_Supported);
        Exit;
    end;
    {$IFDEF LOGS} Log('Format Load: ' + FileName + ' (' + Handler^.UID + ')'); {$ENDIF}
    SetError(Handler^.Load(FileName, P, Size));
    FormatLoad := NoError;
end;

procedure FormatAsset (var Asset : PAsset);
var
    H : PFormatHandler;
    C : integer;
begin
    H := FormatHandler(PtrStr(Asset^.Name), False);
    if Assigned(H) and Assigned(H^.Process) then begin
        {$IFDEF LOGS} Log('Format Asset: ' + PtrStr(Asset^.Name) + ' (' + H^.UID +
        ',' + IntStr(Asset^.MemSize) + ')'); {$ENDIF}
        C := H^.Process(Asset^.Data, Asset^.MemSize);
        {$IFDEF LOGS} Log('Format Done: ' + PtrStr(Asset^.Name) + '(' +
         IntStr(Asset^.MemSize) + '), Code ' + IntStr(C)); {$ENDIF}
    end else begin
        {$IFDEF LOGS} Log('Format Asset: ' + PtrStr(Asset^.Name) + ' (N/A)'); {$ENDIF}
    end;
end;

procedure RegisterFileFormat(Handler : PFormatHandler);
begin
    if not Assigned(Handler) then Exit;
    if InList(Formats, Handler) <> nil then Exit;
    if AddList(Formats, Handler) = nil then
        FatalError(erInsufficient_Memory, 'registering ' + Handler^.UID + ' format ');
end;

procedure UnregisterFileFormat(Handler : PFormatHandler);
var
    P : PListItem;
begin
    if not Assigned(Handler) then Exit;
    RemoveList(Formats, Handler);
end;

{$ENDIF}
