{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$I INFERNO.DEF}
unit Inferno;

{$DEFINE SECTINT}
interface

{$I CONSTS.INC}
{$I TYPES.INC}
{$I BASICS.INC}
{$I EVENTS.INC}
{$I FFORMAT.INC}
{$I DIBITMAP.INC}
{$I DRIVERS.INC}
{$I KEYBOARD.INC}
{$I MOUSE.INC}
{$I JOYSTICK.INC}
{$I COMMANDS.INC}
{$I VIDTOOLS.INC}
{$I ASSETS.INC}

var
    AppInfo      : TAppInfo;
    ExeName      : String;

    Language     : Str20;

    ExitMessage  : String;
    ErrorMessage : String;

    CPUIdle      : word;
    IsDOSBox     : boolean;
    IsVirtual    : boolean;
    MachineType  : byte;
    NLSResult    : byte; { 0 was not found, 1 user lang, 2 default lang }

    TimerTick    : LongInt absolute $0040:$006c; { Timer tick counter }

    Video        : PVideoDriver;
    MousePtr     : PSprite;
    MousePresent : boolean;
    ImageCompress: boolean;
    VRTimer      : PWord;

procedure Ignite;
procedure Extinguish;

implementation
{$UNDEF SECTINT}
{$DEFINE SECTIMP}

uses VideoSub, AudioSub, FFBiltIn;

{$I CONSTS.INC}
{$I TYPES.INC}

var
    OldExitProc          : pointer;
    OldHeapError         : pointer;
    OldKeyboardProc      : pointer;
    OldMouseProc         : pointer;
    OldMouseMask         : word;

    MoveEvent            : TEvent;
{$IFDEF MSGTEXT}
    GoodbyeMessage       : boolean;
{$ENDIF}

{$I VIDEONUL.INC}

{$I BASICS.INC}

procedure Ignite;
var
    P : PAsset;
    VideoModes : PVideoModes;
    E : integer;
    S : String;
begin
    {$IFDEF ASSETSRCDIRS}
        { Disable debug option to die on any and every error }
        DieOnError := False;
        { Also, add dev paths drivers fonts and files }
        AssetIndexSys(FilePath(ExeName) + 'CORE\', '');
        if AsPath(AppInfo.SubDir.Drivers) <> 'DRIVERS\' then
        AssetIndexSys(FilePath(ExeName) + 'DRIVERS\', AppInfo.SubDir.Drivers);
        if AsPath(AppInfo.SubDir.Fonts) <> 'FONTS\' then
            AssetIndexSys(FilePath(ExeName) + 'FONTS', AppInfo.SubDir.Fonts);

        AssetIndexSys(
            FilePath(ExeName) + ReplaceStr(FileRoot(ExeName), '.', ''), '');
        DieOnError := True;
    {$ENDIF}

    {$IFDEF LOGS} Log('Ignite'); {$ENDIF}
    InitNLS;
    {$IFDEF MSGTEXT}
        if AssetLoad('STARTUP.' + Language, asOnlyAssetFile, P) then begin
            PrintText( P );
            Delay(AppInfo.Delay.Startup);
        end else if AssetLoad('STARTUP.EN', asOnlyAssetFile, P) then begin
            PrintText( P );
            Delay(AppInfo.Delay.Startup);
        end;
    {$ELSE}
        S := 'Debug build, welcome text omitted';
        {$IFDEF LOGS} Log(S); {$ENDIF}
        WriteLn(S);
        Delay(1000);
    {$ENDIF}

    if not AssetLoad(asPath(AppInfo.SubDir.Drivers) + AppInfo.Driver.Video,
    asDefault, P) then begin
        if GetError <> 0 then
            FatalError(GetError, FormatStr(NLSData.Error.DriverInit, AppInfo.Driver.Video))
        else
            FatalError(erFile_Not_Found, FormatStr(NLSData.Error.DriverFind, AppInfo.Driver.Video));
    end;
    Video := P^.Data;
    VideoModes := Video^.GetModes;
    if AppInfo.VideoMode <> 0 then begin
        E := Video^.Open(AppInfo.VideoMode);
        if E <> 0 then
            E := Video^.Open(VideoModes^[0].Mode);
    end else
        E := Video^.Open(VideoModes^[0].Mode);
    if E <> 0 then
        FatalError(E, FormatStr(NLSData.Error.DriverMode, '0x' + WordHex(VideoModes^[0].Mode)));
    S := AppInfo.FontName;
    FontBestMatch(S, AppInfo.Font);
    if not Assigned(AppInfo.Font) then begin
        if GetError <> 0 then
            FatalError(GetError, FormatStr(NLSData.Error.FontInit, S))
        else
            FatalError(erFile_Not_Found,  FormatStr(NLSData.Error.FontFind, S));
    end;
    Video^.SetFont(AppInfo.Font);
    AppInfo.FontName := S;
    Video^.Fill(0);
    Video^.Update;
    MouseDefault(False);
    {$IFOPT D+}
    Delay(0);
    {$ELSE}
    Delay(AppInfo.Delay.Init);
    {$ENDIF}

    {$IFDEF LOGS} Log('Ignited'); {$ENDIF}
end;

procedure Extinguish;
begin
    {$IFDEF LOGS} Log('Extinguish'); {$ENDIF}
    if Assigned(Video) then Video^.Close;
    PrintVersion;
    {$IFOPT D+}
    {$ELSE}
        {$IFDEF MSGTEXT}
            if GoodbyeMessage then
                Delay(AppInfo.Delay.Message)
            else
                Delay(AppInfo.Delay.Shutdown);
        {$ELSE}
            Delay(AppInfo.Delay.Shutdown);
        {$ENDIF}
    {$ENDIF}
    Terminate(0);
end;

{$L INFERNO.OBJ}

{$I-}
procedure DoneInferno;  far; external;
function InitInferno : boolean; far; external;

{$I EVENTS.INC}
{$I FFORMAT.INC}
{$I DIBITMAP.INC}
{$I DRIVERS.INC}
{$I KEYBOARD.INC}
{$I MOUSE.INC}
{$I JOYSTICK.INC}
{$I COMMANDS.INC}
{$I VIDTOOLS.INC}
{$I ASSETS.INC}

function OutOfMemory(Size : word) : integer; far;
var
    Action : word;
    PreMax : LongInt;
begin
    Action := 0; { Die }
    if Size <> 0 then begin
        PreMax := MaxAvail;
        {$IFDEF LOGS}
            Log('Out of Memory: ' + IntStr(Size) + ' Requested');
        {$ENDIF}
        AssetDisposeAll;
        if (MaxAvail < Size) or (PreMax = MaxAvail) then
            Action := 0
        else
            Action := 2;
        {$IFDEF LOGS}
            Log('Memory: ' + IntStr(MaxAvail - PreMax) + ' freed, code ' + IntStr(Action));
        {$ENDIF}

        if Action <> 2 then SetError(erInsufficient_memory);
    end;
    OutOfMemory := Action; { return 0=die, 1=return nil, 2=try again }
end;

procedure Finalization; far;
begin
    ExitProc := OldExitProc;
    HeapError := OldHeapError;
    {$IFDEF LOGS}
        Log('Shutdown: Start');
    {$ENDIF}
    SetIntVec($09, OldKeyboardProc);
    if OldMouseProc <> @MouseHandler then
        SwapMouseHandler(OldMouseMask, OldMouseProc);
    if ExitCode <> 0 then begin
        {$IFDEF LOGS}
            Log('Shutdown: Error code ' + IntStr(ExitCode));
        {$ENDIF}
        if ErrorMessage = '' then
            ErrorMessage := RawNLS('ERROR.' + IntStr(ExitCode));
        WriteLn;
        Write('Fatal Error #', ExitCode);
        if ErrorMessage <> '' then
            WriteLn(': ', ErrorMessage)
    end;
    if ExitMessage <> '' then begin
        WriteLn;
        WriteLn(ExitMessage);
    end;
    {$IFDEF LOGS}
        Log('Shutdown: Assets');
    {$ENDIF}
    DoneAssets;
    {$IFDEF LOGS}
        Log('Shutdown: Inferno');
    {$ENDIF}
    DoneInferno;
    {$IFDEF LOGS}
        Log('Shutdown: Messages');
    {$ENDIF}
    ErrorAddr := nil;
    {$IFDEF LOGS}
        Log('Shutdown: Complete');
    {$ENDIF}
end;

{ Initialization }
procedure InitVariables;
begin
    {$IFDEF MSGTEXT}
        GoodbyeMessage := false;
    {$ENDIF}
    SetVRTInterval(1);
    VRTimer := GetVRTimer;
    Video := nil;
    VideoNULL := nil;
    Language := Ucase(Trim(GetEnv('LANG')));
    if Language = '' then Language := 'EN';
    With AppInfo do begin
        Version         := 'ALPHA';
        Year            := '2021';
        Title           := ReplaceStr(FileRoot(ParamStr(0)), '.', #32) +
                          '& Inferno Game Engine';
        Author          := 'Jerome Shidel';
        FontName        := '1012N-' + Language + '.FNT';
        Font            := nil;
        Driver.Video    := 'VGA386.DRV';
        Driver.Audio    := 'NOAUDIO.DRV';
        SubDir.Fonts    := ''; { 'FONTS'; }
        SubDir.Drivers  := ''; { 'DRIVERS'; }
        Delay.Startup   := 5000;
        Delay.Init      := 0;
        Delay.Shutdown  := 0;
        Delay.Message   := 2000;
        VideoMode       := 0;    { use default best/recommended mode }
    end;
    ExitMessage := '';
    ErrorMessage := '';
    ImageCompress := True;
    GetIntVec($09, OldKeyboardProc);
    IntState := 0;
    ShiftState.L := 0;
    ShiftState.H := 0;
    HoldCode := 0;
    ScanCode := 0;
    ExtraCode := 0;
    MouseLag := 0;
    CPUIdle := 0;
    InitSprites;
    MousePtr := nil;
    FormatPaletteMode := ipmMatch;
    MoveEvent.Kind := evNull;
    Events := NewList;
    Formats := NewList;
    AssetFiles   := nil;
    Assets       := nil;
    MaxOpenFiles := 2;
    CurOpenFiles := 0;
    MousePresent := False;
    OldMouseProc := @MouseHandler;
    OldMouseMask := $ffff;
    OldExitProc  := ExitProc;
    OldHeapError := HeapError;
    if IsDOSBox then CPUIdle := 1;
end;

procedure ConfigureBuiltInNull;
var
    BuiltIn : PNullDriver;
begin
    BuiltIn := New(PNullDriver);
    if not Assigned(BuiltIn) then
        FatalError(erInsufficient_memory, 'initializing NULL video driver');
    FillChar(BuiltIn^, SizeOf(TNullDriver), 0);
    with BuiltIn^ do begin
        { Termination marker }
        EndOfProcs.L    := $ffff;
        EndOfProcs.H    := $ffff;
    end;
    BuiltInDrivers[0] := PDriver(BuiltIn);
end;

procedure SetBuiltInDrivers;
begin
    FillChar(BuiltInDrivers, Sizeof(BuiltInDrivers), 0);
    ConfigureBuiltInNull;
    ConfigureBuiltInVideoNull;
    Video := VideoNULL;
    BuiltInDrivers[1] := GetBuiltInVideoFunctions;
end;

begin
    ExeName := ParamStr(0);
    {$IFDEF LOGS}
        { Only when compiled with Logging := True; at this point usually
          it is false. Unless, I realize something deep in the core is broken. }
        if Logging then LogReset;
    {$ENDIF}
    if not InitInferno then
        FatalError(GetError, 'initializing inferno core system');
    InitVariables;
    SetBuiltInDrivers;
    RegisterFormatBuiltIn;
    InitAssets;
    InitNLS;
    ExitProc := @Finalization;
    HeapError := @OutOfMemory;
    MousePresent := ResetMouse <> 0;
    if MousePresent then
        SwapMouseHandler(OldMouseMask, OldMouseProc);
    SetIntVec($09, @KeyboardHandler);
end.
{$UNDEF SECTIMP}
