{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$I INFERNO.DEF}
unit VideoSub; { Video Graphics Sub-system and Built-In functions }

{$DEFINE SECTINT}
interface

uses Inferno;

procedure InitSprites;
function GetBuiltInVideoFunctions : PDriver;

implementation
{$UNDEF SECTINT}
{$DEFINE SECTIMP}

const
    ImageHeaderSize = Sizeof(TImage) - Sizeof(TImageDataBytes);
    MaskHeaderSize = Sizeof(TMask) - Sizeof(TImageDataBytes);

procedure ImageDRE(var Img : PImage); { Simplified RLE scheme }
var
   B : PImage;
   RC : byte;
   CC : byte;
   SI, DI : word;
   MI : LongInt;
   {$IFDEF LOGS}
   S : String;
   {$ENDIF}
begin
    {$IFDEF SAFETY}
    if not Assigned(Img) then exit;
    {$ENDIF}
    if Img^.Compression <> icaUncompressed then exit;
    B := Video^.NewImage(Img^.Width, Img^.Height);
    if not Assigned(B) then Exit;
    B^.ByteWidth := Img^.ByteWidth;
    SI := 0;
    DI := 0;
    MI := Img^.DataSize;
    MI := MI * 9 div 10 - 10;
    RC := 1;
    CC := Img^.ImageData[SI];
    SI := 1;
    while (SI < Img^.DataSize) do begin
        if (CC <> Img^.ImageData[SI]) or (RC = 254) then begin
            B^.ImageData[DI] := RC;
            Inc(DI);
            B^.ImageData[DI] := CC;
            Inc(DI);
            RC := 1;
            CC := Img^.ImageData[SI];
            if DI >= MI then break;
        end else
            Inc(RC);
        Inc(SI);
    end;
    B^.ImageData[DI] := RC;
    Inc(DI);
    B^.ImageData[DI] := CC;
    Inc(DI);
    B^.ImageData[DI] := 0; { terminate Data }
    Inc(DI);
    if DI >= MI then begin
        {$IFDEF LOGS}
            Log('Image ' + PtrHex(Img) + ' should not be compressed, threshold of ' +
                IntStr(MI) +  ' bytes reached' );
        {$ENDIF}
        Img^.Compression := icaCanNotCompress;
    end else begin
        {$IFDEF LOGS}
            S := 'Image ' + PtrHex(Img) + ' ' + IntStr(Img^.DataSize) +
              ' data, compressed';
        {$ENDIF}
        Video^.FreeImage(Img);
        GetMem(Img, DI + ImageHeaderSize);
        if assigned(Img) then begin
            Move(B^, Img^,  DI + ImageHeaderSize);
            Img^.Compression := icaDataRun;
            Img^.DataSize := DI;
        end;
        {$IFDEF LOGS}
            Log(S + ' to ' + PtrHex(Img) + ' ' + IntStr(Img^.DataSize) + ' bytes');
        {$ENDIF}
    end;
    Video^.FreeImage(B);
end;

procedure ImageUnDRE(var Img : PImage); { Simplified RLE Decompression }
var
   B : PImage;
   RC : byte;
   CC : byte;
   SI, DI : word;
begin
    {$IFDEF SAFETY}
    If not Assigned(Img) then Exit;
    {$ENDIF}
    if (Img^.Compression = icaCanNotCompress) then
        Img^.Compression := icaUncompressed;
    if (Img^.Compression = icaUncompressed) then exit;
    B := Video^.NewImage(Img^.Width, Img^.Height);
    if not Assigned(B) then Exit;
    B^.ByteWidth := Img^.ByteWidth;
    SI := 0;
    DI := 0;
    FillChar(B^.ImageData, B^.DataSize, 0);
    while (SI < Img^.DataSize) do begin
        RC := Img^.ImageData[SI];
        if RC = 0 then Break;
        Inc(SI);
        CC := Img^.ImageData[SI];
        Inc(SI);
        if DI + RC > B^.DataSize then RC := B^.DataSize - DI;
        FillChar(B^.ImageData[DI], RC, CC);
        Inc(DI, RC);
        if DI >= B^.DataSize then break;
    end;
    {$IFDEF LOGS}
        Log('Image ' + PtrHex(Img) + ' uncompressed to ' + PtrHex(B));
    {$ENDIF}
    Video^.FreeImage(Img);
    Img := B;
end;

{ Performance }
procedure BuiltIn_Fill (Color : TColor); far;
begin
    Video^.Region(0, 0, Video^.Width - 1, Video^.Height - 1, Color);
end;

procedure BuiltIn_Region (x1, y1, x2, y2 : integer; Color : TColor); far;
var
    X, Y : integer;
begin
    for Y := y1 to y2 do
        for X := x1 to x2 do
            Video^.PutPixel(X, Y, Color);
end;

procedure BuiltIn_ImageFill (var Image : PImage; Color : TColor); far;
begin
    Video^.ImageExplode(Image);
    Video^.ImageRegion(Image, 0, 0, Image^.Width - 1, Image^.Height - 1, Color);
end;

procedure BuiltIn_ImageRegion (var Image : PImage; x1, y1, x2, y2 : integer; Color : TColor); far;
var
    X, Y : integer;
begin
    Video^.ImageExplode(Image);
    for Y := y1 to y2 do
        for X := x1 to x2 do
            Video^.ImagePutPixel(Image, X, Y, Color);
end;

procedure BuiltIn_GetImage (var Image : PImage; X, Y : integer); far;
var
    IX, IY : integer;
begin
    {$IFDEF SAFETY}
    if not Assigned(Image) then exit;
    {$ENDIF}
    if (Image^.Compression <> 0) then begin
        FatalError(erOperation_Not_Supported, 'Get Image ' + PtrHex(Image) + ' compressed');
        Exit;
    end;

    for IY := 0 to Image^.Height - 1 do
        for IX := 0 to Image^.Width - 1 do
            Video^.ImagePutPixel(Image, IX, IY,
                Video^.GetPixel(X + IX, Y + IY));
end;

procedure BuiltIn_PutImage (Image : PImage; X, Y : integer); far;
var
    IX, IY : integer;
    II : PImage;
begin
    if not Assigned(Image) then exit;
    if (Image^.Compression <> 0) then begin
        {$IFDEF LOGS}
            Log('Put Image ' + PtrHex (Image) + ' at ' + IntStr(X) + 'x' +
                IntStr(Y) + ' compressed');
        {$ENDIF}
        II := Video^.CloneImage(Image);
        if Assigned(II) then begin
            if Video^.ImageExplode(II) then
                Video^.PutImage(II, X, Y);
            Video^.FreeImage(II);
        end;
    end else begin
        for IY := 0 to Image^.Height - 1 do
            for IX := 0 to Image^.Width - 1 do
                Video^.PutPixel(X + IX, Y + IY,
                    Video^.ImageGetPixel(Image, IX, IY));
    end;
end;

procedure BuiltIn_PutImageMode (Image : PImage; X, Y : integer; Mode : word); far;
var
    I, J, C, N : word;
    II : PImage;
begin
    if not Assigned(Image) then exit;
    if (Image^.Compression <> 0) then begin
        {$IFDEF LOGS}
            Log('Put Image Mode ' + PtrHex(Image) + ' at ' + IntStr(X) + 'x' +
                IntStr(Y) + ' compressed');
        {$ENDIF}
        II := Video^.CloneImage(Image);
        if Assigned(II) then begin
            if Video^.ImageExplode(II) then
                Video^.PutImageMode(II, X, Y, Mode);
            Video^.FreeImage(II);
        end;
    end else begin
        for J := 0 to Image^.Height - 1 do
            for I := 0 to Image^.Width - 1 do begin
                C := Video^.GetPixel(I + X, J + Y);
                N := Video^.ImageGetPixel(Image, I, J);
                case Mode of
                    imCOPY :  C := N;
                    imAND  :  C := C and N;
                    imNOT  :  C := C and (not N);
                    imOR   :  C := C or N;
                    imXOR  :  C := C xor N;
                end;
                Video^.PutPixel(I + X, J + Y, C);
            end;
    end;
end;

procedure BuiltIn_PutMask (Mask : PMask; X, Y : integer); far;
begin
    Video^.PutMaskMode(Mask, X, Y, imAND);
end;

function MaskGetPixel(Mask : PMask; X, Y : integer) : TColor;
begin
    { yes, I know I don't need these parens. It just looks nicer :-) }
    Y := (Y * Mask^.ByteWidth) + (X shr 3);
    X := 7 - (X and 7);
    if Mask^.ImageData[Y] and (1 shl X) <> 0 then
        MaskGetPixel := Video^.Colors - 1
    else
        MaskGetPixel := 0;
end;

procedure MaskSetPixel(Mask : PMask; X, Y : integer; On : boolean);
begin
    { Don't need these parens either! }
    Y := (Y * Mask^.ByteWidth) + (X shr 3);
    X := 7 - (X and 7);
    if On then
        Mask^.ImageData[Y] := Mask^.ImageData[Y] or (1 shl X)
    else
        Mask^.ImageData[Y] := Mask^.ImageData[Y] and ($ff xor (1 shl X));
end;

procedure BuiltIn_PutMaskMode (Mask : PMask; X, Y : integer; Mode : word); far;
var
    I, J, C, N : word;
    MM : PMask;
begin
    { Should add compressed mask support }
    if not Assigned(Mask) then exit;
    if (Mask^.Compression <> 0) then begin
        {$IFDEF LOGS}
            Log('Put Mask ' + PtrHex (Mask) + ' at ' + IntStr(X) + 'x' +
                IntStr(Y) + ' compressed');
        {$ENDIF}
        MM := Video^.CloneMask(Mask);
        if Assigned(MM) then begin
            if Video^.MaskExplode(MM) then
                Video^.PutMaskMode(MM, X, Y, Mode);
            Video^.FreeMask(MM);
        end;
    end else begin
        for J := 0 to Mask^.Height - 1 do
            for I := 0 to Mask^.Width - 1 do begin
                C := Video^.GetPixel(I + X, J + Y);
                N := MaskGetPixel(Mask, I, J);
                case Mode of
                    imCOPY :  C := N;
                    imAND  :  C := C and N;
                    imNOT  :  C := C and (not N);
                    imOR   :  C := C or N;
                    imXOR  :  C := C xor N;
                end;
                Video^.PutPixel(I + X, J + Y, C);
            end;
    end;
end;

function BuiltIn_ImageToMask (Image : PImage; Color : TColor) : PMask; far;
var
    P : PMask;
    I, J : integer;
begin
    P := nil;
    Video^.ImageExplode(Image);
    if Not Assigned(Image) then exit;
    P := Video^.NewMask(Image^.Width, Image^.Height);
    if Assigned(P) then begin
        FillChar(P^.ImageData, Video^.MaskSizeData(P), 0);
        for J := 0 to P^.Height - 1 do
            for I := 0 to P^.Width - 1 do
                if Video^.ImageGetPixel(Image, I, J) = Color then
                    MaskSetPixel(P, I, J, True);
    end;
    {$IFDEF LOGS}
        Log('Image ' + PtrHex(Image) + ' to Mask ' + PtrHex(P));
    {$ENDIF}
    BuiltIn_ImageToMask := P;
end;

function BuiltIn_MaskToImage (Mask : PMask; Color : TColor) : PImage; far;
var
    P : PImage;
    I, J : integer;
begin
    { Should add compressed mask support }
    P := nil;
    if Not Assigned(Mask) then exit;
    P := Video^.NewImage(Mask^.Width, Mask^.Height);
    if Assigned(P) then begin
        if Color = 0 then
            Video^.ImageFill(P, 255)
        else
            Video^.ImageFill(P, 0);
        for J := 0 to P^.Height - 1 do
            for I := 0 to P^.Width - 1 do
                if MaskGetPixel(Mask, I, J) <> 0 then
                    Video^.ImagePutPixel(P, I, J, Color);
    end;
    {$IFDEF LOGS}
        Log('Mask ' + PtrHex(Mask) + ' to Image ' + PtrHex(P));
    {$ENDIF}
    BuiltIn_MaskToImage := P;
end;

procedure BuiltIn_MaskInvert (var Mask : PMask); far;
var
    I, J : integer;
begin
    if Not Assigned(Mask) then exit;
    for J := 0 to Mask^.Height - 1 do
        for I := 0 to Mask^.Width - 1 do
            MaskSetPixel(Mask, I, J, MaskGetPixel(Mask, I, J) = 0);
end;

procedure BuiltIn_Shift (Direction: word; Count : Word; Fill : TFillColor); far;
begin
    Video^.ShiftRegion(0, 0, Video^.Width - 1, Video^.Height - 1, Direction, Count, Fill);
end;

procedure BuiltIn_ShiftRegion (x1, y1, x2, y2 : integer; Direction: word; Count : Word; Fill : TFillColor); far;
var
    X, Y : integer;
begin
    case Direction of
        dmUp : begin
            for Y := Y1 to Y2 - Count do
                for X := X1 to X2 do
                    Video^.PutPixel(X, Y, Video^.GetPixel(X, Y + Count));
            if Fill <> -1 then
                Video^.Region(X1, Y2 - Count + 1, X2, Y2, Fill);
        end;
        dmDown : begin
            for Y := Y2 downto Y1 + Count do
                for X := X1 to X2 do
                    Video^.PutPixel(X, Y, Video^.GetPixel(X, Y - Count));
            if Fill <> -1 then
                Video^.Region(X1, Y1, X2, Y1 + Count - 1, Fill);
        end;
        dmLeft : begin
            for Y := Y1 to Y2 do
                for X := X1 to X2 - Count do
                    Video^.PutPixel(X, Y, Video^.GetPixel(X + Count, Y));
            if Fill <> -1 then
                Video^.Region(X2 - Count + 1, Y1, X2, Y2, Fill);
        end;
        dmRight : begin
            for Y := Y1 to Y2 do
                for X := X2 downto X1 + Count do
                    Video^.PutPixel(X, Y, Video^.GetPixel(X - Count, Y));
            if Fill <> -1 then
                Video^.Region(X1, Y1, X1 + Count - 1, Y2, Fill);
        end;
        dmUpLeft : begin
            for Y := Y1 to Y2 - Count do
                for X := X1 to X2 - Count do
                    Video^.PutPixel(X, Y, Video^.GetPixel(X + Count, Y + Count));
            if Fill <> -1 then begin
                Video^.Region(X1, Y2 - Count + 1, X2, Y2, Fill);
                Video^.Region(X2 - Count + 1, Y1, X2, Y2, Fill);
            end;
        end;
        dmDownLeft : begin
             for Y := Y2 downto Y1 + Count do
                for X := X1 to X2 - Count do
                    Video^.PutPixel(X, Y, Video^.GetPixel(X + Count, Y - Count));
            if Fill <> -1 then begin
                Video^.Region(X1, Y1, X2, Y1 + Count - 1, Fill);
                Video^.Region(X2 - Count + 1, Y1, X2, Y2, Fill);
           end;
       end;
        dmUpRight : begin
            for Y := Y1 to Y2 - Count do
                for X := X2 downto X1 + Count do
                    Video^.PutPixel(X, Y, Video^.GetPixel(X - Count, Y + Count));
            if Fill <> -1 then begin
                Video^.Region(X1, Y2 - Count + 1, X2, Y2, Fill);
                Video^.Region(X1, Y1, X1 + Count - 1, Y2, Fill);
            end;
        end;
        dmDownRight : begin
            for Y := Y2 downto Y1 + Count do
                for X := X2 downto X1 + Count do
                    Video^.PutPixel(X, Y, Video^.GetPixel(X - Count, Y - Count));
            if Fill <> -1 then begin
                Video^.Region(X1, Y1, X2, Y1 + Count - 1, Fill);
                Video^.Region(X1, Y1, X1 + Count - 1, Y2, Fill);
            end;
        end;
    end;
end;

procedure BuiltIn_ShiftImage (var Image : PImage; Direction: TDirection; Count : Word; Fill : TFillColor); far;
begin
    Video^.ImageExplode(Image);
    Video^.ShiftImageRegion(Image, 0, 0, Image^.Width - 1, Image^.Height - 1, Direction, Count, Fill);
end;

procedure BuiltIn_ShiftImageRegion (var Image : PImage; x1, y1, x2, y2 : integer;
    Direction: TDirection; Count : Word; Fill : TFillColor); far;
var
    X, Y : integer;
begin
    Video^.ImageExplode(Image);
    case Direction of
        dmUp : begin
            for Y := Y1 to Y2 - Count do
                for X := X1 to X2 do
                    Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, X, Y + Count));
            if Fill <> -1 then
                Video^.ImageRegion(Image, X1, Y2 - Count + 1, X2, Y2, Fill);
        end;
        dmDown : begin
            for Y := Y2 downto Y1 + Count do
                for X := X1 to X2 do
                    Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, X, Y - Count));
            if Fill <> -1 then
                Video^.ImageRegion(Image, X1, Y1, X2, Y1 + Count - 1, Fill);
        end;
        dmLeft : begin
            for Y := Y1 to Y2 do
                for X := X1 to X2 - Count do
                    Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, X + Count, Y));
            if Fill <> -1 then
                Video^.ImageRegion(Image, X2 - Count + 1, Y1, X2, Y2, Fill);
        end;
        dmRight : begin
            for Y := Y1 to Y2 do
                for X := X2 downto X1 + Count do
                    Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, X - Count, Y));
            if Fill <> -1 then
                Video^.ImageRegion(Image, X1, Y1, X1 + Count - 1, Y2, Fill);
        end;
        dmUpLeft : begin
            for Y := Y1 to Y2 - Count do
                for X := X1 to X2 - Count do
                    Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, X + Count, Y + Count));
            if Fill <> -1 then begin
                Video^.ImageRegion(Image, X1, Y2 - Count + 1, X2, Y2, Fill);
                Video^.ImageRegion(Image, X2 - Count + 1, Y1, X2, Y2, Fill);
            end;
        end;
        dmDownLeft : begin
             for Y := Y2 downto Y1 + Count do
                for X := X1 to X2 - Count do
                    Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, X + Count, Y - Count));
            if Fill <> -1 then begin
                Video^.ImageRegion(Image, X1, Y1, X2, Y1 + Count - 1, Fill);
                Video^.ImageRegion(Image, X2 - Count + 1, Y1, X2, Y2, Fill);
           end;
       end;
        dmUpRight : begin
            for Y := Y1 to Y2 - Count do
                for X := X2 downto X1 + Count do
                    Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, X - Count, Y + Count));
            if Fill <> -1 then begin
                Video^.ImageRegion(Image, X1, Y2 - Count + 1, X2, Y2, Fill);
                Video^.ImageRegion(Image, X1, Y1, X1 + Count - 1, Y2, Fill);
            end;
        end;
        dmDownRight : begin
            for Y := Y2 downto Y1 + Count do
                for X := X2 downto X1 + Count do
                    Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, X - Count, Y - Count));
            if Fill <> -1 then begin
                Video^.ImageRegion(Image, X1, Y1, X2, Y1 + Count - 1, Fill);
                Video^.ImageRegion(Image, X1, Y1, X1 + Count - 1, Y2, Fill);
            end;
        end;
    end;
end;

procedure BuiltIn_ImageFlip (var Image : PImage); far;
var
    X, Y : integer;
    C : TColor;
begin
    Video^.ImageExplode(Image);
    for Y := 0 to (Image^.Height - 1) div 2 do
        for X := 0 to Image^.Width - 1 do begin
            C := Video^.ImageGetPixel(Image, X, Y);
            Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, X, Image^.Height - 1 - Y));
            Video^.ImagePutPixel(Image, X, Image^.Height - 1 - Y, C);
        end
end;

procedure BuiltIn_ImageMirror (var Image : PImage); far;
var
    X, Y : integer;
    C : TColor;
begin
    Video^.ImageExplode(Image);
    for Y := 0 to Image^.Height - 1 do
        for X := 0 to (Image^.Width - 1) div 2 do begin
            C := Video^.ImageGetPixel(Image, X, Y);
            Video^.ImagePutPixel(Image, X, Y, Video^.ImageGetPixel(Image, Image^.Width - X - 1, Y));
            Video^.ImagePutPixel(Image, Image^.Width - X - 1, Y, C);
        end
end;

{ High Level }

function BuiltIn_MemAlloc(Size : word; Aligned : boolean) : Pointer; far;
var
    P : Pointer;
begin
    if Aligned then
        GetMemAlign(P, Size)
    else
        GetMem(P, Size);
    BuiltIn_MemAlloc := P;
end;

procedure BuiltIn_MemRelease(var P : pointer; Size : Word); far;
begin
    if Assigned(P) then begin
        FreeMem(P, Size);
        P := nil;
    end;
end;

procedure BuiltIn_UpdatePrepare; far;
begin
    if Assigned(MousePtr) and (MousePtr^.NeedsDrawn = False) and
    (MousePtr^.Visible) then
        Video^.SpriteUndraw(MousePtr);
end;

procedure BuiltIn_Update; far;
begin
    Video^.UpdateSprites;
    Video^.UpdateVideo;
end;

function NewImageBySize(Width, Height, Size : word) : PImage;
var
    P : PImage;
begin
    P := nil;
    if Size > 0 then GetMem(P, Size + ImageHeaderSize);
    if Assigned(P) then begin
        P^.Width := Width;
        P^.Height := Height;
        P^.ByteWidth := Width;
        P^.Compression := icaUncompressed;
        P^.DataSize := Size;
        P^.Reserved := 0;
    end;
    {$IFDEF LOGS} Log('New Image ' + PtrHex(P) + ' ' + IntStr(Width) + 'x' + IntStr(Height)
        + ', ' + IntStr(Size + ImageHeaderSize) + ' bytes' );
    {$ENDIF}
    NewImageBySize := P;
end;

function NewMaskBySize(Width, Height, Size : word) : PMask;
var
    P : PMask;
begin
    P := nil;
    if Size > 0 then GetMem(P, Size + MaskHeaderSize);
    if Assigned(P) then begin
        P^.Width := Width;
        P^.ByteWidth := Width shr 3;
        if Width and $07 <> 0 then Inc(P^.ByteWidth);
        P^.Height := Height;
        P^.Compression := icaUncompressed;
        P^.DataSize := Size;
        P^.Reserved := 0;
    end;
    {$IFDEF LOGS} Log('New Mask ' + PtrHex(P) + ' ' + IntStr(Width) + 'x' + IntStr(Height)
        + ', ' + IntStr(Size + MaskHeaderSize) + ' bytes' );
    {$ENDIF}
    NewMaskBySize := P;
end;

function BuiltIn_NewImage (Width, Height : word) : PImage; far;
begin
    BuiltIn_NewImage := NewImageBySize(Width, Height,
        Video^.ImageSize(Width,Height) - ImageHeaderSize);
end;

procedure BuiltIn_FreeImage (var Image : PImage); far;
begin
    if Assigned(Image) then begin
        {$IFDEF LOGS}
            Log('Free Image ' + PtrHex(Image) + ', ' +
                IntStr(Video^.ImageSizeOf(Image)) + ' bytes' );
        {$ENDIF}
        FreeMem(Image, Image^.DataSize + ImageHeaderSize);
        Image := nil;
    end;
end;

function BuiltIn_ImageSizeOf (Image : PImage) : word; far;
begin
    if Assigned(Image) then
        BuiltIn_ImageSizeOf := Image^.DataSize + ImageHeaderSize
    else
        BuiltIn_ImageSizeOf := 0;
end;

function BuiltIn_ImageSizeData(Image : PImage) : word; far;
begin
    BuiltIn_ImageSizeData := 0;
    if not Assigned(Image) then exit;
    BuiltIn_ImageSizeData := Image^.DataSize;
end;

function BuiltIn_CloneImage (Image : PImage) : PImage; far;
var
    P : PImage;
begin
    P := nil;
    if Assigned(Image) then
        GetMem(P, Image^.DataSize + ImageHeaderSize);
    if Assigned(P) then begin
        Move(Image^, P^, Image^.DataSize + ImageHeaderSize);
    end;
    {$IFDEF LOGS}
        Log('Cloned Image ' + PtrHex(Image) + ' as ' + PtrHex(P) + ', ' +
            IntStr(Image^.DataSize + ImageHeaderSize) + ' bytes');
    {$ENDIF}
    BuiltIn_CloneImage := P;
end;

function BuiltIn_NewMask  (Width, Height : word) : PMask; far;
begin
    BuiltIn_NewMask := NewMaskBySize(Width, Height,
        Video^.MaskSize(Width,Height) - MaskHeaderSize);
end;

procedure BuiltIn_FreeMask (var Mask : PMask); far;
begin
    if Assigned(Mask) then begin
        {$IFDEF LOGS} Log('Free Mask ' + PtrHex(Mask) + ', '
            + IntStr(Video^.MaskSizeOf(Mask)) + ' bytes'); {$ENDIF}
        FreeMem(Mask, Mask^.DataSize + MaskHeaderSize);
        Mask := nil;
    end;
end;

function BuiltIn_CloneMask(Mask : PMask) : PMask; far;
var
    P : PMask;
begin
    P := nil;
    if Assigned(Mask) then
        GetMem(P, Mask^.DataSize + MaskHeaderSize);
    if Assigned(P) then begin
        Move(Mask^, P^, Mask^.DataSize + MaskHeaderSize);
    end;
    {$IFDEF LOGS}
        Log('Cloned Mask ' + PtrHex(Mask) + ' as ' + PtrHex(P) + ', ' +
            IntStr(Mask^.DataSize + MaskHeaderSize) + ' bytes');
    {$ENDIF}
    BuiltIn_CloneMask := P;
end;

function BuiltIn_MaskSize  (Width, Height : word) : word; far;
var
    I : LongInt;
begin
    I := (Width shr 3);
    if Width and $07 <> 0 then Inc(I);
    I := I * Height + MaskHeaderSize;
    if I > $0fff then I := 0;
    BuiltIn_MaskSize := I;
end;

function BuiltIn_MaskSizeOf (Mask : PMask) : word; far;
begin
    BuiltIn_MaskSizeOf := 0;
    if not Assigned(Mask) then exit;
    BuiltIn_MaskSizeOf := Video^.MaskSize(Mask^.Width, Mask^.Height);
end;

function BuiltIn_MaskSizeData (Mask : PMask) : word; far;
begin
    BuiltIn_MaskSizeData := 0;
    if not Assigned(Mask) then exit;
    BuiltIn_MaskSizeData := Video^.MaskSizeOf(Mask) - MaskHeaderSize;
end;

procedure BuiltIn_FadePalettes (var Palettes : TRGBPalettes; Percent : word); far;
var
    I : word;
    X : word;
begin
    for I := 0 to Video^.Colors - 1 do begin
        X := Word(Palettes[I].Red) * Percent div 100;
        if X < 1 then X := 1;
        if (X > Palettes[I].Red) or (Palettes[I].Red < 1) then
            Palettes[I].Red := 0
        else
            Dec(Palettes[I].Red, X);
        X := Word(Palettes[I].Green) * Percent div 100;
        if X < 1 then X := 1;
        if (X > Palettes[I].Green) or (Palettes[I].Green < 1) then
            Palettes[I].Green := 0
        else
            Dec(Palettes[I].Green, X);
        X := Word(Palettes[I].Blue) * Percent div 100;
        if X < 1 then X := 1;
        if (X > Palettes[I].Blue) or (Palettes[I].Blue < 1) then
            Palettes[I].Blue := 0
        else
            Dec(Palettes[I].Blue, X);
    end;
end;

procedure BuiltIn_FadeOut (Palettes : TRGBPalettes); far;
var
    I : integer;
    LT : longInt;
    Pal : TRGBPalettes;
begin
    I := 0;
    for I := 0 to 33 do begin
        Pal := Palettes;
        Video^.FadePalettes(Pal, I * 3);
        Video^.SetPalettes(Pal);
        LT := TimerTick;
        while LT = TimerTick do;
    end;
end;

procedure BuiltIn_FadeIn (Palettes : TRGBPalettes); far;
var
    I : integer;
    LT : longInt;
    Pal : TRGBPalettes;
begin
    I := 0;
    for I := 0 to 33do begin
        Pal := Palettes;
        Video^.FadePalettes(Pal, (33 - I) * 3);
        Video^.SetPalettes(Pal);
        LT := TimerTick;
        while LT = TimerTick do;
    end;
    Video^.SetPalettes(Palettes);
end;

procedure BestMatchPalettes(var Image : PImage; var Old, Pal : TRGBPalettes );
type
    TMatchData = array[0..255] of record
        Best   : word;
        State  : word;
        Diff   : LongInt;
    end;
var
    I, J, L : word;
    LD, D : LongInt;
    Data : TMatchData;
begin
    Video^.ImageExplode(Image);
    FillChar(Data, Sizeof(Data), 0);
    for J := 0 to 255 do begin
        L := 255;
        LD := MaxLongInt;
        if Identical(Old[J], Pal[J], Sizeof(Old[J])) then begin
            L := J;
            LD := 0;
        end else
            for I := 0 to 255 do begin
                if Identical(Old[I], Pal[J], Sizeof(Old[I])) then begin
                    L := I;
                    LD := 0;
                    break;
                end else if I > 15 then begin
                    D := Abs(Old[I].Red - Pal[I].Red) +
                         Abs(Old[I].Green - Pal[I].Green) +
                         Abs(Old[I].Blue - Pal[I].Blue);
                    if LD > D then begin
                        L := I;
                        LD := D;
                    end;
                end;

        end;
        Data[J].Best := L;
        Data[J].Diff := LD;
        if J > 15 then
            Data[J].State := 0
        else
            Data[J].State := 1;
    end;

    for I := 16 to 255 do begin
        Inc(Data[Data[I].Best].State);
        if (Data[Data[I].Best].State > 1) then Continue;
        if Data[I].Diff = 0 then continue;
        Old[I] := Pal[Data[I].Best];
    end;

    for I := 0 to Image^.Height - 1 do
        for J := 0 to Image^.Width - 1 do
            Video^.ImagePutPixel(Image, J, I, Data[Video^.ImageGetPixel(Image, J, I)].Best);
end;

procedure BuiltIn_ImageRemap (var Image : PImage; Pal : TRGBPalettes; var ImagePal : TRGBPalettes; Mode : word); far;
var
    Hold : TRGBPalettes;
begin
    if not assigned(Image) then Exit;
    case Mode of
        ipmIgnore : begin { ignore it, but then why did you call this proc? }
            ImagePal := Pal;
        end;
        ipmComplete : begin     { Replace existing color palette }
            { Leave As-Is }
        end;
        ipmMatch : begin  { Remap image to best fit Pal }
            Hold := Pal;
            BestMatchPalettes(Image, Pal, ImagePal);
            ImagePal := Hold;
        end;
        ipmOverride : begin { Remap image to best fit Pal and keep new ImagePal }
            BestMatchPalettes(Image, Pal, ImagePal);
            ImagePal := Pal;
        end;
        ipmPooled : begin
            { not yet implemented, probably won't either, we will see }
            FatalError(erOperation_Not_Supported,'color palette pool');
        end;
    else
        FatalError(erRange_Check_Error,'image remap mode')
    end;
end;

function BuiltIn_NewFont (Width, Height : word) : PFont; far;
var
    Font : PFont;
    Sz, B : Word;
begin
    Font := nil;
    B := Width div 8;
    if Width mod 8 <> 0 then Inc(B);
    Sz := B * Height * 256;
    if (Width <= 32) and (Height <= 32) then begin
        Font := New(PFont);
        if Assigned(Font) then begin
            FillChar(Font^, Sizeof(TFont), 0);
            GetMem(Font^.BitMap, Sz);
            if not Assigned(Font^.BitMap) then begin
                Dispose(Font);
                Font := nil;
            end;
        end;
    end;
    if Assigned(Font) then begin
        FillChar(Font^.Bitmap^, Sz, 0);
        Font^.Width := Width;
        Font^.Height := Height;
        Font^.ByteWidth := B;
    end;
    BuiltIn_NewFont := Font;
end;

procedure BuiltIn_FreeFont (var Font : PFont); far;
var
    Sz : Word;
begin
    if Not Assigned(Font) then Exit;
    if Assigned(Font^.Bitmap) then begin
        Sz := Font^.Width div 8;
        if Font^.Width mod 8 <> 0 then Inc(Sz);
        Sz := Sz * Font^.Height * 256;
        FreeMem(Font^.Bitmap, Sz);
        Dispose(Font);
        Font := nil;
   end;
end;

procedure BuiltIn_PutBits (Bits : Pointer; x, y : integer;
    BytesW, BytesH : byte; Color : TColor); far;
var
    I, J, K : integer;
begin
    for J := 0 to BytesH - 1 do begin
        K := J * BytesW;
        for I := 0 to BytesW * 8 - 1 do
            if (PBytes(Bits)^[K + I shr 3] shl (I and 7)) and $80 = $80 then
                Video^.PutPixel(X + I, Y + J, Color);
    end;
end;

procedure BuiltIn_PutImageBits (var Image : PImage; Bits : Pointer; x, y : integer;
     BytesW, BytesH : byte; Color : TColor); far;
var
    I, J, K : integer;
begin
    for J := 0 to BytesH - 1 do begin
        K := J * BytesW;
        for I := 0 to BytesW * 8 - 1 do
            if (PBytes(Bits)^[K + I shr 3] shl (I and 7)) and $80 = $80 then
                Video^.ImagePutPixel(Image, X + I, Y + J, Color);
    end;
end;

function  BuiltIn_GetFont : PFont; far;
begin
    BuiltIn_GetFont := Video^.FontSettings.Font;
end;

procedure BuiltIn_SetFont (Font : PFont); far;
begin
    Video^.FontSettings.Font := Font;
end;

function  BuiltIn_GetMonospace : boolean; far;
begin
    BuiltIn_GetMonospace := Video^.FontSettings.Monospace;
end;

procedure BuiltIn_SetMonospace (Monospace : boolean); far;
begin
    Video^.FontSettings.Monospace := Monospace;
end;

function  BuiltIn_GetFontDirection : TDirection; far;
begin
    BuiltIn_GetFontDirection := Video^.FontSettings.Direction;
end;

procedure BuiltIn_SetFontDirection(Direction : TDirection); far;
begin
    if (Direction < 1) or (Direction > 9) then begin
        SetError(erRange_Check_Error);
        Exit;
    end;
    { Cause I'm Lazy... For now, Just gonna ignore direction changes. }
    { if Direction < 8 then
        Video^.FontSettings.Direction := Direction; }
end;

procedure BuiltIn_PutChar (X, Y : integer; C : Char; Color : TColor); far;
var
    B, O, H : integer;
begin
    if not Assigned(Video^.FontSettings.Font) then exit;
    H := Video^.FontSettings.Font^.Height;
    B := Video^.FontSettings.Font^.ByteWidth;
    O := B * H * Ord(C);
    Video^.PutBits(@Video^.FontSettings.Font^.BitMap^[O], X, Y, B, H, Color);
end;

procedure BuiltIn_PutText (X, Y : integer; Str : String; Color : TColor); far;
var
    I : word;
    W : integer;
begin
    if not Assigned(Video^.FontSettings.Font) then exit;
    if Video^.FontSettings.Monospace then begin
        W := Video^.FontSettings.Font^.MonoWidth;
        for I := 1 to Length(Str) do
            Video^.PutChar(X + (I - 1) * W, Y, Str[I], Color);
    end else begin
        for I := 1 to Length(Str) do begin
            W := Video^.FontSettings.Font^.Offsets[Ord(Str[I])].Offset;
            if W = Video^.FontSettings.Font^.Width then begin
                Inc(X, Video^.FontSettings.Font^.Offsets[32].Maximum +
                    Video^.FontSettings.Font^.Spacing + 1);
                continue;
            end;
            Dec(X, W);
            Video^.PutChar(X, Y, Str[I], Color);
            Inc(X, Video^.FontSettings.Font^.Offsets[Ord(Str[I])].Maximum +
                Video^.FontSettings.Font^.Spacing + 1);
        end;
    end;
end;

procedure BuiltIn_ImagePutChar (var Image : PImage; X, Y : integer; C : Char; Color : TColor); far;
var
    B, O, I, J, H, W : integer;
    Data : PBytes;
begin
    {$IFDEF SAFETY}
    if not Assigned(Image) then exit;
    {$ENDIF}
    if NOT Assigned(Video^.FontSettings.Font) then exit;
    Video^.ImageExplode(Image);
    {   ^   That not is not lowercase. No reason. Just felt like it. }
    H := Video^.FontSettings.Font^.Height;
    W := Video^.FontSettings.Font^.Width;
    Data := Video^.FontSettings.Font^.BitMap;
    B := W div 8;
    if W mod 8 <> 0 then Inc(B);
    O := B * H * Ord(C);
    for J := 0 to H - 1 do if (J + Y >= 0) and (J + Y < Image^.Height) then
        for I := 0 to W - 1 do if (I + X >= 0) and (I + X < Image^.Width) then
            if (Data^[O + (J * B) + I div 8] shl (I mod 8)) and $80 = $80 then
                Video^.ImagePutPixel(Image, X + I, Y + J, Color);
end;

procedure BuiltIn_ImagePutText (var Image : PImage; X, Y : integer; Str : String; Color : TColor); far;
var
    I : word;
    W : integer;
begin
    if not Assigned(Image) then exit;
    if not Assigned(Video^.FontSettings.Font) then exit;
    Video^.ImageExplode(Image);
    if Video^.FontSettings.Monospace then begin
        W := Video^.FontSettings.Font^.MonoWidth;
        for I := 1 to Length(Str) do
            Video^.ImagePutChar(Image, X + (I - 1) * W, Y, Str[I], Color);
    end else begin
        for I := 1 to Length(Str) do begin
            W := Video^.FontSettings.Font^.Offsets[Ord(Str[I])].Offset;
            if W = Video^.FontSettings.Font^.Width then begin
                Inc(X, Video^.FontSettings.Font^.Offsets[32].Maximum +
                    Video^.FontSettings.Font^.Spacing + 1);
                continue;
            end;
            Dec(X, W);
            Video^.ImagePutChar(Image, X, Y, Str[I], Color);
            Inc(X, Video^.FontSettings.Font^.Offsets[Ord(Str[I])].Maximum +
                Video^.FontSettings.Font^.Spacing + 1);
        end;
    end;
end;

function  BuiltIn_TextWidth (Str : String) : integer; far;
var
    I, X : word;
    W : integer;
begin
    if not Assigned(Video^.FontSettings.Font) then
        BuiltIn_TextWidth := 0
    else
    if Video^.FontSettings.Monospace then begin
        BuiltIn_TextWidth := Video^.FontSettings.Font^.MonoWidth * Length(Str)
    end else begin
        X := 0;
        for I := 1 to Length(Str) do begin
            W := Video^.FontSettings.Font^.Offsets[Ord(Str[I])].Offset;
            if W = Video^.FontSettings.Font^.Width then begin
                Inc(X, Video^.FontSettings.Font^.Offsets[32].Maximum +
                    Video^.FontSettings.Font^.Spacing + 1);
                continue;
            end;
            Dec(X, W);
            Inc(X, Video^.FontSettings.Font^.Offsets[Ord(Str[I])].Maximum +
                Video^.FontSettings.Font^.Spacing + 1);
        end;
        BuiltIn_TextWidth := X;
    end;
end;

function  BuiltIn_TextHeight (Str : String) : integer; far;
begin
    if not Assigned(Video^.FontSettings.Font) then
        BuiltIn_TextHeight := 0
    else
        BuiltIn_TextHeight := Video^.FontSettings.Font^.Height;
end;

procedure BuiltIn_Frame(Thickness : Integer; Color1, Color2 : TColor); far;
begin
    Video^.FrameRegion(0,0,Video^.Width -1,Video^.Height-1, Thickness, Color1, Color2);
end;

procedure BuiltIn_FrameRegion(x1, y1, x2, y2 : integer; Thickness : Integer;
Color1, Color2 : TColor); far;
var
    I : integer;
begin
    for I := 0 to Thickness - 1 do begin
        Video^.Region(x1 + I, y1 + I, x2 - I, y1 + I, Color1);
        Video^.Region(x1 + I, y1 + I, x1 + I, y2 - I, Color1);
        Video^.Region(x1 + I + 1,y2-I,x2-I,y2-I,Color2);
        Video^.Region(x2-I,y1 + I,x2-I,y2-I - 1, Color2);
    end;
end;

procedure BuiltIn_ImageFrame(Image : PImage; Thickness : Integer; Color1, Color2 : TColor); far;
var
    I : integer;
begin
    {$IFDEF SAFETY}
    if not Assigned(Image) then exit;
    {$ENDIF}
    Video^.ImageExplode(Image);
    for I := 0 to Thickness - 1 do begin
        Video^.ImageRegion(Image, I,I,Image^.Width - I - 1, I,Color1);
        Video^.ImageRegion(Image, I,I,I,Image^.Height - I - 1,Color1);
        Video^.ImageRegion(Image, I + 1,Image^.Height-I-1,Image^.Width-I-1,Image^.Height-I-1,Color2);
        Video^.ImageRegion(Image, Image^.Width-I-1,I + 1,Image^.Width-I-1,Image^.Height-I-2, Color2);
    end;
end;

procedure BuiltIn_Flood (Image : PImage; X, Y : integer; Color : TColor); far;

type
    PFillBuf = ^TFillBuf;
    TFillBuf = array[0..1023] of TPoint;

var
    SC: TColor;
    Pixels : LongInt;
    MX, MY : integer;
    BH, BT, BN : word;
    Buf : PFillBuf;

    function Get(X, Y : integer) : TColor;
    begin
        if Assigned(Image) then
            Get := Video^.ImageGetPixel(Image, X, Y)
        else
            Get := Video^.GetPixel(X,Y);
    end;

    procedure Put(X, Y : integer);
    begin
        if Assigned(Image) then
            Video^.ImagePutPixel(Image, X, Y, Color)
        else
            Video^.PutPixel(X,Y,Color);
    end;

    function Add(X, Y :integer) : boolean;
    begin
        Add := False;
        if (X<0) or (Y<0) or (X>MX) or (Y>MY) then exit;
        if Get(X,Y) <> SC then exit;
        BN := BH + 1;
        if BN > High(TFillBuf) then BN := 0;
        if BN <> BT then begin
            BH := BN;
            Buf^[BH].X := X;
            Buf^[BH].Y := Y;
            Add := True;
        end;
    end;
var
    E : TEvent;
begin
    Buf := New(PFillBuf);
    BH := 0;
    BT := 0;
    if not Assigned(Buf) then exit;
    Pixels := 0;
    {$IFDEF LOGS}
    if Assigned(Image) then
        Log('Image ' + PtrHex(Image) + ' Flood: ')
    else
        Log('Stage Flood: ');
    {$ENDIF}
    if Assigned(Image) then begin
        MX := Image^.Width - 1;
        MY := Image^.Height - 1;
    end else begin
        MX := Video^.Width - 1;
        MY := Video^.Height - 1;
    end;
    SC := Get(X,Y);
    { Video^.Fill(SC); }
    if Color <> SC then
    repeat
        if Get(X, Y) = SC then begin
            Inc(Pixels);
            Put(X, Y);
            { Video^.PutPixel(X,Y, Color);
            if Pixels and $01 = 0 then Video^.Update; }
            Add(X+1, Y);
            Add(X-1, Y);
            Add(X, Y+1);
            Add(X, Y-1);
        end;
        if BH = BT then Break;
        Inc(BT);
        if BT > High(TFillBuf) then BT := 0;
        X := Buf^[BT].X;
        Y := Buf^[BT].Y;
    until GetEvent(E) and (E.Kind = evKeyrelease);
    {$IFDEF LOGS}
    Log('End Flood: ' + IntStr(Pixels) + ' pixels');
    {$ENDIF}
    Dispose(Buf);
end;

procedure BuiltIn_FloodFill(X, Y : integer; Color : TColor); far;
begin
    BuiltIn_Flood(nil, X, Y, Color);
end;

procedure BuiltIn_ImageFloodFill (Image : PImage; X, Y : integer; Color : TColor); far;
begin
    if not Assigned(Image) then exit;
    Video^.ImageExplode(Image);
    if not Assigned(Image) then exit;
    BuiltIn_Flood(Image, X, Y, Color);
end;

function BuiltIn_ImageImplode (var Image : PImage) : boolean; far;
begin
    BuiltIn_ImageImplode := False;
    if (not Assigned(Image)) or (Image^.Compression <> 0) then exit;
    ImageDRE(Image);
    BuiltIn_ImageImplode := Image^.Compression <> 0;
end;

function BuiltIn_ImageExplode (var Image : PImage) : boolean; far;
begin
    BuiltIn_ImageExplode := False;
    if (not Assigned(Image)) or (Image^.Compression = 0) then exit;
    ImageUnDRE(Image);
    BuiltIn_ImageExplode := Image^.Compression = 0;
end;

function BuiltIn_MaskImplode (var Mask : PMask) : boolean; far;
begin
    BuiltIn_MaskImplode := False;
    if (not Assigned(Mask)) or (Mask^.Compression <> 0) then exit;
    ImageDRE(PImage(Mask)); { Images and Masks use the same record structure }
    BuiltIn_MaskImplode := Mask^.Compression <> 0;
end;

function BuiltIn_MaskExplode (var Mask : PMask) : boolean; far;
begin
    BuiltIn_MaskExplode := False;
    if (not Assigned(Mask)) or (Mask^.Compression = 0) then exit;
    ImageUnDRE(PImage(Mask));
    BuiltIn_MaskExplode := Mask^.Compression = 0;
end;

{ Optional }
function BuiltIn_ExtendedData : pointer; far;
begin
    BuiltIn_ExtendedData := nil;
end;

procedure BuiltIn_GetViewPort (var Area : TArea); far; { not implemented }
begin
    Area.Left := 0;
    Area.Top := 0;
    Area.Right := Video^.Width - 1;
    Area.Bottom := Video^.Height - 1;
end;

procedure BuiltIn_SetViewPort (Area : TArea); far; { not implemented }
begin
end;

{$I VIDEONUL.INC}
{$I SPRITES.INC}


function GetBuiltInVideoFunctions : PDriver;
var
    BuiltIn : PVideoDriver;
begin
    BuiltIn := New(PVideoDriver);
    if not Assigned(BuiltIn) then
        FatalError(erInsufficient_memory,'init graphics sub-system');
    FillChar(BuiltIn^, SizeOf(TVideoDriver), 0);
    with BuiltIn^ do begin
        { Performance }
        Fill            := BuiltIn_Fill;
        Region          := BuiltIn_Region;
        ImageFill       := BuiltIn_ImageFill;
        ImageRegion     := BuiltIn_ImageRegion;
        GetImage        := BuiltIn_GetImage;
        PutImage        := BuiltIn_PutImage;
        PutImageMode    := BuiltIn_PutImageMode;
        PutMask         := BuiltIn_PutMask;
        PutMaskMode     := BuiltIn_PutMaskMode;
        PutBits         := BuiltIn_PutBits;
        PutImageBits    := BuiltIn_PutImageBits;
        ImageToMask     := BuiltIn_ImageToMask;
        MaskToImage     := BuiltIn_MaskToImage;
        MaskInvert      := BuiltIn_MaskInvert;
        Shift           := BuiltIn_Shift;
        ShiftRegion     := BuiltIn_ShiftRegion;
        ShiftImage      := BuiltIn_ShiftImage;
        ShiftImageRegion:= BuiltIn_ShiftImageRegion;
        ImageFlip       := BuiltIn_ImageFlip;
        ImageMirror     := BuiltIn_ImageMirror;
        { High Level }
        MemAlloc        := BuiltIn_MemAlloc;
        MemRelease      := BuiltIn_MemRelease;
        Prepare         := BuiltIn_UpdatePrepare;
        Update          := BuiltIn_Update;
        NewImage        := BuiltIn_NewImage;
        FreeImage       := BuiltIn_FreeImage;
        CloneImage      := BuiltIn_CloneImage;
        NewMask         := BuiltIn_NewMask;
        FreeMask        := BuiltIn_FreeMask;
        CloneMask       := BuiltIn_CloneMask;
        ImageSizeOf     := BuiltIn_ImageSizeOf;
        ImageSizeData   := BuiltIn_ImageSizeData;
        MaskSize        := BuiltIn_MaskSize;
        MaskSizeOf      := BuiltIn_MaskSizeOf;
        MaskSizeData    := BuiltIn_MaskSizeData;
        FadePalettes    := BuiltIn_FadePalettes;
        FadeOut         := BuiltIn_FadeOut;
        FadeIn          := BuiltIn_FadeIn;
        ImageRemap      := BuiltIn_ImageRemap;
        NewFont         := BuiltIn_NewFont;
        FreeFont        := BuiltIn_FreeFont;
        GetFont         := BuiltIn_GetFont;
        SetFont         := BuiltIn_SetFont;
        GetMonospace    := BuiltIn_GetMonospace;
        SetMonospace    := BuiltIn_SetMonospace;
        GetFontDirection:= BuiltIn_GetFontDirection;
        SetFontDirection:= BuiltIn_SetFontDirection;
        PutChar         := BuiltIn_PutChar;
        PutText         := BuiltIn_PutText;
        ImagePutChar    := BuiltIn_ImagePutChar;
        ImagePutText    := BuiltIn_ImagePutText;
        TextWidth       := BuiltIn_TextWidth;
        TextHeight      := BuiltIn_TextHeight;
        NewSprite       := BuiltIn_NewSprite;
        FreeSprite      := BuiltIn_FreeSprite;
        CloneSprite     := BuiltIn_CloneSprite;
        SpriteSizeOf    := BuiltIn_SpriteSizeOf;
        SpriteSort      := BuiltIn_SpriteSort;
        SpriteRemove    := BuiltIn_SpriteRemove;
        SpriteAdd       := BuiltIn_SpriteAdd;
        SpriteSetVisible:= BuiltIn_SpriteSetVisible;
        SpriteUndraw    := BuiltIn_SpriteUndraw;
        SpriteUndrawArea:= BuiltIn_SpriteUndrawArea;
        SpriteUndrawAll := BuiltIn_SpriteUndrawAll;
        SpriteShow      := BuiltIn_SpriteShow;
        SpriteHide      := BuiltIn_SpriteHide;
        SpriteMove      := BuiltIn_SpriteMove;
        SpriteChange    := BuiltIn_SpriteChange;
        SpriteNext      := BuiltIn_SpriteNext;
        SpriteNextAll   := BuiltIn_SpriteNextAll;
        SpriteCovers    := BuiltIn_SpriteCovers;
        SpriteWhere     := BuiltIn_SpriteWhere;
        SpriteCollide   := BuiltIn_SpriteCollide;
        SpriteGetSeq    := BuiltIn_SpriteGetSeq;
        SpriteSetSeq    := BuiltIn_SpriteSetSeq;
        UpdateSprites   := BuiltIn_UpdateSprites;
        Frame           := BuiltIn_Frame;
        FrameRegion     := BuiltIn_FrameRegion;
        ImageFrame      := BuiltIn_ImageFrame;
        FloodFill       := BuiltIn_FloodFill;
        ImageFloodFill  := BuiltIn_ImageFloodFill;
        ImageImplode    := BuiltIn_ImageImplode;
        ImageExplode    := BuiltIn_ImageExplode;
        MaskImplode     := BuiltIn_MaskImplode;
        MaskExplode     := BuiltIn_MaskExplode;

        { Optional }
        ExtendedData    := BuiltIn_ExtendedData;{ returns nil }
        GetViewPort     := BuiltIn_GetViewPort; { not implemented }
        SetViewPort     := BuiltIn_SetViewPort; { not implemented }

        { Termination marker }
        EndOfProcs.L    := $ffff;
        EndOfProcs.H    := $ffff;
    end;
    GetBuiltInVideoFunctions := PDriver(BuiltIn);
end;

procedure InitSprites;
begin
    SpriteFlag := False;
    SpriteFirst := nil;
    SpriteLast := nil;
end;
{$UNDEF SECTIMP}
end.