/*
** $Id: clock.c, initiated May 09, 2024 $
** Sexagesimal time library library
** See Copyright Notice in agena.h
*/


#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#define clock_c
#define LUA_LIB

#include "agena.h"
#include "agncmpt.h"
#include "agnxlib.h"
#include "agenalib.h"
#include "agnhlps.h"


#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_CLOCKLIBNAME "clock"
LUALIB_API int (luaopen_clock) (lua_State *L);
#endif

#define TMTYPENAME       "tm"

#define isclocktuple(L, idx) (lua_isfunction(L, idx) && agn_isutype(L, idx, TMTYPENAME))

static FORCE_INLINE void checkandgettm (lua_State *L, int idx, double *s, const char *procname) {
  int i;
  lu_byte nupvals;
  if (!(isclocktuple(L, idx))) {
    luaL_error(L, "Error in " LUA_QS ": expected a `tm` value, got %s.",
      procname, luaL_typename(L, idx));
  }
  nupvals = lua_nupvalues(L, idx);
  if (nupvals != 3)
    luaL_error(L, "Error in " LUA_QS ": `tm` value is invalid.", procname);
  for (i=0; i < 3; i++) {
    lua_getupvalue(L, idx, i + 1);
    s[i] = agn_checknumber(L, -1);
    agn_poptop(L);
  }
}


static int clock_adjust (lua_State *L) {
  int i;
  lua_Number s[3], inseconds, sgn, tmod;
  checkandgettm(L, 1, s, "clock.adjust");
  inseconds = s[0]*3600 + s[1]*60 + sun_trunc(s[2]);
  sgn = tools_sign(inseconds);  /* remember sign */
  inseconds = fabs(inseconds);  /* proceed calculation with absolute values */
  tmod = luai_nummod(inseconds, 3600);
  s[0] = luai_numintdiv(inseconds, 3600);  /* hours */
  s[1] = luai_numintdiv(tmod, 60);  /* minutes */
  s[2] = luai_nummod(tmod, 60) + sun_frac(s[2]);  /* seconds, prevent round-off errors if seconds component includes milliseconds */
  luaL_checkstack(L, 2, "not enough stack space");
  for (i=0; i < 3; i++) {  /* in-place operation */
    lua_getupvalue(L, 1, i + 1);
    lua_pushnumber(L, sgn*s[i]);
    lua_setupvalue(L, 1, i + 1);
  }
  return 0;  /* return nothing */
}


static const luaL_Reg clocklib[] = {
  {"adjust", clock_adjust},              /* added on May 09, 2024 */
  {NULL, NULL}
};


/*
** Open clock library
*/
LUALIB_API int luaopen_clock (lua_State *L) {
  luaL_register(L, AGENA_CLOCKLIBNAME, clocklib);
  return 1;
}

