/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: _FEXPAND.C $
   $Locker:  $	$Name:  $	$State: Exp $

	char *_fexpand(char *buf, char *fnam, size_t buflen)

	Expand the filename fnam to a fully-qualified path and, if LFN
	support is available, expand all shortnames into longnames.

	Supports UNC paths.
   	If buffer != NULL, buflen tells its size.
   	If buffer == NULL, a buffer of buflen bytes is malloc'ed.
   	If buflen == 0, the buffer is assumed to be large enough.

	Input:
		fnam != NULL
		buf may be NULL

	Return:
		== NULL: malloc() failed, drive access failed
		!= NULL: buffer or malloc()'ed buffer

	Note:
		<none>

	Conforms to:
		<none>

	See also:
		fullpath

	Target compilers:
		Any C compiler

	Origin:
		Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.2  1998/01/29 07:10:15  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#ifdef LFN_WIN95
#include <alloc.h>
#include <dir.h>
#include <dos.h>
#include <errno.h>
#include <string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: _FEXPAND.C 1.2 1998/01/29 07:10:15 ska Exp $";
#endif


static int fullpathLFN(char ** const buf
	, char const * const fnam, const size_t buflen)
{	char hbuf[PATH_MAX + 2];		/* temporary buffer fed into LFN API */
	struct REGPACK r;
	REG int err, l;

	assert(buf != NULL);
	assert(fnam != NULL);

	/* Win95 LFN API has the bug that it returns #3 if the
		returned path is longer than 255 characters */
	*hbuf = NUL;		/* because of the bug */
	r.r_ax = 0x7160;
	r.r_cx = 0x8002;	/* Get long filename, don't expand SUBST */
	r.r_ds = FP_SEG(fnam);
	r.r_si = FP_OFF(fnam);
	r.r_es = FP_SEG(hbuf);
	r.r_di = FP_OFF(hbuf);

	if(!((err = _callDOS(&r)) == EZERO || (err == 3 && !*hbuf))) {
		*buf = NULL;
		return err;		/* LFN failed */
	}

	l = strlen(hbuf) + 1;

	if(buflen && buflen < l)	/* buffer size supplied, but too small */
		return ERANGE;

	if(!*buf) {					/* allocate buffer */
		if((*buf = malloc(buflen? buflen: l)) == NULL)
			return ENOMEM;
	}

	memcpy(*buf, hbuf, l);

	return EZERO;			/* success */
}

_CLibFunc char *
_fexpand(char * const buf
	, char const * const fnam
	, size_t const buflen)
{	char *hbuf;

	assert(fnam != NULL);

	hbuf = buf;
	return fullpathLFN(&hbuf, fnam, buflen)
		? fullpath(buf, fnam, buflen)		/* no LFN API */
		: hbuf;								/* LFN result */
}
#endif /* LFN_WIN95 */
