/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FGETC.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int fgetc(FILE *fp)
	int getc(FILE *fp)

	Get a character from a stream.

	fgetc is guaranteed to be no preprocessor macro.

	fgetc() ignores all '\r' characters regardless in which context
	this character is found.

	Input:
		fp != NULL; a valid FILE pointer

	Return:
		EOF: if no character could be read (EOF or error)
		0..UCHAR_MAX: character

	Note:
		getc does not automatically fill the buffer nor does it
		translate the character in text mode. Therefore, getc is
		a synonym of fgetc in the current implementation.

	Conforms to:
		ANSI

	See also:
		fget, fread, fputcm fgets

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.2  1998/01/29 07:10:01  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FGETC.C 1.2 1998/01/29 07:10:01 ska Exp $";
#endif

_CLibFunc int
fgetc(FILE *fp)
{	int   ch;

	assert(fp != NULL);

	if (fp->mode & __MODE_WRITING)
		fflush(fp);

try_again:
	/* Can't read or there's been an EOF or error then return EOF */
	if ((fp->mode & (__MODE_READ | __MODE_EOF | __MODE_ERR)) != __MODE_READ)
		return EOF;

	/* Nothing in the buffer - fill it up */
	if (fp->bufpos >= fp->bufread) {
		uchar_t c;

		/* Bind stdin to stdout if it's open and line buffered */
		if( fp == stdin && stdout->fd >= 0 && (stdout->mode & _IOLBF ))
			fflush(stdout);

		if(!fget(&c, sizeof(c), fp))
			return EOF;

		ch = c;
	}
   else ch = *(fp->bufpos++);

#if __MODE_IOTRAN
	/* In MSDOS translation mode; WARN: Doesn't work with UNIX macro */
	if (ch == '\r' && (fp->mode & __MODE_IOTRAN))
		goto try_again;
#endif

	/* Because the buffer is unsigned we don't interfere with EOF */
	return ch;
}
