/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FWRITE.C $
   $Locker:  $	$Name:  $	$State: Exp $

	size_t fwrite(void *buf, size_t size, size_t nelm, FILE *fp)

	Write nelm records of size into the stream fp.

	This function does _not_ translate characters in text mode!

	Note: If this function writes less than nelm records into stream,
	it is possible that the next record has been partly written.

	Input:
		buf != NULL
		fp != NULL; a valid FILE pointer
		size and nelm may be 0

	Return:
		number of completely written records

	Note:
		Like fread() fwrite() may be called to transfer large chunks of
		data to the system. In this case the buffer is written directly
		to the system bypassing the stream buffer. This can be a big
		win.

	Conforms to:
		ANSI

	See also:
		fread, fputs, fput

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.2  1998/01/29 07:10:01  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"
#include <errno.h>
#include <io.h>
#include <string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FWRITE.C 1.2 1998/01/29 07:10:01 ska Exp $";
#endif

_CLibFunc size_t
fwrite(const void *Xbuf, size_t size, size_t nelm, FILE *fp)
{
   REG int v;
   REG const char *buf;
   int   len;
   unsigned bytes, put;

	buf = (const char*)Xbuf;

	assert(buf != NULL);
	assert(fp != NULL);

   /* This could be long, doesn't seem much point tho */
   if((bytes = size * nelm) == 0)
		return 0;		/* prevent division-by-zero trap */

   v = fp->mode;
   /* If last op was a read ... */
   if ((v & __MODE_READING) && fflush(fp))
      return 0;

   /* Can't write or there's been an EOF or error then return 0 */
   if ((v & (__MODE_WRITE | __MODE_EOF | __MODE_ERR)) != __MODE_WRITE)
      return 0;

   len = fp->bufend - fp->bufpos;

   /* Flush the buffer if not enough room */
	if (bytes > len) {
		if(fflush(fp))
			return 0;
		len = fp->bufend - fp->bufpos;
	}

	if (bytes <= len) {		/* It'll fit in the buffer ? */
		fp->mode |= __MODE_WRITING;
		memcpy(fp->bufpos, buf, bytes);
		fp->bufpos += bytes;

		/* If we're not fully buffered, flush immediately */
		if (v & (_IOLBF | _IONBF))
			fflush(fp);

		return nelm;
	}

	/* Too big for the buffer */
	put = bytes;
	do {
		len = write(fp->fd, buf, bytes);
		if( len > 0 ) {
			buf+=len; bytes-=len;
		}
	} while (len > 0 || (len == -1 && errno == EINTR));

	if (len < 0)
		fp->mode |= __MODE_ERR;

	put -= bytes;

	return put / size;
}
