/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $Id: _ALLOC.H 1.6 1998/02/11 07:40:34 ska Exp ska $
   $Locker: ska $	$Name:  $	$State: Exp $

	Definitions for the heap functions

*/

#ifndef ___ALLOC_H
#define ___ALLOC_H

#include <alloc.h>

/*
 *	Macros returned by various heap functions
 */

/* Type of a node */
#define HEAP_END 0			/* end of heap */
#define HEAP_UNUSED 1		/* unused/free node */
#define HEAP_USED 2			/* used node, must be the highest assigned number! */

/* security margin between local heap and stack */
#define HEAP_MARGIN 512		/* in bytes */

/* Return values */
#define HEAP_OK 0			/* heap OK, function successful */
#define HEAP_NOTFOUND 1		/* node/pointer not found */

/*
 *	Alignment definitions
 */
#ifndef __LHEAP_ALIGN		/* alignment of local heap */
#define __LHEAP_ALIGN 0		/* no alignment by default! */
#endif
#ifndef __FHEAP_ALIGN		/* alignment of far/huge heap */
#define __FHEAP_ALIGN 0		/* no alignment by default! */
#endif

#ifdef ___PTR				/* type of pointers of the interface */
#undef ___PTR
#endif

/* Structure of a node in the local heap */
typedef struct {
	byte type;			/* node type, way of useage */
	size_t length;		/* length of data portion */
} _lheapNode;
typedef _lheapNode *_lpHeapNode;
typedef byte *_lpHeapData;		/* local head data pointer */

/* Structure of a node in the far heap */
typedef struct _FHheapNode_ {
	byte type;						/* node type, way of useage */
	struct _FHeapNode_ huge * next;	/* poi to next _node_ */
} _fheapNode;
typedef _fheapNode huge *_fpHeapNode;
typedef byte huge *_fpHeapData;		/* far head data pointer */


#if _IN_ALLOC == 0			/* declarations of the local heap */

/************* name remappings ****************/
#define __HEAP_ALIGN __LHEAP_ALIGN
#define _heap _lheap
#define _heapNode _lheapNode
#define _pHeapNode _lpHeapNode
#define _pHeapData _lpHeapData
#define _heapMax __getSP()		/* the local heap resides below the stack */

/************* specific declarations **********/
/* Advance to next node */
#define nxtNode(node) ((_pHeapNode)((_pHeapData)&((node)[1]) + (node)->length))

/* length a node's data area */
#define nodeLength(node) ((node)->length)
#define setNodeLength(node,size) ((node)->length = (size))
#define setNxtNode(tNode,nNode)										\
	((tNode)->length = (_pHeapData)(nNode) - (_pHeapData)(tNode)	\
	 - sizeof(_heapNode))


/* join two nodes */
#define joinNodes(node,nxtNode)											\
	((node)->length += nodeLength(nxtNode) + sizeof(_heapNode))


#define ___PTR		/* normal pointer */

#else	/* _IN_ALLOC != 0 */

/************* name remappings ****************/
#define __HEAP_ALIGN __FHEAP_ALIGN
#define _heap _fheap
#define _heapNode _fheapNode
#define _pHeapNode _fpHeapNode
#define _pHeapData _fpHeapData

/* Advance to next node */
#define nxtNode(node)	((node)->next)
#define setNodeLength(node,size)										\
	((node)->next = (node) + (size) + sizeof(_heapNode))
#define setNxtNode(tNode,nNode)	((tNode)->next = (nNode))

/* length a node's data area */
#define nodeLength(node)												\
	 (hsize_t)((node)->next - (node))

/* join two nodes */
#define joinNodes(node,nxtNode)											\
	((node)->next = (nxtNode)->next)


/* source in the mapping of the "normal" string functions. We must use the
	huge functions, because any block can be either huge or far. */
#include <string.h>
#include "../memory/_hstring.h"			/**EXPAND**/

#if _IN_ALLOC == 1			/* far functions? */
#define ___PTR far
#else	/* _IN_ALLOC != 1 */
#define ___PTR huge
#define size_t hsize_t
#endif	/* _IN_ALLOC == 1 */

#endif	/* _IN_ALLOC == 0 */

extern _CLibVar _lpHeapNode _lheap;	/* poi to 1st node of local heap */
extern _CLibVar _fpHeapNode _fheap;	/* poi to 1st node of far heap */

/* return pointer to the data portion of the node */
#define nodeData(node) ((_pHeapData)node + sizeof(_heapNode))

/*
 *	_pad(size, pagesize) will calculate the amount of pad bytes to
 *	be added to size that (size + _pad(size, pagesize)) % pagesize == 0
 *
 *	padNode(size, address) will calculate the amount of pad bytes for
 *	a heap node. Such node must align the data area of the current
 *	node and the control area of the next node to a multiple of
 *	_HEAP_PAGE.
 */
#define _HEAP_PAGE (1u << __HEAP_ALIGN)	/* heap page size */
#if 0				/* for pagesize != (1 << x) */
#define _pad(size,pagesize)	(((size) % (pagesize)) == 0			\
	? 0: (pagesize) - ((size) % (pagesize)))
#else
#define _pad(size,pagesize)										\
	(((pagesize) - (unsigned)(size)) & (pagesize - 1))
#endif
#if _HEAP_ALIGN
#define padNode(size,addr) 										\
	(size) + _pad((size) + FP_OFF(addr) + sizeof(_heapNode), _HEAP_PAGE)
#else
#define padNode(size,addr) (size)
#endif


extern _CLibFunc void huge *_retNULL(void);		/* always return NULL */

/* Debugging references */
#ifdef DEBUG_HEAP
extern int _CLibVar dbgHeapFreeCh;	/* ch that fills free()'ed memory */
extern int _CLibVar dbgHeapAllocCh;	/* ch that fills malloc()'ed memory */
#endif


/*****for testing purpose */
#include "prototyp"

#endif
