/*
 * four-in-a-row game -- cgi version
 * Copyright (c) 2017-2018 Andreas K. Foerster <info@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
For this the browser doesn't need any special capacities,
and it's very safe for the user, but it can cause much traffic.

Environment variables:
REQUEST_METHOD, PATH_INFO, HTTP_ACCEPT_LANGUAGE, SCRIPT_NAME,
QUERY_STRING, CONTENT_LENGTH, REMOTE_USER, REMOTE_IDENT
*/

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include "row4.h"

/* workaround for bcc */
#define SOURCE "https:/" "/akfoerster.de/p/row4/"

/*
  "GET" or "POST"
  GET produces ugly URLs and is easier to manipulate,
  but it's more efficient.
*/
#ifndef METHOD
#define METHOD "GET"
#endif

#ifdef HTML
#define TYPE "text/html"
#define HTMLTAG "<html>"
#else /* XHTML */

#define TYPE "application/xhtml+xml"
#define HTMLTAG "<html xmlns='http:/" "/www.w3.org/1999/xhtml'>"
#endif

#define text(s) write (1, (s), sizeof (s) - 1)
#define print(s) write (1, (s), strlen (s))
#define odd(x) ((x) & 1)
#define digit(x) ((x) ^ 0x30)
#define character(c) do{char ch=(char)(c);write(1,&ch,1);}while(0)

#define STYLE "\
html { color:#EEE; background-color:black; }\n\
\n\
body { text-align:center; font-family:sans-serif; }\n\
\n\
table {\n\
  border:solid thin #EEE;\n\
  margin:auto;\n\
  text-align:center;\n\
  background-color:#9B5523;\n\
  border-collapse:separate;\n\
  border-spacing:0.5em;\n\
}\n\
\n\
input, td { width:2em; height:2em; }\n\
\n\
input, button, td {\n\
  background-color:black;\n\
  color:#EEE;\n\
  font-weight:bold;\n\
}\n\
\n\
a {color:#EEE}\n\
\n\
td { border-radius:1em; }\n\
\n\
td.P1 { background-color:#0C0; }\n\
td.P2 { background-color:#C00; }\n\
\n\
@media (min-height:500px) {\n\
  body { font-size:17pt; }\n\
}\n\
\n\
@media (min-height:1200px) {\n\
  body { font-size:35pt; }\n\
}\n"

enum methods
{ GET, HEAD, POST };

static enum methods method;
static enum languages language;
static unsigned int score[2];
static char query[64];


int main P_ ((void));
static void show_board P_ ((int));
static void winmessage P_ ((int));
static enum languages get_language P_ ((void));
static void save_state P_ ((void));
static void get_score P_ ((void));
static void restore_state P_ ((void));
static int thinking P_ ((void));
static void read_post P_ ((void));
static void get_query_string P_ ((void));
static int stylesheet P_ ((void));
static void value P_ ((unsigned int));


int
main ()
{
  int winner = NONE;
  const char *p;

  p = getenv ("REQUEST_METHOD");
  if (!p)
    {
      text ("execute this with a webserver via CGI/1.1.\a\n");
      return EXIT_FAILURE;
    }

  if (!strcmp ("POST", p))
    method = POST;
  else if (!strcmp ("GET", p))
    method = GET;
  else if (!strcmp ("HEAD", p))
    method = HEAD;

  p = getenv ("PATH_INFO");
  if (p && !strcmp ("/row4.css", p))
    return stylesheet ();

  language = get_language ();
  reset_board ();

  if (POST == method)
    read_post ();
  else				/* GET or HEAD */
    get_query_string ();

  if (*query)
    {
      char *p;
      get_score ();
      p = strstr (query, "n=");

      if (!p)
	winner = thinking ();
      else if (p[2] == '2')
	drop (PLAYER2, 3);
    }

  show_board (winner);

  return EXIT_SUCCESS;
}


static void
show_board (winner)
     int winner;
{
  int x, y;
  char *title, *script, *user;

  title = (deutsch == language) ? "Vier in einer Reihe" : "Four In a Row";
  script = getenv ("SCRIPT_NAME");

  /* CGI head */
  text ("Status: 200 Have Fun\nContent-Type: " TYPE "\n"
	"Cache-Control: no-cache\nVary: Accept-Language\n\n");

  if (HEAD == method)
    return;

  text ("<!DOCTYPE html>\n" HTMLTAG "\n<head>\n<title>");
  print (title);
  text ("</title>\n"
	"<meta name='robots' content='noindex' />\n"
	"<meta name='viewport' content='width=device-width' />\n"
	"<meta name='generator' content='row4 (AKFoerster)' />\n"
	"<link rel='stylesheet' type='text/css' href='");
  print (script);
  text ("/row4.css' />\n</head>\n\n<body>\n<h1>");
  print (title);
  text ("</h1>\n\n");

  if (winner)
    winmessage (winner);

  text ("<form method='" METHOD "' action='");
  print (script);
  text ("'>\n");

  text ("<table>\n<thead>\n<tr>\n");

  for (x = 0; x < 7; ++x)
    {
      text ("<th><input type='submit' name='d' value='");
      character (digit (x + 1));
      character ('\'');

      if (winner || full (x))
	text (" disabled='disabled'");

      text (" /></th>\n");
    }

  text ("</tr>\n</thead>\n\n<tbody>\n");

  for (y = 5; y >= 0; --y)
    {
      text ("<tr>\n");

      for (x = 0; x < 7; ++x)
	{
	  switch (get_player (board[x][y]))
	    {
	    case 1:
	      text ("<td class='P1'>X");
	      break;

	    case 2:
	      text ("<td class='P2'>O");
	      break;

	    default:
	      text ("<td class='P0'>&#160;");
	      /* note: "&nbsp;" would not work with XHTML */
	      break;
	    }

	  text ("</td>\n");
	}

      text ("</tr>");
    }

  text ("\n</tbody>\n</table>\n\n");

  text ("<button name='n' value='");
  character (odd (chips) ? '1' : '2');
  text ("'>");
  if (deutsch == language)
    text ("Neuanfang");
  else
    text ("new start");
  text ("</button>\n");

  text ("<p class='score'>");

  user = getenv ("REMOTE_USER");
  if (!(user && *user))
    user = getenv ("REMOTE_IDENT");

  if (user && *user)
    {
      print (user);
      character ('=');
    }

  text ("X: ");
  value (score[0]);
  text (" / O: ");
  value (score[1]);
  text ("</p>\n\n");

  text ("<input type='hidden' name='p' value='");
  value (score[0]);
  character ('-');
  value (score[1]);
  text ("' />\n");

  /* save the state of the game */
  if (!winner)
    save_state ();

  text ("</form>\n\n<footer><hr />\n<p><a href='" SOURCE "' target='_top'>"
	"Software-Homepage</a></p>\n<p>");

  if (deutsch == language)
    text ("Programmierung: ");
  else
    text ("Programming: ");

  text ("<a href='https://akfoerster.de/' target='_top'\n>"
	"Andreas K. F&#246;rster</a></p>\n</footer>\n\n</body></html>");
}


static void
winmessage (winner)
     int winner;
{
  text ("<h2>");
  character ((winner == PLAYER1) ? 'X' : 'O');

  if (deutsch == language)
    text (" hat gewonnen</h2>\n");
  else
    text (" has won</h2>\n");
}


static enum languages
get_language ()
{
  const char *l;

  l = getenv ("HTTP_ACCEPT_LANGUAGE");

  if (!l)
    return english;

  /* limitation: it doesn't check for preferences */

  if (strstr (l, "de") || strstr (l, "DE"))
    return deutsch;

  return english;
}


static void
save_state ()
{
  int x, y;
  char *p, s[7 * 6];

  /* turn the board into row of letters */
  p = s;
  for (x = 0; x < 7; ++x)
    for (y = 0; y < 6; ++y)
      *p++ = 'A' + board[x][y];

  text ("<input type='hidden' name='s'\n  value='");
  write (1, s, sizeof (s));
  text ("' />\n");
}


static void
get_score ()
{
  char *p;

  p = strstr (query, "p=");
  if (!p)
    return;

  score[0] = (unsigned int) strtoul (p + 2, &p, 10);

  if (!p)
    {
      score[0] = score[1] = 0;
      return;
    }

  score[1] = (unsigned int) strtoul (p + 1, NULL, 10);
}


static void
restore_state ()
{
  int x, y, hole;
  char *p;

  p = strstr (query, "s=");
  if (!p)
    return;

  p += 2;

  /*
     The input must be sanitized; it could be hacked.
     Well, they still can cheat, but it's just a game,
     they're only cheating themselves.
   */

  hole = FALSE;
  for (x = 0; x < 7; ++x, hole = FALSE)
    for (y = 0; y < 6; ++y)
      {
	byte c, player;

	c = *p++;

	/* Incomplete? Don't read over the end. */
	if (!c || c == '&' || c == ';')
	  return;

	c -= 'A';
	player = get_player (c);

	if (player && !hole)
	  {
	    /* If both players own a field, I claim it for me. ;-) */
	    if (player > PLAYER2)
	      player = PLAYER2;

	    board[x][y] = player;
	    --chips;
	    ++filled[x];
	  }
	else
	  {
	    board[x][y] = c & 0x0C;	/* extract threat */
	    hole = TRUE;
	  }
      }
}


static int
thinking ()
{
  char *p;
  int winner = NONE;

  restore_state ();

  p = strstr (query, "d=");
  if (!p)
    return NONE;

  p += 2;
  if (drop (PLAYER1, *p - '1') >= 0)
    {
      if (wincheck ())
	{
	  winner = PLAYER1;
	  --chips;		/* to get the next starter right */
	}
      else if (drop (PLAYER2, compute (PLAYER2)) >= 0 && wincheck ())
	winner = PLAYER2;
    }

  if (winner && score[winner - 1] < UINT_MAX)
    ++score[winner - 1];

  return winner;
}


static void
read_post ()
{
  char *content_length;
  size_t l;

  *query = '\0';
  content_length = getenv ("CONTENT_LENGTH");
  if (!content_length)
    return;

  l = (size_t) strtoul (content_length, NULL, 10);
  if (0 >= l || l >= sizeof (query))
    return;

  read (0, query, l);
  query[l] = '\0';
}


static void
get_query_string ()
{
  char *s;
  size_t l;

  *query = '\0';
  s = getenv ("QUERY_STRING");
  if (!s)
    return;

  l = strlen (s);

  if (l < sizeof (query))
    memcpy (query, s, l + 1);
}


static int
stylesheet ()
{
  /* cachable for 24 hours */
  text ("Cache-Control: max-age=86400\nContent-Type: text/css\n");
  text ("Content-Length: ");
  value (sizeof (STYLE) - 1);
  text ("\n\n");

  if (method != HEAD)
    text (STYLE);

  return EXIT_SUCCESS;
}


/* print unsigned value */
static void
value (v)
     unsigned int v;
{
  size_t l = 0;
  register unsigned int i;
  char *p, buffer[20];

  p = &buffer[sizeof (buffer)];

  i = v;
  do
    {
      ++l;
      --p;
      *p = digit (i % 10);
      i /= 10;
    }
  while (i);

  write (1, p, l);
}


#ifdef __DJGPP__
#undef __STRICT_ANSI__
#include <crt0.h>

char **
__crt0_glob_function (arg)
     char *arg;
{
  (void) arg;
  return NULL;
}

/* no need for environment file */
void
__crt0_load_environment_file (name)
     char *name;
{
  (void) name;
}

void
__crt0_setup_arguments ()
{
}

#endif /* __DJGPP__ */
