/*
 * This file is part of the Zmiy project.
 * Copyright (C) 2013-2015 Mateusz Viste
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 */


#include <dos.h>   /* provides int86() along with the union REGS type */
#include <conio.h> /* outpw() */

#include "timer.h"  /* clock() */

#include "io.h"    /* include self headers for control */


/* OpenWatcom doesn't know outport(), it provides outpw() instead. */
#ifdef __WATCOMC__
#define outport outpw
#endif


void cursor_set(int startscanline, int endscanline) {
  union REGS regs;
  regs.h.ah = 0x01;
  regs.h.ch = startscanline;
  regs.h.cl = endscanline;
  int86(0x10, &regs, &regs);
}


/* waits for a keypress and return it. Returns 0 for extended keystroke, then
   function must be called again to return scan code. */
int getkey(void) {
  union REGS regs;
  int res;
  regs.h.ah = 0x08;
  int86(0x21, &regs, &regs);
  res = regs.h.al;
  if (res == 0) { /* extended key - poll again */
    regs.h.ah = 0x08;
    int86(0x21, &regs, &regs);
    res = regs.h.al | 0x100;
  }
  return(res);
}


/* poll the keyboard, and return the next input key if any, or -1 */
int getkey_ifany(void) {
  union REGS regs;
  regs.h.ah = 0x0B;
  int86(0x21, &regs, &regs);
  if (regs.h.al == 0xFF) {
      return(getkey());
    } else {
      return(-1);
  }
}


/* returns 0 if no VGA has been detected, non-zero otherwise */
int detectvga(void) {
  union REGS regs;
  regs.x.ax = 0x1A00; /* can be used to detect the presence of VGA. */
  int86(0x10, &regs, &regs);
  if (regs.h.al == 0x1A) return(1);  /* VGA supported */
  return(0); /* else it's not vga */
}


/* returns the current video mode */
int getcurvideomode(void) {
  union REGS regs;
  regs.h.ah = 0x0F;
  int86(0x10, &regs, &regs);
  return(regs.h.al);
}


/* switches to a new video mode */
void setvideomode(int mode) {
  union REGS regs;
  regs.h.ah = 0;
  regs.h.al = mode;
  int86(0x10, &regs, &regs);
}


void setvideomode_80(int rows) {
  union REGS regs;
  /* first set up a 80x25 video mode */
  setvideomode(0x03); /* mode 0x03 is 80x25, color */
  if (rows == 50) {
    /* now select a 8x8 font */
    regs.h.ah = 0x11;
    regs.h.al = 0x12;
    regs.h.bl = 0;
    int86(0x10, &regs, &regs);
  }
  /* disable blinking effect (enables the use of high-intensity backgrounds).
   * This doesn't change anything on DOSemu nor VBox, but DOSbox is unable to
   * display high intensity backgrounds otherwise. */
  regs.x.ax = 0x1003;  /* toggle intensity/blinking */
  regs.h.bl = 0;       /* enable intensive colors (1 would enable blinking) */
  regs.h.bh = 0;       /* to avoid problems on some adapters */
  int86(0x10, &regs, &regs);
}


/* if ms is negative, it means 'that many ms since last time' */
void milisleep(int ms, int flag) { /* Turbo C provides a delay() function for  */
  static unsigned long finish = 0; /* this, but I am trying to be "green" here */
  union REGS regs;                 /* and not waste CPU in a dumb busy loop.   */
  unsigned long curtime;

  /* set it to be in the future */
  if (ms < 0) {
    finish -= ms;
  } else {
    timer_read(&curtime);
    finish = (curtime / 1000) + ms;
  }
  for (;;) {
    timer_read(&curtime);
    if (curtime / 1000 >= finish) break;
    switch (flag) {
      case MILISLEEP_APM:  /* APM v1.0+ CPU IDLE */
        regs.x.ax = 0x5305;
        int86(0x15, &regs, &regs);
        break;
      case MILISLEEP_DOS:  /* DOS 2+ 'DOS IDLE INTERRUPT' */
        int86(0x28, &regs, &regs);
        break;
      case MILISLEEP_DPMI:  /* DPMI 'RELEASE VM TIME-SLICE' */
        regs.x.ax = 0x1680;
        int86(0x2F, &regs, &regs);
        break;
      case MILISLEEP_BIOS:  /* BIOS call to wait for 1ms */
        regs.h.ah = 0x86;   /* 0x86 is the BIOS WAIT service */
        regs.x.cx = 0;      /* CX contains the high word */
        regs.x.dx = 1000;   /* DX contains the low word */
        int86(0x15, &regs, &regs);
        break;
    }
  }
}


/* Scroll screen function, uses CGA registers programming.
 *
 * 3D4h (W): Index register.
 *           The value written to this register selects which of the data
 *           registers will be accessed at 3D5h.
 *
 * 3D4h index 0Ch (W): Start Address High Register
 *       Bit 0-5  The upper 6 bits of the address of the start of the display.
 *       The lower 8 bits are in index 0Dh.
 *       Note: this register is Read/Write on the CT82c425/6.
 *
 * 3D4h index 0Dh (W): Start Address Low Register
 *       Bit 0-7  The lower 8 bits of the address of the start of the display.
 *       The upper 6 bits are in index 0Ch.
 *       Note: this register is Read/Write on the CT82c425/6. */
void scrollscreen(int offset) {
  int reg;
  /* write upper 6 bits into start address high register */
  reg = 0x0C | (offset & 0x3F00);
  outport(0x3D4, reg);
  /* write lower 8 bits into start address low register */
  reg = 0x0D | ((offset & 0xFF) << 8);
  outport(0x3D4, reg); /* start address low register (lower 8 bits) */
}
