/**
 *	Copyright (c) 2002-2004 Lukas Lipka, Point Mad. All rights reserved.
 *
 *	FILE: button.c
 *	DESCRIPTION: Basic implementation of the button widget
 *	CONTRIBUTORS:
 *					Lukas Lipka
 *					Julien Etelain
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *	CHANGES:
 *	21.12.2003 -  Removed some fucked up bloat which screwed translucency --lukas
 *	22.01.2004 -  Fixed button sensitivity --lukas
 *  24.01.2004 -  Button color and font from skin file --lukas
 *             -  Otimizations to drawing (theoretically faster) --lukas
 *	19.02.2004 -  New flag implemented - BF_ALIGNLEFT --lukas
*/
#include "kernel.h"
#include "widget.h"
#include "button.h"
#include "grfx.h"

l_ulong	AppVersion	= ULONG_ID(0,9,1,0);
l_char AppName[]	= "Button Widget";
l_uid nUID			= "button";
l_uid NeededLibs[]	= { "widget", "skin", "grfx", "" };

PSkin SkinButtonUp			= 0;
PSkin SkinButtonDown		= 0;
PSkin SkinButtonOver		= 0;
PSkin SkinButtonDisabled	= 0;

/**
 * Current clicked button
 */
//PWidget MasterButton		= 0;

/**
*	NAME: ButtonDraw
*	DESCRIPTION: Default button draw function
*/
_PUBLIC void  ButtonDraw ( PWidget o, p_bitmap buffer, PRect w )
{
	FONT *Font				= default_font;
	l_color FontColor		= COL_3DTEXT;
	l_int OffsetX			= 5;
	l_int OffsetY			= 3;
	l_int Shift				= 0;

	if ( UseSkins )
	{
		PSkin SkinItem = SkinButtonUp;

		if ( BUTTON(o)->State == BS_OVER ) SkinItem = SkinButtonOver;
		if ( BUTTON(o)->Flags & BF_DISABLED ) SkinItem = SkinButtonDisabled;
		if ( BUTTON(o)->State == BS_DOWN )
		{
			SkinItem = SkinButtonDown;
			Shift = 1;
		}

		DrawSkin(buffer, SkinItem, o->Absolute.a.x, o->Absolute.a.y , o->Absolute.b.x, o->Absolute.b.y);

		FontColor 	= SkinItem->FontColor;
		Font		= SkinItem->Font;
		OffsetX		= SkinItem->Left;
		OffsetY		= SkinItem->Top;
	}
	else
	{
		l_color Face 	= COL_3DFACE;
		l_color Light	= COL_3DLIGHT;
		l_color Dark 	= COL_3DDARK;

		if ( BUTTON(o)->State == BS_OVER ) Face = COL_3DOVER;
		if ( BUTTON(o)->State == BS_DOWN )
		{
			Light = COL_3DDARK;
			Dark = COL_3DLIGHT ;
			Shift = 1;
		}

		rectfill(buffer, w->a.x, w->a.y, w->b.x, w->b.y, Face);
		Rect3D(buffer, o->Absolute.a.x, o->Absolute.a.y, o->Absolute.b.x, o->Absolute.b.y, Light, Dark);
	}

	if ( BUTTON(o)->Caption )
	{
		if (BUTTON(o)->Icon)
		{
			if ( BUTTON(o)->Flags & BF_TEXTUNDERICON )
			{
				draw_sprite(buffer, BUTTON(o)->Icon, (o->Absolute.b.x+o->Absolute.a.x)/2 - (BUTTON(o)->Icon->w/2) + Shift, (o->Absolute.a.y+o->Absolute.b.y)/2 + Shift - (BUTTON(o)->Icon->h + OffsetY + text_height(Font))/2);
				textout_centre(buffer, Font, BUTTON(o)->Caption, (o->Absolute.b.x+o->Absolute.a.x)/2+Shift, (o->Absolute.a.y+o->Absolute.b.y)/2 + Shift - (BUTTON(o)->Icon->h + OffsetY + text_height(Font))/2 + BUTTON(o)->Icon->h + OffsetY, FontColor);
			}
			else if (BUTTON(o)->Flags & BF_ALIGNLEFT)
			{
				draw_sprite(buffer, BUTTON(o)->Icon, o->Absolute.a.x + OffsetX + Shift, (o->Absolute.b.y + o->Absolute.a.y - BUTTON(o)->Icon->h)/2 + Shift);
				textout(buffer, Font, BUTTON(o)->Caption, o->Absolute.a.x+ OffsetX + Shift + BUTTON(o)->Icon->w + OffsetX, (o->Absolute.a.y+o->Absolute.b.y-text_height(Font))/2 + Shift, FontColor);
			}
			else
			{
				draw_sprite(buffer, BUTTON(o)->Icon, (o->Absolute.a.x+o->Absolute.b.x)/2 + Shift - (BUTTON(o)->Icon->w + OffsetX + text_length(Font, BUTTON(o)->Caption))/2, (o->Absolute.b.y + o->Absolute.a.y - BUTTON(o)->Icon->h)/2 + Shift);
				textout(buffer, Font, BUTTON(o)->Caption, (o->Absolute.a.x+o->Absolute.b.x)/2 + Shift - (BUTTON(o)->Icon->w + OffsetX + text_length(Font, BUTTON(o)->Caption))/2 + BUTTON(o)->Icon->w + OffsetX, (o->Absolute.a.y+o->Absolute.b.y-text_height(Font))/2 + Shift, FontColor);
			}
		}
		else
			textout_centre(buffer, Font, BUTTON(o)->Caption, (o->Absolute.b.x+o->Absolute.a.x)/2 + Shift, (o->Absolute.a.y+o->Absolute.b.y-text_height(Font))/2 + Shift, FontColor);
	}
	else if ( !BUTTON(o)->Caption && BUTTON(o)->Icon )
	{
		draw_sprite(buffer, BUTTON(o)->Icon, (o->Absolute.b.x+o->Absolute.a.x)/2-BUTTON(o)->Icon->w/2+Shift, (o->Absolute.b.y+o->Absolute.a.y)/2-BUTTON(o)->Icon->h/2+Shift);
	}
}

/**
*	NAME: ButtonEventHandler
*	DESCRIPTION: Default event handler for button
*/
_PUBLIC l_bool  ButtonEventHandler ( PWidget o, PEvent Ev )
{
	if (Ev->Type == EV_MOUSE)
	{
		if ((BUTTON(o)->Flags & BF_DISABLED))
			return true;

		if (Ev->Message == WEvMouseEnter)
		{
			BUTTON(o)->State = BS_OVER;
			WidgetDraw(o, NULL);
			return true;
		}

		if (Ev->Message == WEvMouseLeft)
		{
			BUTTON(o)->State = 0;
			WidgetDraw(o, NULL);
			return true;
		}

		if (Ev->Message == WEvMouseLDown)
		{
			BUTTON(o)->State = BS_DOWN;
			WidgetDraw(o, NULL);
			return true;
		}

		if (Ev->Message == WEvMouseLUp && BUTTON(o)->State == BS_DOWN)
		{
			BUTTON(o)->State = BS_OVER;
			WidgetDraw(o, NULL);
			if ( BUTTON(o)->Message )
				WidgetSendSEvent(o, EV_MESSAGE, BUTTON(o)->Message, NULL);
			return true;
		}
	}

	return false;
}

void ButtonFreeEx ( PWidget o )
{
	if ( BUTTON(o)->Icon )
		destroy_bitmap(BUTTON(o)->Icon);

	if ( BUTTON(o)->Caption )
		free(BUTTON(o)->Caption);
}

/**
*	NAME: CreateButton
*	DESCRIPTION: Creates a new button
*/
_PUBLIC PButton  CreateButton ( PApplication App, TRect r, l_text Caption, l_ulong Message )
{
	PButton b = malloc(sizeof(TButton));

	if (!b) return NULL;

	memset(b, 0, sizeof(TButton));

	if ( Caption )
		b->Caption = TextDup(Caption);

	b->Message = Message;

	IntialiseWidget(App, WIDGET(b), r, "BUTTON");

	WIDGET(b)->Draw = &ButtonDraw;
	WIDGET(b)->EventHandler = &ButtonEventHandler;
	WIDGET(b)->FreeEx = &ButtonFreeEx;

	return b;
}

l_bool LibMain ( l_text Args )
{
	SkinButtonUp		= GetSkinItem("Button");
	SkinButtonDown		= GetSkinItem("ButtonClick");
	SkinButtonOver		= GetSkinItem("ButtonOver");
	SkinButtonDisabled	= GetSkinItem("ButtonDisabled");

	APPEXPORT(CreateButton);

	return true;
}

void Close (void)
{

}
