// This program was written by Jrgen Hoffmann in the year 2009
// and compiled under Borland C++ Version 3.1
// using the Small model and the "Compile via assembler" option
//
#include <dos.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <conio.h>

#define MAXBUFS     8  	                /* maximum number of Ethernet buffers */
#define BPMASK      7
#define BUFSIZE     1520

#define DOS         0x21
#define GETVECT     0x35

#define INT_FIRST 0x60
#define INT_LAST  0x80
#define PD_DRIVER_INFO	0x1ff
#define PD_ACCESS 	0x200
#define PD_RELEASE	0x300
#define PD_SEND 	0x400
#define PD_GET_ADDRESS	0x600
#define PD_RESET        0x700
#define PD_SET_MODE	0x1400
#define CARRY 		1	        /* carry bit in flags register */

#if defined __TINY__
#define MEMORY_MODEL	"ti"
#elif defined __SMALL__
#define MEMORY_MODEL	"sm"
#elif defined __MEDIUM__
#define MEMORY_MODEL	"me"
#elif defined __COMPACT__
#define MEMORY_MODEL	"co"
#elif defined __LARGE__
#define MEMORY_MODEL	"la"
#else
#define MEMORY_MODEL	"??"
#endif

typedef unsigned short word;            /* 16 bits */
typedef unsigned char  byte;            /*  8 bits */

char *version  = "V1.4";

// forwards and externs
char *eadr2asc(unsigned char *p, int mode);

byte  brief       = 0;
byte  filter_mode = 0;
byte  filter_adr1[6];
byte  filter_adr2[6];
word  filter_type = 0;
word  filter_offs = 0;
word  filter_plen = 0;
byte  filter_patt[8];

byte  capabilities;
word  int_first   = INT_FIRST;
word  int_last    = INT_LAST;
word  pkt_interrupt;
word  pkt_type    = 0XFFFF;                /* any type */
word  pkt_handle;
byte  eth_addr[6] ;
char *pkt_line    = "PKT DRVR";
byte  old_driver  = 0;
byte  write_file  = 0;
FILE *outfil;
char  filnam[82]  = ".\\ETHDUMP.DMP";

word  bad_frame   = 0X003C;
byte  bad_frame1[14] = { 0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0 };
char  bad_frame2[34] = "Buffer overflow   packet lost    ";
byte  bad_frame3[12] = { 0,0,0,0, 0,0,0,0, 0,0,0,0 };

typedef struct {
  word  pkt_num;
  word  pkt_len;
  byte  pkt_data[BUFSIZE];
  } pkt_buffer;

word  aux_seq           = 0;
word  total_lost        = 0;
word  pkt_last          = 0;
word  pkt_get           = 0;
word  pkt_put           = 0;
word  pkt_seq           = 0;
word  num_bufs          = MAXBUFS;
word  bpmask            = BPMASK;
pkt_buffer *pb;
//pkt_buffer pb[MAXBUFS];

// compiled with Borland C++  3.1
// with "Compile via assembler" option
// in Options | Compiler | Code generation ...
void far pkt_callback(void) {
  asm {
    pop  di; // compensate for compiler generated PUSH DI / POP DI
    push ds;                   // save driver's data segment
    mov  di,DGROUP;            // NOT in huge model !!!
    mov  ds,di;                // set C's data segment
    }
  disable();
  if(_AX) {
    pb[pkt_put & bpmask].pkt_len |= 0X8000;
    pkt_put++;
    }
  else {
    if(pb[pkt_put & bpmask].pkt_len || _CX >= BUFSIZE) {
      pkt_seq++;
      _ES = 0;
      _DI = 0;
      }
    else {
      pb[pkt_put & bpmask].pkt_len = _CX;
      pb[pkt_put & bpmask].pkt_num = pkt_seq;
      pkt_seq++;
      _ES = FP_SEG(pb[pkt_put & bpmask].pkt_data);
      _DI = FP_OFF(pb[pkt_put & bpmask].pkt_data);
      }
    }
  enable();
  asm {
    pop  ds; // restore driver's data segment
    push di; // compensate for compiler generated PUSH DI / POP DI
    }
  }

int pkt_init(void) {
  struct REGPACK regs;
  char far *temp;
  int pd_type;	                /* packet driver type */
  int pd_class;
  for (pkt_interrupt = int_first; pkt_interrupt <= int_last; ++pkt_interrupt ) {
    temp = (char far *)getvect( pkt_interrupt );
    if (!_fmemcmp( &(temp[3]), pkt_line, strlen( pkt_line ))) break;
    }
  if ( pkt_interrupt > int_last ) {
    fprintf(stderr,"NO PACKET DRIVER FOUND ");
    if(int_first==int_last) fprintf(stderr,"AT INTERRUPT 0x%02X\n",int_first);
    else fprintf(stderr,"IN RANGE 0x%02X .. 0x%02X\n",int_first,int_last);
    exit(2);
    }

  /* lets find out about the driver */
  regs.r_ax = PD_DRIVER_INFO;
  intr( pkt_interrupt, &regs );
  capabilities = regs.r_ax & 7;

  /* handle old versions, assume a class and just keep trying */
  if (regs.r_flags & CARRY ) {
    for ( pd_class = 0; pd_class < 19; ++pd_class ) {
      for (pd_type = 1; pd_type < 128; ++pd_type ) {
	regs.r_ax = PD_ACCESS | pd_class;  /* ETH, SLIP */
	regs.r_bx = pd_type;		     /* type */
	regs.r_dx = 0;		     /* if number */
	regs.r_cx = 0;                     // sizeof( pkt_type);
	regs.r_ds = FP_SEG( &pkt_type );
	regs.r_si = FP_OFF( &pkt_type );
	regs.r_es = FP_SEG( pkt_callback );
	regs.r_di = FP_OFF( pkt_callback );
	intr( pkt_interrupt, &regs );
	if ( ! (regs.r_flags & CARRY) ) break;
	}

      if (pd_type < 128 ) {
	/* get ethernet address */
	regs.r_ax = PD_GET_ADDRESS;
	regs.r_bx = regs.r_ax;	/* handle */
	regs.r_es = FP_SEG( eth_addr );
	regs.r_di = FP_OFF( eth_addr );
	regs.r_cx = sizeof( eth_addr );
	intr( pkt_interrupt, &regs );
	/* we have found a working type, so kill it */
	regs.r_ax = PD_RELEASE;
	intr( pkt_interrupt, &regs );
	old_driver++;
	fprintf(stderr,"Found OLD type of packet driver\n");
	break;
	}

      if ((pd_type == 128 ) && (pd_class > 18)) {
	fprintf(stderr,"ERROR initializing packet driver\n\r");
	return( 1 );
	}
      }
    }
  else {
    pd_type = regs.r_dx;
    pd_class = regs.r_cx >> 8;
    /* get ethernet address */
    regs.r_ax = PD_GET_ADDRESS;
    regs.r_bx = 0;
    regs.r_es = FP_SEG( eth_addr );
    regs.r_di = FP_OFF( eth_addr );
    regs.r_cx = sizeof( eth_addr );
    intr( pkt_interrupt, &regs );
    fprintf(stderr,"Found packet driver with basic ");
    switch (capabilities) {
      case 2:
      case 6: fprintf(stderr,"and extended ");
	      if(capabilities==2) break;
      case 5: fprintf(stderr,"and high-performance ");
      }
    fprintf(stderr,"functions\n");
    }
  fprintf(stderr,"Vector=%02X class=%d type=%d ethernet address=%s\n",
	    pkt_interrupt, pd_class, pd_type, eadr2asc(eth_addr,0));

  regs.r_ax = PD_ACCESS | pd_class;
  regs.r_bx = pd_type;                // any type
  regs.r_dx = 0;		      // if number
  regs.r_cx = 0;                      // sizeof( pkt_type );
  regs.r_ds = FP_SEG( &pkt_type );
  regs.r_si = FP_OFF( &pkt_type );
  regs.r_es = FP_SEG( pkt_callback );
  regs.r_di = FP_OFF( pkt_callback );
  intr( pkt_interrupt, &regs );
  if(regs.r_flags & CARRY) {
    fprintf(stderr,"ERROR #%02X accessing packet driver\n",(regs.r_dx>>8));
    return( 1 );
    }
  pkt_handle = regs.r_ax;
  return( 0 );
  }

void pkt_setmode(int mode) {
  struct REGPACK regs;
  // get ethernet address
  regs.r_ax = PD_SET_MODE;
  regs.r_bx = pkt_handle;
  regs.r_cx = mode;
  intr( pkt_interrupt, &regs );
  if(regs.r_flags & CARRY) fprintf(stderr,"WARNING cannot set driver into mode: %d\n",mode);
  }

void pkt_release(void) {
  struct REGPACK regs;
  int error;
  regs.r_ax = PD_RELEASE;
  regs.r_bx = pkt_handle;
  intr( pkt_interrupt, &regs );
  if (regs.r_flags & CARRY ) fprintf(stderr,"ERROR releasing packet driver\n");
  return;
  }

int pkt_send( char *buffer, int length ) {
  struct REGPACK regs;
  int retries;

  retries = 5;
  while (retries--) {
    regs.r_ax = PD_SEND;
    regs.r_ds = FP_SEG( buffer );
    regs.r_si = FP_OFF( buffer );
    regs.r_cx = length;
    intr( pkt_interrupt, &regs );
    if ( regs.r_flags & CARRY ) continue;
    return( 0 );
    }
  return( 1 );
  }


//---------------------------------------------------------------

char *eadr2asc(unsigned char *p, int mode) {
  static char outbuf[20];
  char *p2;
  int i;
  for(i=0, p2=outbuf; i<6; i++, p++) {
    if(mode) {
      sprintf(p2,"%02X",*p);
      p2 += 2;
      if(i==2) *p2++ = ':';
      }
    else {
      sprintf(p2,"%02X:",*p);
      p2 += 3;
      }
    }
  if(!mode) p2--;
  *p2 = '\0';
  return(outbuf);
  }

word htoi(char *p, int count) {
  int i;
  for(i=0; *p && count; p++,count--) {
    if(!isxdigit(*p)) break;
    i <<= 4;
    if(isdigit(*p)) i |= (*p&0x0F);
    else i |= ((*p&0X0F) + 9);
    }
  return(i);
  }

void get_eth_addr(char *p, byte *d) {
  int i;
  for(i=0; *p && i < 6; i++) {
    *d++ = htoi(p,2);
    p += 2;
    if((*p==':')||(*p=='-')) p++;
    }
  }

word swap(word arg) {
  return((arg & 0xFF00) >> 8) | ((arg & 0xFF) << 8);
  }

void set_type_filter(char *p) {
  byte flag;
  flag = 0;
  if(*p=='+')      { flag=0X80; p++; }
  else if(*p=='-') { flag=0XC0; p++; }
  filter_type = swap(htoi(p,4));
  if(!flag) flag = 0X40;
  if(filter_type) filter_mode |= flag;
  }

void set_filter_pattern(char *p) {
  int i;
  if((*p=='0')&&(toupper(p[1])=='X')) i = htoi(&p[2],2);
  else i = atoi(p);
  while(*p&&(*p!=':')) p++;
  if(*p==':'&&i>0&&i<1500) {
    for(p++,filter_plen=0; *p&&filter_plen<8; ) {
      filter_patt[filter_plen++] = htoi(p,2);
      p += 2;
      }
    if(filter_plen) {
      filter_offs = i;
      filter_mode |= 0X20;
      }
    }
  }

void set_buffer_size(char *p) {
  word i,j;
  i=atoi(p);
  if(i) {
    if((i>3)&&(i<6)) {
      for(j=1; i; i--) j <<= 1;
      num_bufs = j;
      bpmask = j - 1;
      }
    }
  }

//---------------------------------------------------------------

void seq_check(word pkt_num) {
  int i;
  if(pkt_num>(++pkt_last)) {
    printf("%6d  %-60s\n\n",pkt_num-pkt_last,"packet(s) lost");
    total_lost += pkt_num-pkt_last;
    if(!filter_mode && outfil) for(i=pkt_last; i<pkt_num; i++) {
      fwrite(&i,sizeof(byte),2,outfil);
      fwrite(&bad_frame,sizeof(byte),62,outfil);
      }
    }
  pkt_last = pkt_num;
  }

int filter_check(pkt_buffer *pb) {
  int res;
  if(!filter_mode) return(1);
  switch (filter_mode&0XC0) {
    case 0X40: if(memcmp(&pb->pkt_data[12],(byte *)&filter_type,2)!=0) return(0); else break;
    case 0X80: if(memcmp(&pb->pkt_data[12],(byte *)&filter_type,2)<0)  return(0); else break;
    case 0XC0: if(memcmp(&pb->pkt_data[12],(byte *)&filter_type,2)>0)  return(0); else break;
    }
  if(filter_mode&0X20) if(memcmp(&pb->pkt_data[filter_offs],filter_patt,filter_plen)) return(0);
  if(!(filter_mode&0X0F)) return(1);
  switch (filter_mode&3) {
    case 0: if((res=memcmp(pb->pkt_data,filter_adr1,6))!=0)
	    res = memcmp(&pb->pkt_data[6],filter_adr1,6);      break;
    case 1: res = memcmp(&pb->pkt_data[6],filter_adr2,6);      break;
    case 2: res = memcmp(pb->pkt_data,filter_adr1,6);          break;
    case 3: if((res=(memcmp(pb->pkt_data,filter_adr1,6)
		   || memcmp(&pb->pkt_data[6],filter_adr2,6)))!=0)
		res=(memcmp(pb->pkt_data,filter_adr2,6)
		   || memcmp(&pb->pkt_data[6],filter_adr1,6)); break;
    }
  if(filter_mode&8) return(res);
  else  return(!res);
  }

void process_packet(pkt_buffer *pb) {
  int i,j,pl;
  seq_check(pb->pkt_num);
  if(filter_check(pb)) {
    pl = pb->pkt_len & 0X7FFF;
    printf("%4d   DST=%s",pb->pkt_num,eadr2asc(pb->pkt_data,1));
    printf("  SRC=%s   TYP=%02X%02X  LEN=%-4d%8d\n", eadr2asc(&pb->pkt_data[6],1),
	   pb->pkt_data[12],pb->pkt_data[13],pl,(pkt_get&bpmask));
    if(!brief) {
      for(i=14; i<pl; i+=16) {
	printf("%04X  ",i);
	for(j=0; j<16; j++)
	  if((i+j)<pl) printf(" %02X",pb->pkt_data[i+j]);
	  else printf("   ");
	printf("  ");
	for(j=0; j<16; j++)
	  if((i+j)<pl) printf("%c",(pb->pkt_data[i+j]<32)?'':pb->pkt_data[i+j]);
	printf("\n");
	}
      printf("\n");
      }
    if(outfil) {
      if(filter_mode) {
	fwrite(&aux_seq,sizeof(byte),2,outfil);
	aux_seq++;
	}
      else fwrite(&pb->pkt_num,sizeof(byte),2,outfil);
      fwrite(&pl,sizeof(byte),2,outfil);
      fwrite(pb->pkt_data,sizeof(byte),pl,outfil);
      }
    }
  }

void help_text(void) {
  printf("\n\n\n");
  printf("  ETHDUMP %s%s by Jrgen Hoffmann (2009) j_hoff@hrz1.hrz.tu-darmstadt.de\n\n",version,MEMORY_MODEL);
  printf("  usage: ethdump [ options ]\n\n");
  printf("  valid options:\n");
  printf("    /I<int> define packet driver interrupt  (default: automatic)\n");
  printf("    /A<adr> only packets with source OR destination equal to <adr>\n");
  printf("    /D<dst> only packets with destination equal to <dst>\n");
  printf("    /S<src> only packets with source equal to <src>\n");
  printf("            If /S<src> and /D<dst> are both specified:\n");
  printf("            only packets from <src> to <dst> OR vice versa\n");
  printf("    /N      (negate) all packets BUT those specified by /A /D /S\n");
  printf("    /T[+|-]<typ>  only pakets [(+)above|(-)below or] equal <typ>\n");
  printf("    /P<off>:<pat> only packets matching pattern <pat> at offset <off>\n");
  printf("            <pat> is a string of at most 16 hexadecimal digits\n");
  printf("            <off> (22 or 0x16) counts from begin of ethernet frame\n");
  printf("    /F[fil] write data to file \"fil\" (default: .\\ETHDUMP.DMP)\n");
  printf("            file format is compatible with ETHVIEW and ETHSHOW\n");
  printf("    /B      (brief) show headers only\n");
  printf("    /Z<n>   allocate 16(n=4) or 32(n=5) packet buffers (default: 8)\n");
  printf("    /H /?   print this help text\n\n");
  }


void main(int argc, char *argv[]) {
  int  i;
  char *p;

  for(i = 1; i < argc; i++)
    if(*argv[i]=='-' || *argv[i]=='/') {
      p = argv[i];
      switch(toupper(p[1])) {
	case 'B': brief++;                                          break;
	case 'I': if(isdigit(p[2])) {
		    if(toupper(p[3])=='X') int_first = htoi(&p[4],2);
		    else int_first = atoi(&p[2]);
		    int_last = int_first;
		    }                                               break;
	case 'F': if(p[2]) strncpy(filnam,&p[2],80); write_file++;  break;
	case 'N': filter_mode|= 8;                                  break;
	case 'A': get_eth_addr(&p[2],filter_adr1); filter_mode|= 4; break;
	case 'D': get_eth_addr(&p[2],filter_adr1); filter_mode|= 2; break;
	case 'S': get_eth_addr(&p[2],filter_adr2); filter_mode|= 1; break;
	case 'T': set_type_filter(&p[2]);                           break;
	case 'P': set_filter_pattern(&p[2]);                        break;
	case 'Z': set_buffer_size(&p[2]);                           break;
	case 'H':
	case '?': help_text(); exit(1);
	} /* switch */
      } /* if */
  if(!(filter_mode&7)) filter_mode &= 0XF0;
  pkt_init();
  pkt_setmode(6);
  pb=calloc(num_bufs,sizeof(pkt_buffer));
  if(pb==NULL) {
    fprintf(stderr,"ERROR: cannot allocate enough buffer space\n");
    exit(2);
    }
  else if(num_bufs>MAXBUFS) fprintf(stderr,"Allocated %d packet buffers\n",num_bufs);
  if(filter_mode) {
    fprintf(stderr,"Filtering packets ");
    switch(filter_mode&0XC0) {
      case 0X40: fprintf(stderr,"of type: %04X\n",swap(filter_type));             break;
      case 0X80: fprintf(stderr,"equal or above type: %04X\n",swap(filter_type)); break;
      case 0XC0: fprintf(stderr,"equal or below type: %04X\n",swap(filter_type)); break;
      }
    if(filter_mode&0x20) {
      if(filter_mode&0XC0) fprintf(stderr,"AND ");
      fprintf(stderr,"matching at offset: %d the pattern: ",filter_offs,eadr2asc(filter_adr1,0));
      for(i=0; i<filter_plen; i++) fprintf(stderr,"%02X",filter_patt[i]);
      fprintf(stderr,"\n");
      }
    if(filter_mode&0X0F) {
      if(filter_mode&0XC0) fprintf(stderr,"AND ");
      if(filter_mode&8) fprintf(stderr,"NOT ");
      switch (filter_mode&3) {
	case 0: fprintf(stderr,"from OR to address: %s\n",eadr2asc(filter_adr1,1)); break;
	case 1: fprintf(stderr,"from address: %s\n",eadr2asc(filter_adr2,1));       break;
	case 2: fprintf(stderr,"to address: %s\n",eadr2asc(filter_adr1,1));         break;
	case 3: fprintf(stderr,"from: %s",eadr2asc(filter_adr1,1));
		fprintf(stderr," to: %s OR vice versa\n",eadr2asc(filter_adr2,1));          break;
	}
      }
    }
  if(write_file) {
    if(NULL==(outfil=fopen(filnam,"wb")))
      fprintf(stderr,"ERROR: cannot create file: %s\n",filnam);
      else fprintf(stderr,"Writing data to file: %s\n",filnam);
    }
  fprintf(stderr,"\n");

  do {
    fprintf(stderr," %d packet%c so far",pkt_seq,((pkt_seq==1)?' ':'s'));
    if(total_lost) fprintf(stderr," (%d%% lost)",(total_lost*100)/(pkt_seq));
    fprintf(stderr,"   Press any key to terminate ...\r");

    if(pb[pkt_get & bpmask].pkt_len&0x8000) {
      process_packet(&pb[pkt_get & bpmask]);
      pb[pkt_get & bpmask].pkt_len = 0;
      pkt_get++;
      if(pkt_get >= pkt_put) {
	pkt_get &= bpmask;
	pkt_put &= bpmask;
	}
      }
    else seq_check(pkt_seq);
    } while(!kbhit());
  getch();
  fprintf(stderr,"%-75s\n","");

  if(outfil) fclose(outfil);
  pkt_setmode(3);
  pkt_release();
  }



