/*
 * picoSNTP is an SNTP client for DOS, built on the picotcp4dos library.
 * Must be compiled using the Watcom C compiler for proper timezone support.
 *
 * Copyright (C) 2015-2016 Mateusz Viste ; http://picosntp.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <dos.h>
#include <stdio.h>
#include <stdlib.h> /* getenv() */
#include <string.h>
#include <time.h>

#include <picotcp.h>

#include "dostime.h"

/* uncomment the line below for debug */
/* #define DEBUG */

#define PVER "0.9.1"
#define PDATE "2015-2016"

#define FLAG_QUIET    1
#define FLAG_VERBOSE  2
#define FLAG_SIMULATE 4


int sntp_ready = 0;

static void cb_sntp(pico_err_t status) {
  if (status == 0) {
    sntp_ready = 1;
  } else {
    sntp_ready = -1;
  }
}


/* returns the absolute time difference between two time_t values, in seconds */
static time_t timediff(time_t t1, time_t t2) {
  if (t1 > t2) return(t1 - t2);
  return(t2 - t1);
}


static int do_work(char *sntpsrv, int cmdflags, int synchdiff) {
  int dos_year, dos_month, dos_day, dos_hour, dos_min, dos_sec;
  time_t starttime, ntptime, dostime;
  struct pico_timeval tv;
  struct tm *ntp_tm, dos_tm;

  if (pico_sntp_sync(sntpsrv, &cb_sntp) < 0) {
    printf("picoSNTP error: pico_sntp_sync() failure\n");
    return(1);
  }

  /* wait for SNTP synchronization to happen */
  starttime = time(NULL);
  while (sntp_ready == 0) {
    if (time(NULL) - starttime > 2) sntp_ready = -1;
    int86(0x28, NULL, NULL); /* DOS 2+ IDLE INTERRUPT */
    pico_stack_tick();
  }
  if ((sntp_ready < 0) || (pico_sntp_gettimeofday(&tv) < 0)) {
    printf("picoSNTP error: SNTP synchronization failed\n");
    return(1);
  }

  /* get current DOS time */
  dostime_get(&dos_year, &dos_month, &dos_day, &dos_hour, &dos_min, &dos_sec);

  /* convert time to local time zone */
  ntptime = tv.tv_sec;
  ntp_tm = localtime(&ntptime);
  if (ntp_tm == NULL) {
    printf("picoSNTP error: localtime() failure\n");
    return(1);
  }

  /* print out verbose data, if verbosity enabled */
  if ((cmdflags & FLAG_VERBOSE) != 0) {
    printf("NTP time: %d-%02d-%02d %02d:%02d:%02d\n"
           "DOS time: %d-%02d-%02d %02d:%02d:%02d\n",
           1900+ntp_tm->tm_year, 1+ntp_tm->tm_mon, ntp_tm->tm_mday, ntp_tm->tm_hour, ntp_tm->tm_min, ntp_tm->tm_sec,
           dos_year, dos_month, dos_day, dos_hour, dos_min, dos_sec);
  }

  /* The mktime() function converts a broken-down time structure, expressed
   * as local time, to calendar time representation. The function ignores
   * the values supplied by the caller in the tm_wday and tm_yday fields.
   * The value specified in the tm_isdst field informs mktime() whether or
   * not daylight saving time (DST) is in effect for the time supplied in
   * the tm structure: a positive value means DST is in effect; zero means
   * that DST is not in effect; and a negative value means that mktime()
   * should (use timezone information to) attempt to determine whether DST
   * is in effect at the specified time. */
  dos_tm.tm_year = dos_year - 1900;
  dos_tm.tm_mon = dos_month - 1;
  dos_tm.tm_mday = dos_day;
  dos_tm.tm_hour = dos_hour;
  dos_tm.tm_min = dos_min;
  dos_tm.tm_sec = dos_sec;
  dos_tm.tm_isdst = -1;
  dostime = mktime(&dos_tm);

  /* synch the CMOS clock if it deviates too much */
  if (timediff(dostime, ntptime) > synchdiff) {
    if ((cmdflags & FLAG_SIMULATE) == 0) dostime_set(1900+ntp_tm->tm_year, 1+ntp_tm->tm_mon, ntp_tm->tm_mday, ntp_tm->tm_hour, ntp_tm->tm_min, ntp_tm->tm_sec);
    if ((cmdflags & FLAG_QUIET) == 0) printf("picoSNTP: time synchronized (%d-%02d-%02d %02d:%02d:%02d -> %d-%02d-%02d %02d:%02d:%02d)\n", dos_year, dos_month, dos_day, dos_hour, dos_min, dos_sec, 1900+ntp_tm->tm_year, 1+ntp_tm->tm_mon, ntp_tm->tm_mday, ntp_tm->tm_hour, ntp_tm->tm_min, ntp_tm->tm_sec);
  } else {
    if ((cmdflags & FLAG_QUIET) == 0) printf("picoSNTP: local time within acceptable tolerance, no synchronization needed.\n");
  }

  return(0);
}


/* parses arguments and returns a pointer to the server string on success,
 * NULL otherwise */
static char *parseargv(int argc, char **argv, int *synchdiff, int *cmdflags) {
  int i;
  char *ntpsrv = NULL;
  /* load default values */
  *synchdiff = 2;
  *cmdflags = 0;
  /* iterate on arguments */
  for (i = 1; i < argc; i++) {
    #ifdef DEBUG
    printf("analyzing arg #%d: %s\n", i, argv[i]);
    #endif
    /* if not a switch, it must be a server, if not provided yet */
    if ((argv[i][0] != '/') && (argv[i][0] != '-') && (ntpsrv == NULL)) {
      ntpsrv = argv[i];
      continue;
    }
    /* if not a switch of some kind, then quit */
    if (argv[i][0] != '/') return(NULL);
    /* look up what switch we are talking about */
    if ((argv[i][1] == 'q') || (argv[i][1] == 'Q')) {
      *cmdflags |= FLAG_QUIET;
      continue;
    }
    if ((argv[i][1] == 'v') || (argv[i][1] == 'V')) {
      *cmdflags |= FLAG_VERBOSE;
      continue;
    }
    if ((argv[i][1] == 's') || (argv[i][1] == 'S')) {
      *cmdflags |= FLAG_SIMULATE;
      continue;
    }
    if ((argv[i][1] == 'd') || (argv[i][1] == 'D')) {
      if ((argv[i][2] > '0') && (argv[i][2] <= '9') && (argv[i][3] == 0)) {
        *synchdiff = argv[i][2] - '0';
        continue;
      }
    }
    /* nothing matched, so it is an invalid option */
    return(NULL);
  }
  return(ntpsrv);
}


int main(int argc, char **argv) {
  int res;
  struct pico_device picodev;
  char *sntpsrv;
  int synchdiff;
  int cmdflags;

  /* arguments parsing */
  sntpsrv = parseargv(argc, argv, &synchdiff, &cmdflags);

  /* if arguments are invalid, display the help screen */
  if (sntpsrv == NULL) {
    printf("picoSNTP v" PVER " - SNTP client for DOS, based on the picoTCP library\n"
           "Copyright (C) " PDATE " Mateusz Viste ; http://picosntp.sourceforge.net\n"
           "\n"
           "usage: picosntp ntpsrv [/Q] [/V] [/S] [/Dx]\n"
           "\n"
           "where:  ntpsrv  NTP server's address (either IP or hostname)\n"
           "        /Q      quiet mode (acts silently, unless an error happen)\n"
           "        /V      verbose mode (displays additional information)\n"
           "        /S      simulation mode (no clock update)\n"
           "        /Dx     synch if time diff is more than x seconds (1..9, default 2)\n"
           "\n"
           "This application is linked to picoTCP for DOS version %s.\n", picover());
    return(1);
  }

  /* I expect the TZ variable to be set */
  if (getenv("TZ") == NULL) {
    printf("picoSNTP error: TZ variable not set. See picosntp.txt for details.\n");
    return(1);
  }

  #ifdef DEBUG
  printf("picoinit()\n");
  #endif
  if (picoinit(&picodev, 0) != 0) {
    printf("picoSNTP error: picoinit() failure\n");
    return(1);
  }
  #ifdef DEBUG
  printf("picoinit() succeeded\n");
  sleep(1);
  #endif

  res = do_work(sntpsrv, cmdflags, synchdiff);

  #ifdef DEBUG
  printf("picoquit()\n");
  #endif
  picoquit(&picodev);
  #ifdef DEBUG
  printf("picoquit() succeeded\n");
  sleep(1);
  #endif
  return(res);
}
