//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: FFMPEG file handling (loadlist/open/close/read/seek)

//#define MPXPLAY_USE_DEBUGF 1
#define MPXPLAY_DEBUGOUT_SEEK NULL // stdout
#define MPXPLAY_DEBUGOUT_READ NULL // stdout
#define MPXPLAY_DEBUGOUT_FILEOUT NULL // stderr

#include "ffmpgdec.h"

#ifdef MPXPLAY_LINK_INFILE_FF_MPEG

#include "mpxinbuf.h"
#include <control/control.h>
#include <diskdriv/diskdriv.h>
#include <display/display.h>

#define MPXPLAY_OUTFILE_PACKETCOUNT_INITIALLIMIT 100  // if packet count is less, we can reopen the file (clearing the wrote data)
#define MPXPLAY_OUTFILE_ASYNC_FRAME_LIMIT 25          // drop frame if dts is less

#define MPXPLAY_CONFIG_VIDEO_EXTRSTREAM_SUBDIRS_SEPARATOR ';'

extern unsigned long mpxplay_programcontrol, mpxplay_config_videoplayer_control;
extern char mpxplay_config_video_extstream_subdirs[MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM][MPXINI_MAX_CHARDATA_LEN];

static const char *ffmpeg_supported_extensions_subtitle[] = {
"AQT",  // aqtitledec.c
"ASS",  // assdec.c
"JACO", // jacosubdec.c
"LRC",  // lrcdec.c
"MPL2", // mpl2dec.c
"PJS",  // pjsdec.c
"RT",   // realtextdec.c
"SMI",  // smidec.c
"SAMI", // smidec.c
"SCC",  // sccdec.c
"SRT",  // srtdec.c
"STL",  // stldec.c
"SUB",  // mpsubdec.c / subviewer1dec.c / subviewerdec.c
"SUP",  // supdec.c
"TED",  // tedcaptionsdec.c
"TXT",  // vplayerdec.c
"VTT",  // webvttdec.c
NULL
};

static const char *ffmpeg_supported_extensions_audiostreams[] = {
"AAC", "AC3", "AIF", "APE", "DTS", "FLAC", "MKA", "MP2", "MP3", "M4A", "MPC", "OGG", "OGA", "OPUS", "TAK", "THD", "TTA", "VQF", "WAV", "W64", "WMA", "WV", NULL
};

static int inffmpfile_search_supported_extension(int streamtype_select, char *original_filename_in, char *filename_noext_out, const char **found_extension, const char **language_extension)
{
	int streamtype_index = (streamtype_select < 0)? MPXPLAY_STREAMTYPEINDEX_AUDIO : streamtype_select;
	char *fn_extension, strtmp[MAX_PATHNAMELEN];

	if(!filename_noext_out)
	{
		filename_noext_out = &strtmp[0];
		language_extension = NULL;        // we can use this only with filename_noext_out argument
	}

	fn_extension = pds_getfilename_noext_from_fullname(filename_noext_out, original_filename_in);
	if(!fn_extension)
		return streamtype_select;

	if(found_extension)
		*found_extension = NULL;

	do{
		const char *supported_extension, **extension_list;
		unsigned int i;

		if(streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO)
			extension_list = &ffmpeg_supported_extensions_audiostreams[0];
		else if(streamtype_index == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
			extension_list = &ffmpeg_supported_extensions_subtitle[0];
		else
			continue;

		i = 0;
		do {
			supported_extension = extension_list[i++];
			if(pds_stricmp((char *)supported_extension, fn_extension) == 0)
			{
				if(found_extension)
					*found_extension = supported_extension;
				if(language_extension)
				{   // check language extension (videoname.lng.srt)
					char *le = pds_strrchr(filename_noext_out,'.');
					if(le)
					{
						unsigned int lang_len;
						le++;
						lang_len = pds_strlen(le);
						if((lang_len >= INFFMPG_LANGCODE_ISO6391_LEN) && (lang_len <= INFFMPG_LANGCODE_ISO6392_LEN) && pds_chkstr_letters(le)) // language extension is valid (like "HU" or"HUN" or "ENG")
							*language_extension = le;
					}
				}
				return streamtype_index;
			}
		}while(supported_extension);
	}while((streamtype_select < 0) && (++streamtype_index < MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW));

	return streamtype_select; // if the extension of an exact streamtype cannot be detected, we simply pass the streamtype without reporting error (usefull for video files)
}

static inline unsigned int inffmpfile_get_streamindex_start(int streamtype_index)
{
	return (INFFMPG_STREAM_INDEX_EXTERNALFILE_BEGIN + streamtype_index * INFFMPG_STREAM_INDEX_EXTERNALFILE_NB_MAX);
}

static int inffmpgfile_ioctx_file_read(void *opaque, uint8_t *readbuf, int readsize)
{
	struct ffmpeg_external_files_info_s *extfilehandler = (struct ffmpeg_external_files_info_s *)opaque;
	int retcode = AVERROR(ENOMEM);

	if(!extfilehandler || !extfilehandler->fbfs || !extfilehandler->fbds || !readbuf)
		return retcode;
	if(!readsize)
		return AVERROR(EINVAL);

	retcode = extfilehandler->fbfs->fread(extfilehandler->fbds, readbuf, readsize);
	if(retcode < readsize)
	{
		if(!extfilehandler->fbfs->eof(extfilehandler->fbds))
		{
			if(retcode > 0)
				extfilehandler->fbfs->fseek(extfilehandler->fbds, -retcode, SEEK_CUR); // !!! assume buffer seek
			retcode = AVERROR(EAGAIN);
		}
		else if(retcode == 0)
		{
			retcode = AVERROR_EOF;
		}
	}

	in_ffmp_epg_mpegts_demux_extfile(extfilehandler, readbuf, retcode);

	mpxplay_debugf(MPXPLAY_DEBUGOUT_READ,"inffmpgsub_ioctx_file_read st:%d rs:%d ret:%d", extfilehandler->streamtype_index, readsize, retcode);

	return retcode;
}

static int64_t inffmpgfile_ioctx_file_seek(void *opaque, int64_t offset, int whence)
{
	struct ffmpeg_external_files_info_s *extfilehandler = (struct ffmpeg_external_files_info_s *)opaque;
	int64_t retcode = AVERROR(ENOMEM);

	if(!extfilehandler || !extfilehandler->fbfs || !extfilehandler->fbds)
		return retcode;

	if(whence == AVSEEK_SIZE)
		retcode = extfilehandler->fbfs->filelength(extfilehandler->fbds);
	else
	{
		whence &= 0x03;
		retcode = extfilehandler->fbfs->fseek(extfilehandler->fbds, offset, whence);
		in_ffmp_epg_mpegts_clearbuf_extfile(extfilehandler);
	}
	mpxplay_debugf(MPXPLAY_DEBUGOUT_SEEK,"inffmpgfile_ioctx_file_seek st:%d of:%lld wh:%d ret:%lld fn:%s", extfilehandler->streamtype_index,
			offset, whence, retcode, pds_getfilename_from_fullname(extfilehandler->external_file_selected->external_filename));

	return retcode;
}

//----------------------------------------------------------------------------------------------------------------------------------
extern AVInputFormat ff_aac_demuxer;
extern AVInputFormat ff_ac3_demuxer;
extern AVInputFormat ff_dts_demuxer;
extern AVInputFormat ff_mp3_demuxer;
extern AVInputFormat ff_ogg_demuxer;
extern AVInputFormat ff_mpegps_demuxer;
extern AVInputFormat ff_mpegts_demuxer;

static mpxp_bool_t infmpeg_is_fileformat_native_audio_stream(AVFormatContext *fctx) // contains no any header (like AAC or MP3)
{
	if( (fctx->iformat == &ff_aac_demuxer) || (fctx->iformat == &ff_ac3_demuxer) || (fctx->iformat == &ff_dts_demuxer)
	 || (fctx->iformat == &ff_mp3_demuxer) // not used in FFmpeg
	){  // TODO: for other formats
		return TRUE;
	}
	return FALSE;
}

mpxp_bool_t infmpeg_is_fileformat_video_stream(AVFormatContext *fctx) // contains no any header (like TS or MPG)
{
	if((fctx->iformat == &ff_mpegts_demuxer) || (fctx->iformat == &ff_mpegps_demuxer))  // TODO: for other formats
		return TRUE;
	return FALSE;
}

mpxp_bool_t infmpeg_is_fileformat_real_stream(AVFormatContext *fctx) // contains no any header (like TS or MPG)
{
	if(infmpeg_is_fileformat_native_audio_stream(fctx) || infmpeg_is_fileformat_video_stream(fctx))
		return TRUE;
	return FALSE;
}

//----------------------------------------------------------------------------------------------------------------------------------

int in_ffmpfile_filelist_file_add(struct ffmpg_demuxer_data_s *ffmpi, char *filename, int streamtype_index)
{
	struct ffmpeg_external_files_info_s *extfiles;
	struct ffmpeg_external_filelist_elem_s **next_externalfile_in_chainlist, *file_elem;
	unsigned int streamindex_start, stream_index, streamindex_max;
	const char *supported_extension = NULL, *language_extension = NULL;
	char ff_filename_noext[MAX_PATHNAMELEN];

	if(!filename || !filename[0])
		return MPXPLAY_ERROR_INFILE_MEMORY;

	if(streamtype_index < 0)
	{
		streamtype_index = inffmpfile_search_supported_extension(streamtype_index, filename, &ff_filename_noext[0], &supported_extension, &language_extension);
		if(streamtype_index < 0)
			streamtype_index = MPXPLAY_STREAMTYPEINDEX_AUDIO; // if an unknown file is added to the streamlist, it's used as an audio stream (getting the first audio stream)
	}

	stream_index = streamindex_start = inffmpfile_get_streamindex_start(streamtype_index);
	streamindex_max = streamindex_start + INFFMPG_STREAM_INDEX_EXTERNALFILE_NB_MAX - 1;

	file_elem = (struct ffmpeg_external_filelist_elem_s *)pds_calloc(1, sizeof(struct ffmpeg_external_filelist_elem_s));
	if(!file_elem)
		return MPXPLAY_ERROR_INFILE_FILEOPEN;
	pds_strcpy(file_elem->format_name, (char *)supported_extension);
	pds_strcpy(file_elem->language_code, (char *)language_extension);
	pds_strcpy(file_elem->language_name, (char *)mpxplay_ffmplang_getlangname_by_langcode(language_extension));
	pds_strcpy(file_elem->external_filename, filename);

	extfiles = &ffmpi->external_file_infos[streamtype_index];
	next_externalfile_in_chainlist = &(extfiles->external_files_chain);
	while(*next_externalfile_in_chainlist)
	{
		next_externalfile_in_chainlist = &((*next_externalfile_in_chainlist)->next_external_file);
		if(++stream_index >= streamindex_max)
		{
			pds_free(file_elem);
			return MPXPLAY_ERROR_INFILE_FILEOPEN;
		}
	}

	file_elem->stream_index = stream_index;
	extfiles->streamtype_index = streamtype_index;
	*next_externalfile_in_chainlist = file_elem;
	extfiles->nb_external_files++;
	return stream_index;
}

// add multiply stream files to the primary video (called from video context menu)
int in_ffmpfile_filelist_files_add_lfcbds(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_loadfile_callbackdata_s *mcbs)
{
	int i;

	if(!mcbs)
		return MPXPLAY_ERROR_INFILE_MEMORY;

	if(mcbs->filenames)
	{
		for(i = 0; i < mcbs->nb_filenames; i++)
		{
			if(mcbs->filenames[i])
			{
				in_ffmpfile_filelist_file_add(ffmpi, mcbs->filenames[i], -1);
				pds_free(mcbs->filenames[i]);
			}
		}
		pds_free(mcbs->filenames);
	}

	pds_free(mcbs);

	return MPXPLAY_ERROR_INFILE_OK;
}

// clear mcbs informations
void in_ffmpfile_filelist_files_clear_lfcbds(struct mpxplay_loadfile_callbackdata_s *mcbs)
{
	int i;

	if(!mcbs)
		return;

	if(mcbs->filenames)
	{
		int i;

		for(i = 0; i < mcbs->nb_filenames; i++)
		{
			if(mcbs->filenames[i])
			{
				pds_free(mcbs->filenames[i]);
			}
		}
		pds_free(mcbs->filenames);
	}

	pds_free(mcbs);
}

void in_ffmpfile_filelist_load_list_by_ext(struct ffmpeg_external_files_info_s *extfiles, char *video_fullname, unsigned int streamtype_index)
{
	struct ffmpeg_external_filelist_elem_s **next_externalfile_in_chainlist = &extfiles->external_files_chain;
	unsigned int streamindex_start = inffmpfile_get_streamindex_start(streamtype_index);
	unsigned int stream_index = streamindex_start, streamindex_max = streamindex_start + INFFMPG_STREAM_INDEX_EXTERNALFILE_NB_MAX - 1;
	char path_video[MAX_PATHNAMELEN], path_ffb[MAX_PATHNAMELEN], scanfilename[MAX_PATHNAMELEN], fullname[MAX_PATHNAMELEN];
	char search_pathes[MPXINI_MAX_CHARDATA_LEN] = "", *curr_srch_path, *video_shortname;
	struct pds_find_t ffblk;

	// search for the same filename with any extension in mpxplay_config_video_extstream_subdirs (relative to video) (if video_extstream_subdirs empty, it searches in the current dir only)
	video_shortname = pds_getpath_from_fullname(path_video, video_fullname);
	pds_getfilename_any_noext_from_fullname(scanfilename, video_fullname);
	pds_strcat(scanfilename, ".*");

	if(streamtype_index < MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM)
		pds_strcpy(search_pathes, &mpxplay_config_video_extstream_subdirs[streamtype_index][0]);
	curr_srch_path = &search_pathes[0];

	do{
		char *next_srch_path = pds_strchr(curr_srch_path,MPXPLAY_CONFIG_VIDEO_EXTRSTREAM_SUBDIRS_SEPARATOR); // pathes in mpxplay_config_video_extstream_subdirs are separated by ';'
		if(next_srch_path)                                                     // end of current search path from mpxplay_config_video_extstream_subdirs
			*next_srch_path++ = 0;
		pds_str_clean(curr_srch_path);
		pds_filename_assemble_fullname(path_ffb, path_video, curr_srch_path);  // assemble full path for current video_extstream_subdir
		curr_srch_path = next_srch_path;                                       // set the next searchpath for the next loop

		pds_filename_assemble_fullname(fullname, path_ffb, scanfilename);      // assemble full searchname (videofilename.*) for the scan of the current video_extstream_subdir
		if(mpxplay_diskdrive_findfirst(NULL, fullname, _A_NORMAL, &ffblk))     // if we haven't found any external media file in this path
			continue;

		extfiles->streamtype_index = streamtype_index;

		do{
			const char *supported_extension = NULL, *language_extension = NULL;
			struct ffmpeg_external_filelist_elem_s *file_elem;
			char ff_filename_noext[MAX_PATHNAMELEN];

			if(pds_stricmp(ffblk.name, video_shortname) == 0)                   // if we've found the reference media file itself
				continue;                                                       // we don't put it on the ext list
			if(inffmpfile_search_supported_extension(streamtype_index, ffblk.name, &ff_filename_noext[0], &supported_extension, &language_extension) != streamtype_index) // is file format (extension) supported?
				continue;
			if(!supported_extension)
				continue;

			// store external file informations
			file_elem = pds_calloc(1, sizeof(struct ffmpeg_external_filelist_elem_s));
			if(!file_elem)
				break;

			file_elem->stream_index = stream_index++;
			pds_strcpy(file_elem->format_name, (char *)supported_extension);
			pds_strcpy(file_elem->language_code, (char *)language_extension);
			pds_strcpy(file_elem->language_name, (char *)mpxplay_ffmplang_getlangname_by_langcode(language_extension));
			pds_filename_assemble_fullname(fullname, path_ffb, ffblk.name);
			pds_strcpy(file_elem->external_filename, fullname);
			*next_externalfile_in_chainlist = file_elem;
			next_externalfile_in_chainlist = &file_elem->next_external_file;
			extfiles->nb_external_files++;

		}while(!mpxplay_diskdrive_findnext(NULL, &ffblk) && (stream_index < streamindex_max));

		mpxplay_diskdrive_findclose(NULL, &ffblk);

	}while(curr_srch_path);
}

void in_ffmpfile_filelist_close(struct ffmpeg_external_files_info_s *extfiles)
{
	struct ffmpeg_external_filelist_elem_s *externalfile_in_chainlist;

	in_ffmpfile_file_close(extfiles);

	externalfile_in_chainlist = extfiles->external_files_chain;
	extfiles->external_files_chain = NULL;
	while(externalfile_in_chainlist)
	{
		struct ffmpeg_external_filelist_elem_s *next = externalfile_in_chainlist->next_external_file;
		pds_free(externalfile_in_chainlist);
		externalfile_in_chainlist = next;
	}

	in_ffmp_epg_mpegts_close(&extfiles->epg_infos);
}

int in_ffmpfile_filelist_search_by_language(struct ffmpeg_external_files_info_s *extfiles, char *lang_code)
{
	struct ffmpeg_external_filelist_elem_s *externalfile_in_chainlist = extfiles->external_files_chain;
	int stream_index = INFFMPG_STREAM_INDEX_DISABLE;

	if(lang_code && lang_code[0]) // search by lang_name
	{
		while(externalfile_in_chainlist)
		{
			if(pds_stricmp(externalfile_in_chainlist->language_code, lang_code) == 0)
			{
				stream_index = externalfile_in_chainlist->stream_index;
				break;
			}
			externalfile_in_chainlist = externalfile_in_chainlist->next_external_file;
		}
	}

	if(stream_index < 0) // then search stream without language
	{
		externalfile_in_chainlist = extfiles->external_files_chain;
		while(externalfile_in_chainlist)
		{
			if(!externalfile_in_chainlist->language_code[0])
			{
				stream_index = externalfile_in_chainlist->stream_index;
				break;
			}
			externalfile_in_chainlist = externalfile_in_chainlist->next_external_file;
		}
	}

	return stream_index;
}

static struct ffmpeg_external_filelist_elem_s *in_ffmpfile_filelist_search_by_streamindex(struct ffmpeg_external_files_info_s *extfiles, int stream_index)
{
	struct ffmpeg_external_filelist_elem_s *externalfile_in_chainlist = extfiles->external_files_chain;

	while(externalfile_in_chainlist)
	{
		if((stream_index < 0) || (externalfile_in_chainlist->stream_index == stream_index)) // (stream_index < 0) selects the first file
			break;
		externalfile_in_chainlist = externalfile_in_chainlist->next_external_file;
	}

	return externalfile_in_chainlist;
}

static unsigned int in_ffmpfile_file_utftexttype_detect(struct ffmpeg_external_files_info_s *extfilehandler)
{
	unsigned int utftype = 0;
	mpxp_uint8_t readbuf[4];

	if(extfilehandler->fbfs->fread(extfilehandler->fbds, readbuf, 3) == 3)
	{
		if((readbuf[0] == 0xff) && (readbuf[1] == 0xfe))
			utftype = MPXPLAY_TEXTCONV_TYPE_UTF16LE;
		else if((readbuf[0] == 0xfe) && (readbuf[1] == 0xff))
			utftype = MPXPLAY_TEXTCONV_TYPE_UTF16BE;
		else if((readbuf[0] == 0xef) && (readbuf[1] == 0xbb) && (readbuf[2] == 0xbf))
			utftype = MPXPLAY_TEXTCONV_TYPE_UTF8;
		else if(funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_SUBTITLES_UTF8DEC_ENABLE))
			utftype = MPXPLAY_TEXTCONV_TYPE_UTF8;
	}

	extfilehandler->fbfs->fseek(extfilehandler->fbds, 0, SEEK_SET);

	return utftype;
}

// the sync can be stream based (if any of the streams is live (http)) or it's connected to the audio (all av streams belong to non-live-streams (local / ftp) files)
static void inffmppfile_stream_avsync_decision(struct ffmpg_demuxer_data_s *ffmpi)
{
	struct ffmpeg_external_files_info_s *extfilehandler =  &ffmpi->external_file_infos[0];
	int stream_type, nb_used_streams = 0, nb_live_streams = 0;

	for(stream_type = 0; stream_type < ffmpi->streamtype_limit; stream_type++, extfilehandler++)
	{
		if(extfilehandler->external_file_avctx)
		{
			nb_used_streams++;
			if(funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_IS_LIVESTREAM)) // TODO: what should we do with the non-seekable (but non endless) files?
			{
				nb_live_streams++;
			}
		}
	}

	extfilehandler =  &ffmpi->external_file_infos[0];
	for(stream_type = 0; stream_type <= MPXPLAY_STREAMTYPEINDEX_SUBTITLE; stream_type++, extfilehandler++)
	{
		extfilehandler->avclocktime_us = INFFMPG_INVALID_PTS;
		extfilehandler->avtimestamp_us = INFFMPG_INVALID_PTS;
	}
	if((nb_used_streams > 1) && (nb_live_streams > 0))
		funcbit_enable(ffmpi->flags, INFFMPG_FLAG_INDEPENDENTSTREAMSYNC);
	else
		funcbit_disable(ffmpi->flags, INFFMPG_FLAG_INDEPENDENTSTREAMSYNC);
}

// returns the number of streams (curently 1)
int in_ffmpfile_file_open_by_streamindex(struct ffmpg_demuxer_data_s *ffmpi, unsigned int stream_type, int stream_index, AVStream **selected_avstream, AVCodec **codec_av_new)
{
	struct ffmpeg_external_files_info_s *extfilehandler = &ffmpi->external_file_infos[stream_type];
	struct ffmpeg_external_filelist_elem_s *extfileinfo;
	int block_size = 0, retcode = MPXPLAY_ERROR_INFILE_FILEOPEN, prereadbytes, scan_all_pmts_set = 0;
	AVInputFormat *demuxer = NULL;
	struct mpxpframe_s *frp; // !!! hack
	char *filename = NULL;
	AVFormatContext *fctx;

	in_ffmpfile_file_close(extfilehandler);

	extfileinfo = in_ffmpfile_filelist_search_by_streamindex(extfilehandler, stream_index);
	if(!extfileinfo)
		return retcode;
	extfilehandler->external_file_selected = extfileinfo;
	extfilehandler->epg_infos.control_cb = (ffmpi->miis)? ffmpi->miis->control_cb : NULL;
	extfilehandler->epg_infos.ccb_data = ffmpi->fbds;
	extfilehandler->epg_infos.ffmpi = (void *)ffmpi;

	if(stream_type == MPXPLAY_STREAMTYPEINDEX_VIDEO) // means primary media file too
	{
		extfilehandler->fbfs = ffmpi->fbfs;
		extfilehandler->fbds = ffmpi->fbds;
		extfilehandler->control_flags = ffmpi->flags;
	}
	else // allocate Mpxplay's file handler structure (mpxinbuf, diskdriv) for secondary (parallel/external) files
	{
		extfilehandler->fbds = mpxplay_infile_filehandler_alloc(&extfilehandler->fbfs);
		if(!extfilehandler->fbds)
			return retcode;
		extfilehandler->control_flags = 0;
	}
	extfilehandler->streamtype_index = stream_type;

	if(!extfilehandler->fbfs->fopen(extfilehandler->fbds, extfileinfo->external_filename, ffmpi->primary_file_openmode | O_BINARY, INFFMPG_IOCTX_SIZE_MAX)){
		mpxplay_debugf(MPXPLAY_DEBUG_HEADER, "inffmpg_infile_check fopen failed");
		return MPXPLAY_ERROR_INFILE_FILEOPEN;
	}

	if(ffmpi->miis->control_cb(extfilehandler->fbds, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_ISSEEKABLE, NULL, NULL))
		funcbit_enable(extfilehandler->control_flags, INFFMPG_FLAG_IS_SEEKABLE);
	if(ffmpi->miis->control_cb(extfilehandler->fbds, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_ISSTREAM, NULL, NULL))
		funcbit_enable(extfilehandler->control_flags, INFFMPG_FLAG_IS_LIVESTREAM);
	if(ffmpi->miis->control_cb(extfilehandler->fbds, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_ISASYNCREAD, NULL, NULL))
		funcbit_enable(extfilehandler->control_flags, INFFMPG_FLAG_IS_ASYNCREAD);

	if(!funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_IS_LIVESTREAM) && extfilehandler->fbfs->filelength && (extfilehandler->fbfs->filelength(extfilehandler->fbds) <= 44)) // too small file
		goto err_out_open;

	// pre-load stream data
	frp = (struct mpxpframe_s *)extfilehandler->fbds;
	prereadbytes = mpxplay_diskdrive_file_config(frp->filehand_datas, MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_PREREADBUFBYTES, NULL, NULL);
	if(prereadbytes > 0)
	{
		int retry = ((prereadbytes / frp->prebufferblocksize) + 1 ) * 2;
		while(frp->prebufferbytes_forward < prereadbytes)
		{
			const long prev_pbf = frp->prebufferbytes_forward;
			if(mpxplay_mpxinbuf_buffer_protected_check(frp) != MPXPLAY_ERROR_OK)
				break;
			if(--retry <= 0)
				break;
			if((frp->prebufferbytes_forward - prev_pbf) < frp->prebufferblocksize)
				pds_threads_sleep(MPXPLAY_THREADS_SHORTTASKSLEEP);
		}
	}

	if((stream_type == MPXPLAY_STREAMTYPEINDEX_VIDEO) || (stream_type == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)) // means the primary media file (can be live-stream too)
	{
		if(funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_IS_LIVESTREAM))
		{
			funcbit_disable(ffmpi->primary_file_openmode, MPXPLAY_INFILE_OPENMODE_INFO_LENGTH);
			ffmpi->miis->timemsec = 0x7eeeeeee;
		}
		filename = (ffmpi->primary_file_openmode & MPXPLAY_INFILE_OPENMODE_LOAD_PLAY)? NULL : &extfileinfo->external_filename[0];
		demuxer = (ffmpi->primary_file_openmode & MPXPLAY_INFILE_OPENMODE_AUTODETECT)? NULL : ffmpi->primary_file_demuxer;
		extfilehandler->control_flags |= (MPXPLAY_TEXTCONV_TYPE_UTF8 << INFFMPG_FLAG_TEXTTYPE_BITPOS); // FIXME: hack for internal subtitles (are they always UTF-8?)
	}
	else
	{
		if(stream_type == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)  // get text coding type (utf)
		{
			extfilehandler->control_flags |= in_ffmpfile_file_utftexttype_detect(extfilehandler) << INFFMPG_FLAG_TEXTTYPE_BITPOS;
		}
	}

	// initialize FFmpeg's AVFormatContext with own ioctx
	fctx = avformat_alloc_context();
	if(!fctx)
		goto err_out_open;
	extfilehandler->external_file_avctx = fctx;

	// allocate own AVIOContext (low level file handler and buffering)
	block_size = ffmpi->miis->control_cb(extfilehandler->fbds, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_RBLOCKSIZE, NULL, NULL);

	if(funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_IS_LIVESTREAM))
		fctx->format_probesize = prereadbytes;

	if(block_size < INFFMPG_IOCTX_SIZE_MIN)
		block_size = INFFMPG_IOCTX_SIZE_MIN;
	else if(block_size > INFFMPG_IOCTX_SIZE_MAX)
		block_size = INFFMPG_IOCTX_SIZE_MAX;

	extfilehandler->iobuffer = av_malloc(block_size + 1024);
	if(!extfilehandler->iobuffer)
		goto err_out_open;

	extfilehandler->ioctx = avio_alloc_context(extfilehandler->iobuffer, block_size, 0, (void *)extfilehandler, inffmpgfile_ioctx_file_read, NULL, (extfilehandler->control_flags & INFFMPG_FLAG_IS_SEEKABLE)? inffmpgfile_ioctx_file_seek : NULL);
	if(!extfilehandler->ioctx)
	{
		av_freep(&extfilehandler->iobuffer);
		goto err_out_open;
	}
	fctx->pb = extfilehandler->ioctx;
	fctx->max_streams = INFFMPG_MAX_STREAMS;

	if(!(ffmpi->primary_file_openmode & MPXPLAY_INFILE_OPENMODE_LOAD_SEEKTAB))
		funcbit_enable(fctx->flags, (AVFMT_FLAG_IGNIDX | AVFMT_FLAG_NOPARSE | AVFMT_FLAG_NOFILLIN));
	else if(!funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_IS_SEEKABLE))
		funcbit_enable(fctx->flags, AVFMT_FLAG_IGNIDX);
	funcbit_enable(fctx->flags, (AVFMT_FLAG_CUSTOM_IO | AVFMT_FLAG_NOBUFFER));

	if(stream_type == MPXPLAY_STREAMTYPEINDEX_VIDEO)
	{   // for better MPEG-TS stream parsing // TODO: check
	    if(!av_dict_get(extfilehandler->external_file_avctx->metadata, "scan_all_pmts", NULL, AV_DICT_MATCH_CASE)) {
	        av_dict_set(&extfilehandler->external_file_avctx->metadata, "scan_all_pmts", "1", AV_DICT_DONT_OVERWRITE);
	        scan_all_pmts_set = 1;
	    }
	}
	else if((stream_type == MPXPLAY_STREAMTYPEINDEX_AUDIO) || (stream_type == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW))
	{
		av_dict_set(&extfilehandler->external_file_avctx->metadata, "track", "42", 0);  // TODO: check (disable the reading of metadata for faster open (this seems to be not working?))
	}

	retcode = avformat_open_input(&extfilehandler->external_file_avctx, filename, demuxer, NULL);
	if(retcode != 0)
		goto err_out_open;

	if(scan_all_pmts_set)
		av_dict_set(&extfilehandler->external_file_avctx->metadata, "scan_all_pmts", NULL, AV_DICT_MATCH_CASE);

	if(fctx->duration <= 0)
		funcbit_enable(extfilehandler->control_flags, INFFMPG_FLAG_IS_STREAM); // TODO: check

	if( (extfilehandler->control_flags & INFFMPG_FLAG_IS_STREAM) && (extfilehandler->control_flags & INFFMPG_FLAG_IS_SEEKABLE)
	 && (infmpeg_is_fileformat_video_stream(fctx) || (fctx->iformat == &ff_ogg_demuxer)) // TODO: for other formats
	){
		funcbit_enable(extfilehandler->control_flags, INFFMPG_FLAG_STREAM_PARSING);
	}
	else if((stream_type == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW) && (fctx->duration < AV_TIME_BASE))
	{
		fctx->duration = ffmpi->fctx->duration;
	}

	if(!infmpeg_is_fileformat_real_stream(fctx))
	{
		funcbit_enable(((struct mpxpframe_s *)extfilehandler->fbds)->flags, MPXPLAY_MPXPFRAME_FLAG_FILEFORMAT_CONTAINER); // TODO: config API
	}

	if(codec_av_new && selected_avstream)
	{
		AVStream *stream;

		if(avformat_find_stream_info(fctx, NULL) < 0)
			goto err_out_open;

		stream = in_ffmpstrm_find_videostream_and_decoder(fctx, stream_type, ((stream_type == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)? ffmpi->program_number_current : -1), codec_av_new);
		if(!stream || !*codec_av_new)
			goto err_out_open;

		*selected_avstream = stream;

		extfileinfo->codec_id = stream->codecpar->codec_id;
		extfileinfo->bitrate = stream->codecpar->bit_rate;
#if MPXPLAY_USE_FFMPEG_V7X
		extfileinfo->channels = stream->codecpar->ch_layout.nb_channels;
#else
		extfileinfo->channels = (stream->codecpar->ch_layout.nb_channels > 0)? stream->codecpar->ch_layout.nb_channels : stream->codecpar->channels;
#endif
		extfileinfo->samplerate = stream->codecpar->sample_rate;

		if(funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_STREAM_PARSING))
			fctx->duration = in_ffmppars_get_stream_duration_pts(extfilehandler, stream->index);

		if(fctx->duration < AV_TIME_BASE)
			fctx->duration = AV_TIME_BASE;

		if((stream_type == MPXPLAY_STREAMTYPEINDEX_AUDIO) || (stream_type == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)) // seek (sync extra stream) to video current pos
		{
			mpxp_int64_t seek_pos_us = 0;
			if((ffmpi->avsync_timestamp >= 0) && !funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
				seek_pos_us = ffmpi->avsync_timestamp;
			in_ffmpfile_file_seek(extfilehandler, seek_pos_us, AVSEEK_FLAG_BACKWARD | AVSEEK_FLAG_ANY);
		}
	}

	funcbit_enable(fctx->flags, AVFMT_FLAG_FAST_SEEK);

	if(stream_type != MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
		inffmppfile_stream_avsync_decision(ffmpi);

	return 1;

err_out_open:
	in_ffmpfile_file_close(extfilehandler);
	return retcode;
}

void in_ffmpfile_file_close(struct ffmpeg_external_files_info_s *extfilehandler)
{
	if(extfilehandler->external_file_avctx)
	{
		if(extfilehandler->external_file_avctx->pb)
			avio_flush(extfilehandler->external_file_avctx->pb);
		avformat_flush(extfilehandler->external_file_avctx);
		avformat_close_input(&extfilehandler->external_file_avctx);
	}
	if(extfilehandler->fbfs && extfilehandler->fbds)
		extfilehandler->fbfs->fclose(extfilehandler->fbds);
	if(extfilehandler->streamtype_index != MPXPLAY_STREAMTYPEINDEX_VIDEO)
		mpxplay_infile_filehandler_free(extfilehandler->fbds);
	extfilehandler->fbds = NULL;
	extfilehandler->fbfs = NULL;
	extfilehandler->ioctx = NULL;
	extfilehandler->iobuffer = NULL;
	extfilehandler->external_file_selected = NULL;
	extfilehandler->control_flags = 0;
	extfilehandler->avclocktime_us = INFFMPG_INVALID_PTS;
	extfilehandler->avtimestamp_us = INFFMPG_INVALID_PTS;
	extfilehandler->last_seek_dts = INFFMPG_INVALID_PTS;
}

int in_ffmpfile_file_packet_read(struct ffmpeg_external_files_info_s *extfilehandler, AVPacket *pkt)
{
	int retcode = -1;

	if(!extfilehandler->external_file_avctx)
		return retcode;

	av_packet_unref(pkt);
	retcode = av_read_frame(extfilehandler->external_file_avctx, pkt);

	return retcode;
}

int in_ffmpfile_file_seek(struct ffmpeg_external_files_info_s *extfilehandler, mpxp_int64_t seek_pos_us, unsigned int seek_flags)
{
#ifdef INFFMP_USE_DEMUXER_SEEK
	struct mpxpframe_s *frp = (struct mpxpframe_s *)extfilehandler->fbds;
	unsigned int buffertype_save;
#endif
	int retcode = MPXPLAY_ERROR_INFILE_EOF;

	if(!extfilehandler->external_file_avctx)
		return retcode;
	if(funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_IS_LIVESTREAM))
		return MPXPLAY_ERROR_INFILE_OK;

#ifdef INFFMP_USE_DEMUXER_SEEK
	if(frp)
	{
		buffertype_save = frp->buffertype;
		funcbit_disable(frp->buffertype, PREBUFTYPE_INT);
	}
#endif

	if(funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_STREAM_PARSING)) // !!! means audio/video vbr stream (FFmpeg cannot seek properly them)
	{
		seek_pos_us =  (mpxp_int64_t)((float)extfilehandler->fbfs->filelength(extfilehandler->fbds) * (float)seek_pos_us / (float)extfilehandler->external_file_avctx->duration);
		funcbit_enable(seek_flags, AVSEEK_FLAG_BYTE);
	}

	if(extfilehandler->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
		seek_pos_us = (seek_pos_us > 10 * AV_TIME_BASE)? (seek_pos_us - 10 * AV_TIME_BASE) : 0; // !!! seek subtitle to an earlier pos to surely catch the begin of sentence (-10 secs)

	retcode = av_seek_frame(extfilehandler->external_file_avctx, -1, seek_pos_us, seek_flags);
	if(retcode < 0)
	{
		if(!funcbit_test(seek_flags, AVSEEK_FLAG_BACKWARD))
			funcbit_enable(seek_flags, AVSEEK_FLAG_BACKWARD);
		else
			funcbit_enable(seek_flags, AVSEEK_FLAG_ANY);
		retcode = av_seek_frame(extfilehandler->external_file_avctx, -1, seek_pos_us, seek_flags);
		mpxplay_debugf(MPXPLAY_DEBUGOUT_SEEK,"in_ffmpfile_file_seek FAILED st:%d pos:%lld ret:%d fn:%s", extfilehandler->streamtype_index,
				seek_pos_us, retcode, pds_getfilename_from_fullname(extfilehandler->external_file_selected->external_filename));
		if(retcode < 0)
		{
			if(extfilehandler->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
			{
				retcode = av_seek_frame(extfilehandler->external_file_avctx, -1, 0, 0);
				seek_pos_us = 0;
			}
			else
			{
				if(extfilehandler->fbfs && extfilehandler->fbfs->filelength && extfilehandler->fbfs->fseek && extfilehandler->fbds)
				{
					mpxp_filesize_t seek_pos_byte;
					seek_pos_byte = (seek_flags & AVSEEK_FLAG_BYTE)? seek_pos_us : ((mpxp_int64_t)((float)extfilehandler->fbfs->filelength(extfilehandler->fbds) * (float)seek_pos_us / (float)extfilehandler->external_file_avctx->duration));
					if(extfilehandler->fbfs->fseek(extfilehandler->fbds, seek_pos_byte, SEEK_SET) == seek_pos_byte)
						retcode = MPXPLAY_ERROR_INFILE_OK;
					else
						seek_pos_us = extfilehandler->avtimestamp_us;
				}
			}
		}
	}

#ifdef INFFMP_USE_DEMUXER_SEEK
	if(frp)
	{
		funcbit_smp_copy(frp->buffertype, buffertype_save, PREBUFTYPE_INT);
	}
#endif

	retcode = (retcode < 0)? MPXPLAY_ERROR_INFILE_EOF : MPXPLAY_ERROR_INFILE_OK;

	if(retcode == MPXPLAY_ERROR_INFILE_OK)
	{
		extfilehandler->avclocktime_us = INFFMPG_INVALID_PTS;
		extfilehandler->avtimestamp_us = INFFMPG_INVALID_PTS;
	}

	mpxplay_debugf(MPXPLAY_DEBUGOUT_SEEK,"in_ffmpfile_file_seek st:%d pos:%lld f:%8.8X ret:%d", extfilehandler->streamtype_index, seek_pos_us, seek_flags, retcode);

	return retcode;
}

/*****************************************************************************
 * file output (record) multiplexing by FFmpeg
 *****************************************************************************/
#ifdef MPXPLAY_GUI_QT
static AVPacket *in_ffmpfile_file_output_collect_packet(struct ffmpeg_external_outfile_infos_s *outfile, AVPacket *pkt_in, int stream_idx);

static void in_ffmpfile_file_output_display_progress(struct ffmpeg_external_outfile_infos_s *outfile, mpxp_int32_t elapsed_time_sec)
{
	char *codec_names[MPXPLAY_STREAMTYPEINDEX_PLAYNUM], elapsed_time_str[64] = "\nto\n", sout[256];
	int stream_idx;
	for(stream_idx = 0; stream_idx < MPXPLAY_STREAMTYPEINDEX_PLAYNUM; stream_idx++)
		codec_names[stream_idx] = (char *)((outfile->out_streams[stream_idx])? avcodec_get_name(outfile->out_streams[stream_idx]->codecpar->codec_id) : NULL);
	if(elapsed_time_sec > 0)
		snprintf(elapsed_time_str, sizeof(elapsed_time_str), " at %d:%2.2d to ", (elapsed_time_sec / 60), (elapsed_time_sec % 60));
	snprintf(sout, sizeof(sout), "Writing streams (%s%s%s)%s%s",
			(codec_names[MPXPLAY_STREAMTYPEINDEX_AUDIO])? codec_names[MPXPLAY_STREAMTYPEINDEX_AUDIO] : "",
			((codec_names[MPXPLAY_STREAMTYPEINDEX_AUDIO] && codec_names[MPXPLAY_STREAMTYPEINDEX_VIDEO])? "/" : ""),
			(codec_names[MPXPLAY_STREAMTYPEINDEX_VIDEO])? codec_names[MPXPLAY_STREAMTYPEINDEX_VIDEO] : "",
			elapsed_time_str, pds_getfilename_from_fullname(outfile->output_filename));
	display_timed_message(sout); // display progress informations in the status line
	//if(!elapsed_time_sec)
	//	display_textwin_openwindow_timedmessage(sout); // display initial informations in a timed window
	if((elapsed_time_sec < 10) || !(elapsed_time_sec % (elapsed_time_sec / 10))) // update time-length info of the new playlist entry
	{
		mpxplay_playlist_modify_entry_alltab_sheduler_init(NULL, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_FILESIZE,
			(mpxp_ptrsize_t)&outfile->output_filename[0], sizeof(outfile->output_filename), (mpxp_ptrsize_t)outfile->record_duration_bytes, 0);
		mpxplay_playlist_modify_entry_alltab_sheduler_init(NULL, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_TIMELENMS,
			(mpxp_ptrsize_t)&outfile->output_filename[0], sizeof(outfile->output_filename), (mpxp_ptrsize_t)(elapsed_time_sec * 1000), 0);
	}
}

static int in_ffmpfile_file_output_open(struct ffmpg_demuxer_data_s *ffmpi)
{
	struct ffmpeg_external_outfile_infos_s *outfile = &ffmpi->recorded_file_infos;
	AVFormatContext *outfile_avctx = NULL;
	int i, ret = -1;
	char sout[256];

	ret = avformat_alloc_output_context2(&outfile_avctx, NULL, NULL, outfile->output_filename);
	if (!outfile_avctx || (ret < 0)) {
		mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "in_ffmpfile_file_output_init avformat_alloc_context failed ret:%d", ret);
		goto fail;
	}

	for(i = 0; i < MPXPLAY_STREAMTYPEINDEX_PLAYNUM; i++)
	{
		AVCodecParameters *in_codecpar, *out_codecpar;
		AVStream *in_stream = ffmpi->selected_avstream[i];
		AVDictionaryEntry *t;

		if(!(ffmpi->selected_stream_number[i] < INFFMPG_STREAM_INDEX_VALID) != !(in_stream == NULL)) // data inconsistency -> something went wrong
		{
			goto fail;
		}
		if(ffmpi->selected_stream_number[i] < INFFMPG_STREAM_INDEX_VALID) // no stream found for this type
		{
			outfile->selected_stream_nums[i] = INFFMPG_STREAM_INDEX_DISABLE;
			continue;
		}

		in_codecpar = ffmpi->selected_avstream[i]->codecpar;

		if(!in_codecpar || (in_codecpar->format < 0))
		{
			if(i <= MPXPLAY_STREAMTYPEINDEX_VIDEO) // data inconsistency -> something went wrong
				goto fail;
			outfile->selected_stream_nums[i] = INFFMPG_STREAM_INDEX_DISABLE;
			continue;
		}

		outfile->selected_stream_nums[i] = ffmpi->selected_stream_number[i];
		outfile->selected_ff_stream_indexes[i] = ffmpi->selected_ff_stream_index[i];

		ret = AVERROR(ENOMEM);
		outfile->out_streams[i] = avformat_new_stream(outfile_avctx, NULL);
		if(!outfile->out_streams[i]){
			mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "in_ffmpfile_file_output_init Could not create output stream");
			goto fail;
		}

		outfile->out_streams[i]->id = i;

		out_codecpar = outfile->out_streams[i]->codecpar = avcodec_parameters_alloc();
		if(!out_codecpar){
			mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "in_ffmpfile_file_output_init Could not create output codecpar");
			goto fail;
		}

		if(avcodec_parameters_copy(out_codecpar, in_codecpar) < 0)
			goto fail;

		t = av_dict_get(in_stream->metadata, "language", NULL, 0);

		if(t && t->key && t->value)
		{
			av_dict_set(&outfile->out_streams[i]->metadata, "language", t->value, 0);
		}

		out_codecpar->codec_tag = 0;

		outfile->input_time_base[i] = in_stream->time_base;
		outfile->dts_last[i] = AV_NOPTS_VALUE;

	    mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "in_ffmpfile_file_output_init %d w:%d h:%d f:%d sar:%d:%d",
	       i, out_codecpar->width, out_codecpar->height, out_codecpar->format,
	       out_codecpar->sample_aspect_ratio.num, out_codecpar->sample_aspect_ratio.den);
	}

	ret = avio_open(&outfile_avctx->pb, outfile->output_filename, AVIO_FLAG_WRITE);
	if (ret < 0) {
		mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "Could not open output file '%s' ret:%d", outfile->output_filename, ret);
		goto fail;
	}

	outfile_avctx->avoid_negative_ts = AVFMT_AVOID_NEG_TS_MAKE_NON_NEGATIVE; // FIXME: this doesn't work in FFmpeg?

	ret = avformat_write_header(outfile_avctx, NULL);
	if (ret < 0) {
		mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "Error occurred when avformat_write_header ret:%d", ret);
		goto fail;
	}

	mpxplay_playlist_modify_entry_alltab_sheduler_init(NULL, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_ADDDIRENTRY, (mpxp_ptrsize_t)&outfile->output_filename[0], sizeof(outfile->output_filename), 0, 0);

	in_ffmpfile_file_output_display_progress(outfile, 0);

	outfile->program_control_save = mpxplay_programcontrol;
	funcbit_enable(mpxplay_programcontrol, MPXPLAY_PROGRAMC_CONFIRMEXIT);

	outfile->output_file_avctx = outfile_avctx;

	mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "in_ffmpfile_file_output_init END");

	return ret;

fail:
	snprintf(sout, sizeof(sout), "Start of recording FAILED for\n%s\nTry again with a different output type!\n(or if it seems that something went wrong, restart the playing)", outfile->output_filename);
	outfile->error = TRUE;
	outfile->output_file_avctx = outfile_avctx;
	in_ffmpfile_file_output_close(ffmpi);
	display_textwin_openwindow_errormsg_ok(sout);
	return ret;
}

int in_ffmpfile_file_output_init(struct ffmpg_demuxer_data_s *ffmpi, struct mpxplay_loadfile_callbackdata_s *mcbs)
{
	struct ffmpeg_external_outfile_infos_s *outfile = &ffmpi->recorded_file_infos;

	in_ffmpfile_file_output_close(ffmpi);

	if(mcbs && mcbs->filenames && mcbs->filenames[0])
		pds_strncpy(outfile->output_filename, mcbs->filenames[0], sizeof(outfile->output_filename) - 1);
	in_ffmpfile_filelist_files_clear_lfcbds(mcbs);

	mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "in_ffmpfile_file_output_init BEGIN %s", outfile->output_filename);

	if(!outfile->output_filename[0])
		return -1;

	outfile->recording_initiated = TRUE;

	return 0;
}

void in_ffmpfile_file_output_close(struct ffmpg_demuxer_data_s *ffmpi)
{
	struct ffmpeg_external_outfile_infos_s *outfile = &ffmpi->recorded_file_infos;
	AVFormatContext *outfile_avctx = outfile->output_file_avctx;
	char sout[512];

	outfile->recording_initiated = FALSE;
	if(outfile_avctx) {
		int i;
		outfile->output_file_avctx = NULL;
		if(!outfile->error) {
			outfile_avctx->duration = outfile->record_duration_us;
			av_write_trailer(outfile_avctx);
		}
		if(outfile_avctx->oformat && outfile_avctx->pb && !(outfile_avctx->oformat->flags & AVFMT_NOFILE))
			avio_closep(&outfile_avctx->pb);
		avformat_free_context(outfile_avctx);
		for(i = 0; i < MPXPLAY_STREAMTYPEINDEX_PLAYNUM; i++)
			if(outfile->input_packet_collector[i])
				av_packet_free(&outfile->input_packet_collector[i]); // TODO: last collected packet is not flushed
		if(!outfile->error) {
			display_clear_timed_message();
			mpxplay_playlist_modify_entry_alltab_sheduler_init(NULL, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_FILESIZE,
					(mpxp_ptrsize_t)&outfile->output_filename[0], sizeof(outfile->output_filename), (mpxp_ptrsize_t)outfile->record_duration_bytes, 0);
			mpxplay_playlist_modify_entry_alltab_sheduler_init(NULL, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_TIMELENMS,
					(mpxp_ptrsize_t)&outfile->output_filename[0], sizeof(outfile->output_filename), (mpxp_ptrsize_t)(outfile->record_duration_us / 1000), 0);
			snprintf(sout, sizeof(sout), "Recording to\n%s\nhas finished at %d:%2.2d", pds_getfilename_from_fullname(outfile->output_filename),
					(outfile->record_duration_us / (60 * AV_TIME_BASE)), ((outfile->record_duration_us / AV_TIME_BASE) % 60));
			display_textwin_openwindow_timedmsg_ok(sout);
		} else {
			pds_unlink(outfile->output_filename);
			mpxplay_playlist_modify_entry_alltab_sheduler_init(NULL, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_DELDIRENTRY, (mpxp_ptrsize_t)&outfile->output_filename[0], sizeof(outfile->output_filename), 0, 0);
		}
		funcbit_copy(mpxplay_programcontrol, outfile->program_control_save, MPXPLAY_PROGRAMC_CONFIRMEXIT);
	}
	pds_memset(outfile, 0, sizeof(*outfile));
	mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "in_ffmpfile_file_output_close END");
}

static int in_ffmpfile_file_output_reopen(struct ffmpg_demuxer_data_s *ffmpi)
{
	struct ffmpeg_external_outfile_infos_s *outfile = &ffmpi->recorded_file_infos;
	char filename_save[MAX_PATHNAMELEN];

	outfile->error = TRUE; // don't write trailer
	pds_strcpy(filename_save, outfile->output_filename);
	in_ffmpfile_file_output_close(ffmpi);
	pds_strcpy(outfile->output_filename, filename_save);
	return in_ffmpfile_file_output_open(ffmpi);
}

mpxp_bool_t in_ffmpfile_file_output_running(struct ffmpg_demuxer_data_s *ffmpi)
{
	return (ffmpi->recorded_file_infos.output_file_avctx)? TRUE : FALSE;
}

int in_ffmpfile_file_output_writeframe(struct ffmpg_demuxer_data_s *ffmpi, AVFormatContext *avform_in, AVPacket *pkt_in)
{
	struct ffmpeg_external_outfile_infos_s *outfile = &ffmpi->recorded_file_infos;
	int stream_idx, elapsed_timesec, retval = -1;
	int64_t reference_dts_scaled;
	AVStream *out_stream;
	AVPacket *pkt_out;
	char sout[256];

	if(!pkt_in || !pkt_in->size)
		return retval;

	if(!outfile->output_file_avctx)
	{
		if(!outfile->recording_initiated)
			return retval;
		outfile->recording_initiated = FALSE;
		if(in_ffmpfile_file_output_open(ffmpi) < 0)
			return retval;
	}

	if(!outfile->reference_first_flag && (pkt_in->dts < 0)) // check that the packet is useful for first frame
		return retval;

	// we validate the input packet here (it shall match with one the initial selected record A/V streams)
	out_stream = NULL;
	for(stream_idx = 0; stream_idx < MPXPLAY_STREAMTYPEINDEX_PLAYNUM; stream_idx++)
	{
		if((outfile->selected_stream_nums[stream_idx] >= INFFMPG_STREAM_INDEX_VALID) && (avform_in == ffmpi->fctx)) // TODO: external stream handling
		{
			if(pkt_in->stream_index == outfile->selected_ff_stream_indexes[stream_idx])
			{
				if(outfile->output_file_avctx)
					out_stream = outfile->output_file_avctx->streams[stream_idx];
				break;
			}
		}
	}
	if(!out_stream) // packet is not valid to write it
		return retval;

	outfile->packet_counter++;

	if( (out_stream->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) && (outfile->dts_last[stream_idx] < 0)  // video stream shall begin with keyframe (else other players (VLC) can crash on the output file)
	 && !funcbit_test(pkt_in->flags, AV_PKT_FLAG_KEY) && (outfile->packet_counter < MPXPLAY_OUTFILE_PACKETCOUNT_INITIALLIMIT)
	){
		return retval;
	}

	if(pkt_in->dts >= 0) // save first packet for base/reference/zero-point value
	{
		if(!outfile->reference_first_flag)
		{
			outfile->reference_dts = pkt_in->dts;
			outfile->reference_stream_idx = stream_idx;
			outfile->reference_first_flag = TRUE;
		}
		else if((outfile->dts_last[stream_idx] > 0) && (pkt_in->dts <= (outfile->dts_last[stream_idx] - (MPXPLAY_OUTFILE_ASYNC_FRAME_LIMIT * pkt_in->duration))))
		{
			mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "DTS DROP %d %lld %lld %d", stream_idx, pkt_in->dts, outfile->dts_last[stream_idx], pkt_in->duration);
			if(outfile->packet_counter < MPXPLAY_OUTFILE_PACKETCOUNT_INITIALLIMIT){ // if continuity error has detected near to start, we can simply reopen the file
				retval = in_ffmpfile_file_output_reopen(ffmpi);
				mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "DTS REOPEN ret:%d", retval);
			}
			return retval;
		}

		outfile->dts_last[stream_idx] = pkt_in->dts;
	}

	// input packets are delayed/collected in a buffer, because FFmpeg sometimes gives incomplete packets with NO_PTS timestamp
	pkt_out = in_ffmpfile_file_output_collect_packet(outfile, pkt_in, stream_idx);
	if(!pkt_out)
		return retval;

	reference_dts_scaled = (stream_idx == outfile->reference_stream_idx)? outfile->reference_dts : av_rescale_q_rnd(outfile->reference_dts, outfile->input_time_base[outfile->reference_stream_idx], outfile->input_time_base[stream_idx], AV_ROUND_NEAR_INF|AV_ROUND_PASS_MINMAX);

	// subtract reference (start point) dts from the current dts/pts (to start first frame at a zero dts)
	{
		int64_t pkt_in_dts = pkt_out->dts, pkt_in_corrected_dts;
		if(pkt_in_dts < 0)
		{   // fill invalid (negative) dts value with a calculated (interlaced) one
			pkt_in_dts = outfile->dts_last[stream_idx] + outfile->duration_last[stream_idx];
			mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "DTS CORR dts:%lld dtsc:%lld rdur:%lld", pkt_out->dts, pkt_in_dts, outfile->record_duration_us);
		}
		if(pkt_in_dts < reference_dts_scaled){ // !!! we drop frame if dts is less than reference (TODO: we should store first frames in a queue for analyzing)
			mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "DTS REFERENCED DROP %d dts:%lld ref:%lld", stream_idx, pkt_out->dts, reference_dts_scaled);
			av_packet_free(&pkt_out);
			return retval;
		}
		outfile->dts_last[stream_idx] = pkt_in_dts;
		pkt_in_corrected_dts = pkt_in_dts - reference_dts_scaled;
		pkt_out->dts = av_rescale_q_rnd(pkt_in_corrected_dts, outfile->input_time_base[stream_idx], out_stream->time_base, AV_ROUND_NEAR_INF|AV_ROUND_PASS_MINMAX);
		if(stream_idx == outfile->reference_stream_idx)
			outfile->record_duration_us = av_rescale_q(pkt_in_corrected_dts, outfile->input_time_base[outfile->reference_stream_idx], (AVRational){1, AV_TIME_BASE});
	}
	{
		int64_t pkt_in_pts = pkt_out->pts;
		if(pkt_in_pts < 0)
		{	// fill invalid (negative) pts value with calculated one
			pkt_in_pts = outfile->pts_last[stream_idx] + outfile->duration_last[stream_idx];
			mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "PTS CORR pts:%lld ptsc:%lld rdur:%lld", pkt_out->pts, pkt_in_pts, outfile->record_duration_us);
		}
		if(pkt_in_pts < reference_dts_scaled){ // !!! we drop frame if pts is less than reference (TODO: we should store first frames in a queue for analyzing)
			mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "PTS REFERENCED DROP %d dts:%lld pts:%lld ref:%lld", stream_idx, pkt_in->dts, pkt_in->pts, reference_dts_scaled);
			av_packet_free(&pkt_out);
			return retval;
		}
		outfile->pts_last[stream_idx] = pkt_in_pts;
		pkt_out->pts = av_rescale_q_rnd((pkt_in_pts - reference_dts_scaled), outfile->input_time_base[stream_idx], out_stream->time_base, AV_ROUND_NEAR_INF|AV_ROUND_PASS_MINMAX);
	}

	outfile->duration_last[stream_idx] = pkt_out->duration;

	pkt_out->stream_index = stream_idx;
	pkt_out->duration = av_rescale_q(pkt_out->duration, outfile->input_time_base[stream_idx], out_stream->time_base);
	pkt_out->pos = -1;

	outfile->record_duration_bytes += pkt_out->size;
	elapsed_timesec = outfile->record_duration_us / AV_TIME_BASE;
	if(outfile->last_framewrite_time_sec != elapsed_timesec)
	{
		in_ffmpfile_file_output_display_progress(outfile, elapsed_timesec);
		outfile->last_framewrite_time_sec = elapsed_timesec;
	}

	if(pkt_out->pts < pkt_out->dts)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "F!!! %d dts:%lld pts:%lld", stream_idx, pkt_out->dts, pkt_out->pts);
	}

	if(stream_idx == 1)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "FILE %d dts_in:%7lld pts_in:%7lld dur_in:%3d dts:%6lld pts:%6lld dur:%2d rec:%5d sden:%d snum:%d",
			stream_idx, pkt_in->dts, pkt_in->pts, pkt_in->duration, pkt_out->dts, pkt_out->pts, pkt_out->duration,
			(int)outfile->record_duration_us / 1000, out_stream->time_base.den, out_stream->time_base.num);
	}

	if(outfile->output_file_avctx)
		retval = av_interleaved_write_frame(outfile->output_file_avctx, pkt_out);
	else
		av_packet_free(&pkt_out);

	return retval;
}

// check dts, append pkt data if dts is NO_PTS, return the collected packet (or the previous complete one), store new packet
static AVPacket *in_ffmpfile_file_output_collect_packet(struct ffmpeg_external_outfile_infos_s *outfile, AVPacket *pkt_in, int stream_idx)
{
	AVPacket *pkt_collector = outfile->input_packet_collector[stream_idx];
	mpxp_bool_t is_new_collector_pkt = TRUE, is_collector_pkt_flushed = FALSE;
	AVPacket *pkt_out = NULL;

	if(pkt_in->dts < 0)
	{
		if(pkt_collector) // add new packet to the stored one
		{
			int stored_pkt_size = pkt_collector->size;
			av_grow_packet(pkt_collector, pkt_in->size);
			memcpy(&pkt_collector->data[stored_pkt_size], pkt_in->data, pkt_in->size);
			pkt_collector->duration += pkt_in->duration;
			if(pkt_collector->duration >= outfile->duration_last[stream_idx]) // TODO: check
			{
				is_collector_pkt_flushed = TRUE;
			}
			is_new_collector_pkt = FALSE;
		}
	}
	else
	{
		is_collector_pkt_flushed = TRUE;
	}

	if(is_collector_pkt_flushed)
	{
		pkt_out = pkt_collector;
		outfile->input_packet_collector[stream_idx] = NULL;
	}

	if(is_new_collector_pkt)
	{
		pkt_collector = av_packet_clone(pkt_in);
	    if(!pkt_collector) {
	        mpxplay_debugf(MPXPLAY_DEBUGOUT_FILEOUT, "Error occurred when av_packet_clone");
	        return pkt_collector;
	    }
	    outfile->input_packet_collector[stream_idx] = pkt_collector;
	}

	return pkt_out;
}

#endif // MPXPLAY_GUI_QT

#endif // MPXPLAY_LINK_INFILE_FF_MPEG
