//**************************************************************************
//*                     This file is part of the                           *
//*                    Mpxplay-MMC - video player.                         *
//*                The source code of Mpxplay-MMC is                       *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: digital TV (DVB) handling by BDA driver

//#define MPXPLAY_USE_DEBUGF
#define MPXPLAY_DEBUG_OUTPUT     stdout
#define MPXPLAY_DEBUGOUT_OPTIONS NULL // stdout
#define MPXPLAY_DEBUGOUT_PROGEV  stdout
#define MPXPLAY_DEBUGOUT_READ    NULL
#define MPXPLAY_DEBUGOUT_ERROR   stderr

#include "mpxplay.h"

#if defined(MPXPLAY_LINK_INFILE_FF_MPEG) && defined(MPXPLAY_WIN32)

#include "diskdriv.h"
#include "dtv_drv.h"
#include "tcpcomon.h"
#include "control/cntfuncs.h"
#include "display/display.h"

#define DRVDTV_EVENTS_AUTO_REFRESH_TIMESECS 60 // update interval of auto dtv-drive refresh of programs/events

typedef enum {DTVDRV_DIRTYPE_ROOT = 0, DTVDRV_DIRTYPE_CONTROL, DTVDRV_DIRTYPE_CHANNELS, DTVDRV_DIRTYPE_SCANDEV, DTVDRV_DIRTYPE_NUM} dtvdrv_currdir_type;

typedef struct dtvdrv_virtual_dirlist{
	const char *dirname[3];
}dtvdrv_virtual_dirlist;

static const struct dtvdrv_virtual_dirlist dtvdrv_dir_list_names[DTVDRV_DIRTYPE_NUM] = {
 { "CONTROL", NULL},                 // DTVDRV_DIRTYPE_ROOT
 { "CHANNELS", "SCAN_DEVICE", NULL}, // DTVDRV_DIRTYPE_CONTROL
 { NULL},                            // DTVDRV_DIRTYPE_CHANNELS
 { NULL }                            // DTVDRV_DIRTYPE_SCANDEV
};

// shall match with DTVDRV_DIRTYPE_xxx
static const char *dtvdrv_path_names[DTVDRV_DIRTYPE_NUM] = { "", "CONTROL", "CONTROL"PDS_DIRECTORY_SEPARATOR_STR"CHANNELS", "CONTROL"PDS_DIRECTORY_SEPARATOR_STR"SCAN_DEVICE" };

// datas belonging to a dtv-drive
typedef struct dtvdrive_info_s
{
	mpxp_int32_t (*psifn_callback)(void *mdds,mpxp_uint32_t funcnum,void *argp1,void *argp2);
	void *psifn_cb_data;
	struct mpxplay_dvb_epg_protocol_data_s *ff_protocol_data;     // current protocol at findfirst/findnext
	struct mpxplay_dvb_epg_frequency_data_s *ff_frequency_data;
	struct mpxplay_dvb_epg_program_data_s *ff_program_data;
	unsigned int ff_subdirelem;                            // number of elem in dtvdrv_virtual_dirlist at findfirst/findnext
	dtvdrv_currdir_type currdir_dirtype;                   // one of the DTVDRV_DIRTYPE_ (depending on the current subdir)
}dtvdrive_info_s;

// datas belonging to a dtv-file
typedef struct dtvfile_info_s
{
	mpxp_uint32_t flags;
	unsigned int openmode;
	dtvdrive_info_s *dtvdi;
	dvb_device_t *dtv_device;
	char *filename_store;
	mpxp_int32_t (*psifn_callback)(void *mdds,mpxp_uint32_t funcnum,void *argp1,void *argp2);
	void *psifn_cb_data;
	mpxp_int64_t filesize, filepos;
	char content_artist[MAX_STRLEN]; // a fake title assembled from mhz and time
}dtvfile_info_s;

static const char *drvdtv_protocol_names_list[BDADEVTYPE_NUM] =
{
	"atsc:",   // BDADEVTYPE_ATSC
	"cqam:",   // BDADEVTYPE_CQAM
	"dvb-c:",  // BDADEVTYPE_DVB_C
	"dvb-c2:", // BDADEVTYPE_DVB_C2
	"dvb-s:",  // BDADEVTYPE_DVB_S
	"dvb-s2:", // BDADEVTYPE_DVB_S2
	"dvb-t:",  // BDADEVTYPE_DVB_T
	"dvb-t2:", // BDADEVTYPE_DVB_T2
	"isdb_c:", // BDADEVTYPE_ISDB_C
	"isdb_s:", // BDADEVTYPE_ISDB_S
	"isdb_t:"  // BDADEVTYPE_ISDB_T
};

const char *mpxplay_drvdtv_protocol_family_list[5] =
{
	"atsc:",   // BDADEVTYPE_ATSC
	"cqam:",   // BDADEVTYPE_CQAM
	"dvb-c:",  // BDADEVTYPE_DVB_C
	"dvb-s:",  // BDADEVTYPE_DVB_S
	"dvb-t:"   // BDADEVTYPE_DVB_T
};

static const char *drvdtv_protocol_name_file = "file:";

extern unsigned long mpxplay_signal_events, mpxplay_config_dvbepg_control_flags;

static const char *dtv_drive_protocol_name = "dtv:"; // for FTP open
static const char dtv_drive_session_text[] = "Session=Digital TV Channels (DVB)\nProtocol=dtv\n"; // for FTP open
static const char *dtv_drive_demuxer_file_extension = "TS"; // for FFmpeg demux
static void *drvdtv_mutexhnd_open = NULL;
struct dtvdrive_shared_data_s mpxplay_drvdtv_shared_drive_datas;

static void dtvdrive_drive_database_update_all_programs_timed(void *drivehand_data);
static void dtvdrive_file_close(void *filehand_data);

//----------------------------------------------------------------------------------------

enum { DRVDTV_OPTIONTYPE_STRING = 0, DRVDTV_OPTIONTYPE_CHAR, DRVDTV_OPTIONTYPE_INTEGER, DRVDTV_OPTIONTYPE_HEXA};

typedef struct dtv_option_list_s
{
	const char *option_name;
	unsigned int option_type;
	unsigned int cfg_pos;
}dtv_option_list_s;

static const struct dtv_option_list_s dtv_options[] =
{
	{"dvb-adapter",           DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,adapter_num)},
	{"dvb-frequency",         DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,frequency)},
	{"dvb-inversion",         DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,inversion)},
	{"dvb-network-name",      DRVDTV_OPTIONTYPE_STRING,  offsetof(dvb_device_t,network_name)},

	{"dvb-program-id",        DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,program_number_id)}, // program_id is used in FFmpeg only

	{"dvb-scan-begin",        DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,scan_freq_begin)},   // to start a device scan (kHz)
	{"dvb-scan-end",          DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,scan_freq_end)},     //
	{"dvb-scan-bandwidth",    DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,scan_freq_bandwidth)}, //
	{"dvb-scan-epgload",      DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,scan_do_epgload)},   // load epg for scanned channels (and scan device if it's not done yet)

	// ATSC
	{"dvb-physical-channel",  DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,physical_channel)},
	{"dvb-major-channel",     DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,major_channel)},
	{"dvb-minor-channel",     DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,minor_channel)},

	// DVB-S, ISDB_S
	{"dvb-polarization",      DRVDTV_OPTIONTYPE_CHAR,    offsetof(dvb_device_t,lnb_polarization)},
	{"dvb-srate",             DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,srate)},
	{"dvb-lnb-low",           DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,lowf)},
	{"dvb-lnb-high",          DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,highf)},
	{"dvb-lnb-switch",        DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,switchf)},
	{"dvb-network-id",        DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,network_id)},
	{"dvb-azimuth",           DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,azimuth)},
	{"dvb-elevation",         DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,elevation)},
	{"dvb-longitude",         DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,longitude)},
	{"dvb-range",             DRVDTV_OPTIONTYPE_STRING,  offsetof(dvb_device_t,range)},
	{"dvb-fec",               DRVDTV_OPTIONTYPE_HEXA,    offsetof(dvb_device_t,fec)},

	// DVB-T
	{"dvb-bandwidth",         DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,bandwidth)},
	{"dvb-transmission",      DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,transmission)},
	{"dvb-hierarchy",         DRVDTV_OPTIONTYPE_INTEGER, offsetof(dvb_device_t,hierarchy)},
	{"dvb-modulation",        DRVDTV_OPTIONTYPE_STRING,  offsetof(dvb_device_t,mod)},
	{"dvb-code-rate-hp",      DRVDTV_OPTIONTYPE_HEXA,    offsetof(dvb_device_t,fec_hp)},
	{"dvb-code-rate-lp",      DRVDTV_OPTIONTYPE_HEXA,    offsetof(dvb_device_t,fec_lp)},
	{"dvb-guard",             DRVDTV_OPTIONTYPE_HEXA,    offsetof(dvb_device_t,guard)},
#ifdef MPXPLAY_DRVTV_ENABLE_DVBT2
	// DVB-T2
	{"dvb-plp-id",            DRVDTV_OPTIONTYPE_INTEGER,  offsetof(dvb_device_t,plp_id)},
#endif
	{NULL,0,0}
};

int mpxplay_drvdtv_get_protocol_id(char *protocol_name)
{
	int i, protocol_id = -1;

	if(strnicmp(protocol_name, drvdtv_protocol_name_file, sizeof("file:") - 1) == 0)
		return MPXPLAY_DRVDTV_BDADEVTYPE_LOCALFILE;

	for(i = 0; i < BDADEVTYPE_NUM; i++)
	{
		char *prot = (char *)drvdtv_protocol_names_list[i];
		if(strnicmp(prot, protocol_name, pds_strlen(prot)) == 0)
		{
			protocol_id = i;
			break;
		}
	}

	return protocol_id;
}

const char *mpxplay_drvdtv_get_protocol_name(unsigned int protocol_id)
{
	if(protocol_id >= BDADEVTYPE_NUM)
		return drvdtv_protocol_name_file;
	return drvdtv_protocol_names_list[protocol_id];
}

static int dtvdrive_get_virtual_directory_type(char *path)
{
	int i;

	if(!path || !*path)
		return -1;

	if(pds_getdrivenum_from_path(path) >= 0)
	  path += sizeof(PDS_DIRECTORY_DRIVE_STR) - 1;

	while(*path == PDS_DIRECTORY_SEPARATOR_CHAR)
		path++;

	for(i = 0; i < DTVDRV_DIRTYPE_NUM; i++)
	{
		if(pds_stricmp(path, (char *)dtvdrv_path_names[i]) == 0)
		{
			return i;
		}
	}

	return -1;
}

static mpxp_bool_t dtvdrive_parse_options(char *pathname, char **opt_store, dvb_device_t *dev_cfg)
{
	char *cmdline_dvb_options_str;
	int protocol_id;

	mpxplay_debugf(MPXPLAY_DEBUGOUT_OPTIONS, "dtvdrive_parse_options BEGIN %s", pathname);

	if(!pathname || !pathname[0])
		return FALSE;

	*opt_store = cmdline_dvb_options_str = (char *)pds_malloc(pds_strlen(pathname) + 1);
	if(!cmdline_dvb_options_str)
		return FALSE;
	pds_strcpy(cmdline_dvb_options_str, pathname);

	protocol_id = mpxplay_drvdtv_get_protocol_id(cmdline_dvb_options_str);
	if(protocol_id < 0)
		return FALSE;

	dev_cfg->protocol_number_id = protocol_id;

	cmdline_dvb_options_str += pds_strlen((char *)mpxplay_drvdtv_get_protocol_name(protocol_id));

	if(cmdline_dvb_options_str <= *opt_store)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_ERROR, "dtvdrive_parse_options: no valid protocol found!");
		return FALSE;
	}

	while((cmdline_dvb_options_str[0] == '/') || (cmdline_dvb_options_str[0] == ':'))
		cmdline_dvb_options_str++;

	while(cmdline_dvb_options_str && cmdline_dvb_options_str[0])
	{
		char *next = pds_strchr(cmdline_dvb_options_str, ':'), *value_str;
		mpxp_bool_t found = FALSE;
		if(next)
			*next++ =0;
		value_str = pds_strchr(cmdline_dvb_options_str,'=');
		//mpxplay_debugf(MPXPLAY_DEBUGOUT_OPTIONS, "optstr: %s val: %s", cmdline_dvb_options_str, value_str);
		if(value_str)
		{
			const struct dtv_option_list_s *opts = &dtv_options[0];
			*value_str++ = 0;
			pds_strcutspc(cmdline_dvb_options_str);
			pds_strcutspc(value_str);
			do{
				if( (pds_stricmp(cmdline_dvb_options_str, (char *)opts->option_name) == 0)
				 || (pds_stricmp(cmdline_dvb_options_str, (char *)&opts->option_name[sizeof("dvb-")-1]) == 0) // !!!
				){
					void *cfgval_ptr = (void *)dev_cfg + opts->cfg_pos;
					switch(opts->option_type)
					{
						case DRVDTV_OPTIONTYPE_STRING:
							*((char **)cfgval_ptr) = value_str;
							break;
						case DRVDTV_OPTIONTYPE_CHAR:
							*((char *)cfgval_ptr) = value_str[0];
							break;
						case DRVDTV_OPTIONTYPE_INTEGER:
							*((mpxp_int32_t *)cfgval_ptr) = pds_atol(value_str);
							break;
						case DRVDTV_OPTIONTYPE_HEXA:
							*((mpxp_int32_t *)cfgval_ptr) = pds_atol16(value_str);
							break;
					}
					found = TRUE;
					mpxplay_debugf(MPXPLAY_DEBUGOUT_OPTIONS, "optstr: \"%s\" value: %s", cmdline_dvb_options_str, value_str);
					break;
				}
				opts++;
			}while(opts->option_name);
		}
		if(!found)
		{
			char sout[256];
			snprintf(sout, sizeof(sout), "DRVDTV: unknown/invalid option: \"%s\"", cmdline_dvb_options_str);
			display_timed_message(sout);
		}

		cmdline_dvb_options_str = next;
	}

	if(!dev_cfg->frequency)
	{
		mpxplay_debugf(MPXPLAY_DEBUGOUT_ERROR, "frequency is not parsed ZERO! %s", pathname);
	}

	mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "dtvdrive_parse_options END %s", pathname);

	return TRUE;
}

//---------------------------------------------------------------------------------------------------------------------------------
// assemble standardized dvb program entry filename (to identify a playlist entry with a program)
void mpxplay_dtvdrv_assemble_filename(char *destbuf, unsigned int buflen, mpxp_uint32_t protocol_id, mpxp_uint32_t frequency, mpxp_uint32_t program_id)
{
	snprintf(destbuf, buflen, "%s//dvb-frequency=%d:dvb-program-id=%d", mpxplay_drvdtv_get_protocol_name(protocol_id), frequency, program_id);
}

// disassemble standardized dvb program entry filename, return program_id (to identify a playlist entry with a program)
mpxp_int32_t mpxplay_dtvdrv_parse_filename(char *filename, int *protocol_id, mpxp_int32_t *freqency_hz)
{
	mpxp_int32_t program_id;
	char *opt_store = NULL;
	dvb_device_t dev_tmp;

	pds_memset(&dev_tmp, 0, sizeof(dev_tmp));
	dev_tmp.program_number_id = MPXPLAY_DISKDRIV_CFGERROR_INVALID_DRIVE;

	if(dtvdrive_parse_options(filename, &opt_store, &dev_tmp))
	{
		if(protocol_id)
			*protocol_id = dev_tmp.protocol_number_id;
		if(freqency_hz)
			*freqency_hz = dev_tmp.frequency;
	}
	program_id = dev_tmp.program_number_id;

	if(opt_store)
		pds_free(opt_store);

	return program_id;
}

// get program id from a specified filename, if duration argument is given, search in the global database for the current event, else use the related file infos only
static int dtvdrv_get_programid_and_duration(struct dtvfile_info_s *dtvfi, char *chkfilename, mpxp_uint64_t *duration)
{
	long program_id = MPXPLAY_DISKDRIV_CFGERROR_INVALID_DRIVE;
	char *opt_store = NULL;
	dvb_device_t dev_tmp;

	pds_memset(&dev_tmp, 0, sizeof(dev_tmp));
	dev_tmp.program_number_id = MPXPLAY_DISKDRIV_CFGERROR_INVALID_DRIVE;

	if(dtvdrive_parse_options(chkfilename, &opt_store, &dev_tmp))
	{
		if(duration)
		{
			struct mpxplay_dvb_epg_program_data_s *prog_data = mpxplay_dtvdrv_database_program_search(dev_tmp.protocol_number_id, dev_tmp.frequency, dev_tmp.program_number_id);
			if(prog_data)
			{
				struct mpxplay_dvb_epg_event_data_s *curr_event = mpxplay_dtvdrv_database_get_current_event(prog_data);
				if(curr_event)
				{
					mpxp_int32_t dur_secs = pds_timeval_to_seconds(curr_event->event_duration_time);
					if(dur_secs < MPXPLAY_DVBEPG_EVENT_DURATION_MIN)
						dur_secs = MPXPLAY_DVBEPG_EVENT_DURATION_MIN;
					*duration = (mpxp_uint64_t)dur_secs;
				}
				program_id = dev_tmp.program_number_id;
			}
		}
		else if((dev_tmp.protocol_number_id == dtvfi->dtv_device->protocol_number_id) && (dev_tmp.adapter_num == dtvfi->dtv_device->adapter_num) && (dev_tmp.frequency == dtvfi->dtv_device->frequency))
		{
			program_id = (dev_tmp.program_number_id > 0)? dev_tmp.program_number_id : 0; // if program not found, but freq is valid
		}
	}
	if(opt_store)
	{
		pds_free(opt_store);
	}
	return program_id;
}

//---------------------------------------------------------------------------------------------------------------------------------

static unsigned int dtvdrive_drive_check(char *pathname)
{
	if(pds_strlicmp(pathname, (char *)dtv_drive_protocol_name) == 0)
		return 1;
	return 0;
}

static long dtvdrive_drive_config(void *drive_data,unsigned long funcnum,void *argp1,void *argp2)
{
	struct dtvdrive_info_s *dtis = (struct dtvdrive_info_s *)drive_data;
	struct dtvfile_info_s *dtvfi;
	char strtmp[256];

	switch(funcnum)
	{
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_CLOSEDRIVE:
			mpxplay_dtv_bda_resources_stop();
			mpxplay_drvdtv_database_close();
			mpxplay_dtv_bda_resources_close();
			return MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_ISALWAYSSTREAM:
			return TRUE;
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_CHKBUFBLOCKBYTES:
			return 262144;
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_PREBUFFERSIZE:
			return 32 * 1048576;                               // ??? (should be time based, not byte)
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_PREREADBUFBYTES:  //
			return 1 * 1048576;                                // ??? (should be time based, not byte)
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_CONSTSESSIONS:
			if(!argp1 || !argp2)
				return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;
			*(const char **)argp1 = &dtv_drive_session_text[0];
			*((unsigned long *)argp2) = sizeof(dtv_drive_session_text);
			return MPXPLAY_DISKDRIV_CFGERROR_GET_OK;
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_UTFTYPE:
		    return MPXPLAY_TEXTCONV_TYPE_UTF8;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_GET_ISNONSEEKABLE:
			return TRUE;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_GET_ISLIVESTREAM:
			return TRUE;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_GET_ISASYNCREAD:
			return TRUE;
	}

	if(!dtis)
		return MPXPLAY_DISKDRIV_CFGERROR_INVALID_DRIVE;

	switch(funcnum)
	{
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_ISDIREXISTS:
			return (dtvdrive_get_virtual_directory_type((char *)argp1) >= 0)? 1 : 0;
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_SET_CALLBACKPSIFN:
			dtis->psifn_callback = argp1;
			dtis->psifn_cb_data = argp2;  // can be NULL
			return MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_DRVTYPENAME:
			if(!argp1 || !argp2)
				return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;
			pds_strncpy((char *)argp1,"Digital TV",*((unsigned long *)argp2));
			return MPXPLAY_DISKDRIV_CFGERROR_GET_OK;
		case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_REALLYFULLPATH:
			if(!argp1 || !argp2)
				return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;
			tcpcomon_str_localname_to_remote(strtmp,(char *)argp1,sizeof(strtmp));
			if(!strtmp[0] || (strtmp[0]==PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP))   // assumed directory name
				snprintf((char *)argp1, MAX_PATHNAMELEN, "%s%s%s%s", dtv_drive_protocol_name, PDS_DIRECTORY_SEPARATOR_STR_UNXFTP, ((strtmp[0]==PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP)? "" : PDS_DIRECTORY_SEPARATOR_STR_UNXFTP), strtmp);
			else
				pds_strcpy((char *)argp1,(char *)argp2);
			return MPXPLAY_DISKDRIV_CFGERROR_GET_OK;
	}

	// drvdtv_file handling from here
	dtvfi = drive_data;

	switch(funcnum)
	{
		case MPXPLAY_DISKFILE_CFGFUNCNUM_CMD_TERMINATE:
			funcbit_enable(dtvfi->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE);
			if(dtvfi->dtv_device)
			{
				funcbit_enable(dtvfi->dtv_device->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE);
				funcbit_disable(dtvfi->dtv_device->flags, MPXPLAY_DRVDTV_FLAG_READWAIT);
			}
			return MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
	}

	if(!argp1)
		return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;

	switch(funcnum)
	{
		case MPXPLAY_DISKFILE_CFGFUNCNUM_SET_READWAIT:
			if(!dtvfi->dtv_device)
				return MPXPLAY_DISKDRIV_CFGERROR_INVALID_DRIVE;
			if(*((unsigned long *)argp1))
				funcbit_enable(dtvfi->dtv_device->flags, MPXPLAY_DRVDTV_FLAG_READWAIT);
			else
				funcbit_disable(dtvfi->dtv_device->flags, MPXPLAY_DRVDTV_FLAG_READWAIT);
			return MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_GET_CONTENTTYPEEXT:
			*((char **)argp1) = (char *)dtv_drive_demuxer_file_extension;
			return MPXPLAY_DISKDRIV_CFGERROR_GET_OK;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_SET_CALLBACKPSIFN:
			dtvfi->psifn_callback = argp1;
			dtvfi->psifn_cb_data = argp2;  // can be NULL
			return MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_GET_PROGRAMID:
			return dtvdrv_get_programid_and_duration(dtvfi, (char *)argp1, (mpxp_uint64_t *)argp2);
	}

	if(!argp2)
		return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;

	switch(funcnum)
	{
		case MPXPLAY_DISKFILE_CFGFUNCNUM_GET_CONTENT_ARTIST:
			pds_strncpy((char *)argp1,dtvfi->content_artist,*((unsigned long *)argp2));
			((char *)argp1)[*((unsigned long *)argp2) - 1] = 0;
			return MPXPLAY_DISKDRIV_CFGERROR_GET_OK;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_GET_FNBYPRGID:
			if(dtvfi->dtv_device)
			{
				mpxplay_dtvdrv_assemble_filename((char *)argp1, MAX_PATHNAMELEN, dtvfi->dtv_device->protocol_number_id, dtvfi->dtv_device->frequency, *((mpxp_uint32_t *)argp2));
				return MPXPLAY_DISKDRIV_CFGERROR_GET_OK;
			}
			return MPXPLAY_DISKDRIV_CFGERROR_INVALID_DRIVE;

	}

	return MPXPLAY_DISKDRIV_CFGERROR_UNSUPPFUNC;
}

static void dtvdrive_drive_unmount(void *drivehand_data)
{
	struct dtvdrive_info_s *dtis = (struct dtvdrive_info_s *)drivehand_data;
	if(dtis)
	{
		mpxplay_timer_deletefunc(&dtvdrive_drive_database_update_all_programs_timed, (void *)dtis);
		pds_free(dtis);
	}
}

static void *dtvdrive_drive_mount(char *pathname)
{
	struct dtvdrive_info_s *dtis;

	if(!dtvdrive_drive_check(pathname))
		return NULL;

	mpxplay_drvdtv_database_load_from_file();

	dtis = (struct dtvdrive_info_s *)pds_calloc(1, sizeof(*dtis));
	if(!dtis)
		return dtis;

	mpxplay_timer_addfunc(&dtvdrive_drive_database_update_all_programs_timed, (void *)dtis, MPXPLAY_TIMERTYPE_REPEAT, mpxplay_timer_secs_to_counternum(DRVDTV_EVENTS_AUTO_REFRESH_TIMESECS)); // FIXME: initiated from all opening dtv drive (tab) parallel (they do the same)

	return dtis;

err_out_mount:
	dtvdrive_drive_unmount((void *)dtis);
	return NULL;
}

// get next protocol entry from database
static mpxp_bool_t dtvdrive_drive_database_find_next_protocol(struct dtvdrive_info_s *dtis)
{
	if(dtis->ff_protocol_data)
	{
		dtis->ff_protocol_data = dtis->ff_protocol_data->next_protocol;
	}
	else
	{
		dtis->ff_protocol_data = mpxplay_drvdtv_shared_drive_datas.protocol_data_chain;
	}
	if(!dtis->ff_protocol_data)
	{
		return FALSE;
	}
	return TRUE;
}

// get next protocol family entry from the database
static mpxp_bool_t dtvdrive_drive_database_find_next_prot_family(struct dtvdrive_info_s *dtis)
{
	mpxp_bool_t result = FALSE;
	while(!result && dtvdrive_drive_database_find_next_protocol(dtis))
	{
		switch(dtis->ff_protocol_data->protocol_id)
		{
			case BDADEVTYPE_ATSC:
			case BDADEVTYPE_CQAM:
			case BDADEVTYPE_DVB_C:
			case BDADEVTYPE_DVB_S:
			case BDADEVTYPE_DVB_T: result = TRUE;
		}
	}
	return result;
}

// get next freqency entry from database
static mpxp_bool_t dtvdrive_drive_database_find_next_frequency(struct dtvdrive_info_s *dtis)
{
	do{
		if(dtis->ff_frequency_data)
		{
			dtis->ff_frequency_data = dtis->ff_frequency_data->next_frequency;
		}
		if(!dtis->ff_frequency_data)
		{
			if(!dtvdrive_drive_database_find_next_protocol(dtis))
				return FALSE;
			dtis->ff_frequency_data = dtis->ff_protocol_data->frequency_data_chain;
		}
		if(dtis->ff_frequency_data)
		{
			if(funcbit_test(dtis->ff_frequency_data->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_SDT)) // TODO: check
				break;
		}
	}while(1);

	return TRUE;
}

// get next program entry from database
static mpxp_bool_t dtvdrive_drive_database_find_next_program(struct dtvdrive_info_s *dtis)
{
	do{
		if(dtis->ff_program_data)
		{
			dtis->ff_program_data = dtis->ff_program_data->next_program;
		}
		if(!dtis->ff_program_data)
		{
			if(!dtvdrive_drive_database_find_next_frequency(dtis))
				return FALSE;
			dtis->ff_program_data = dtis->ff_frequency_data->program_data_chain;
		}
		if(mpxplay_dtvdrv_database_check_program_validity(dtis->ff_frequency_data, dtis->ff_program_data, mpxplay_config_dvbepg_control_flags))
		{
			break;
		}
	}while(TRUE);

	return TRUE;
}

static mpxp_bool_t dtvdrive_drive_prepare_find_next_dirname(struct dtvdrive_info_s *dtis, struct pds_find_t *ffblk)
{
	const char *subdirname = dtvdrv_dir_list_names[dtis->currdir_dirtype].dirname[dtis->ff_subdirelem];
	if(!subdirname)
		return FALSE;
	pds_strcpy(&ffblk->name[0], (char *)subdirname);
	ffblk->attrib = _A_SUBDIR;
	dtis->ff_subdirelem++;
	return TRUE;
}

// assemble standardized dvb channel entry filename
static void mpxplay_dtvdrv_assemble_channel_filename(char *destbuf, unsigned int buflen, struct mpxplay_dvb_epg_protocol_data_s *prot_data, struct mpxplay_dvb_epg_frequency_data_s *freq_data)
{
	snprintf(destbuf, buflen, "%s//dvb-frequency=%d", mpxplay_drvdtv_get_protocol_name(prot_data->protocol_id), freq_data->frequency);
}

// assemble standardized dvb scan entry filename
static void mpxplay_dtvdrv_assemble_scan_filename(char *destbuf, unsigned int buflen, struct mpxplay_dvb_epg_protocol_data_s *prot_data)
{
	mpxp_uint32_t scan_bandwidth = (prot_data->freq_bandwidth)? prot_data->freq_bandwidth : MPXPLAY_DRVDTV_FREQUENCY_BANDWIDTH;
	snprintf(destbuf, buflen, "%s//dvb-scan-begin=%d:dvb-scan-end=%d:dvb-scan-bandwidth=%d",
			mpxplay_drvdtv_get_protocol_name(prot_data->protocol_id), prot_data->scan_freq_begin / 1000, prot_data->scan_freq_end / 1000, scan_bandwidth / 1000);
}

static int dtvdrive_drive_prepare_find_next_filename(struct dtvdrive_info_s *dtis, struct pds_find_t *ffblk)
{
	if(dtvdrive_drive_prepare_find_next_dirname(dtis, ffblk))
		return 0;

	ffblk->attrib = 0;

	switch(dtis->currdir_dirtype)
	{
		case DTVDRV_DIRTYPE_ROOT:
			if(dtvdrive_drive_database_find_next_program(dtis))
			{
				mpxplay_dtvdrv_assemble_filename(&ffblk->name[0], sizeof(ffblk->name), dtis->ff_protocol_data->protocol_id, dtis->ff_frequency_data->frequency, dtis->ff_program_data->program_id);
				ffblk->size = MPXPLAY_DISKDRIV_FILESIZE_UNKNOWN;
				return 0;
			}
			break;
		case DTVDRV_DIRTYPE_CHANNELS:
			if(dtvdrive_drive_database_find_next_frequency(dtis))
			{
				mpxplay_dtvdrv_assemble_channel_filename(&ffblk->name[0], sizeof(ffblk->name), dtis->ff_protocol_data, dtis->ff_frequency_data);
				ffblk->size = MPXPLAY_DISKDRIV_FILESIZE_UNKNOWN;
				return 0;
			}
			break;
		case DTVDRV_DIRTYPE_SCANDEV:
			if(dtvdrive_drive_database_find_next_prot_family(dtis))
			{
				struct mpxplay_dvb_epg_protocol_data_s *prot_data = dtis->ff_protocol_data;
				mpxplay_dtvdrv_assemble_scan_filename(&ffblk->name[0], sizeof(ffblk->name), dtis->ff_protocol_data);
				ffblk->size = MPXPLAY_DISKDRIV_FILESIZE_UNKNOWN;
				return 0;
			}
			break;
	}
	return 1;
}

// update program name and event in playlist editor of application
static void dtvdrive_drive_ff_update_entry_metadata(struct dtvdrive_info_s *dtis, struct pds_find_t *ffblk)
{
	char sout[256];
	if(dtis->psifn_callback && dtis->ff_protocol_data)
	{
		switch(dtis->currdir_dirtype)
		{
			case DTVDRV_DIRTYPE_ROOT:
				if(dtis->ff_frequency_data && dtis->ff_program_data)
				{
					struct mpxplay_dvb_epg_program_data_s *prog_data;
					struct mpxplay_dvb_epg_event_data_s *curr_event;
					mpxp_ptrsize_t duration_ms = 1000;
					mpxplay_dtvdrv_assemble_filename(&ffblk->name[0], sizeof(ffblk->name), dtis->ff_protocol_data->protocol_id, dtis->ff_frequency_data->frequency, dtis->ff_program_data->program_id);
					dtis->psifn_callback(dtis->psifn_cb_data, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_I3I_ARTIST, &ffblk->name[0], dtis->ff_program_data->program_name);
					prog_data = mpxplay_dtvdrv_database_program_search(dtis->ff_protocol_data->protocol_id, dtis->ff_frequency_data->frequency, dtis->ff_program_data->program_id);
					curr_event = mpxplay_dtvdrv_database_get_current_event(prog_data);
					if(curr_event)
					{
						duration_ms *= pds_timeval_to_seconds(curr_event->event_duration_time);
						dtis->psifn_callback(dtis->psifn_cb_data, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_I3I_TITLE, (void *)&ffblk->name[0], (void *)curr_event->event_shortdesc_name);
					}
					dtis->psifn_callback(dtis->psifn_cb_data, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_TIMELENMS, (void *)&ffblk->name[0], (void *)duration_ms);
				}
				break;
			case DTVDRV_DIRTYPE_CHANNELS:
				if(!dtis->ff_frequency_data)
					break;
				// @suppress("No break at end of case")
			case DTVDRV_DIRTYPE_SCANDEV:
			{
				unsigned int len;
				char prot_name[32];
				pds_strcpy(&prot_name[0], (char *)mpxplay_drvdtv_get_protocol_name(dtis->ff_protocol_data->protocol_id));
				len = pds_strlen(prot_name);
				if(len > 1)
				{
					prot_name[len - 1] = 0; // cut ':' from the end
					pds_str_uppercase(prot_name); // convert to uppercase
				}

				if(dtis->currdir_dirtype == DTVDRV_DIRTYPE_CHANNELS)
				{
					mpxplay_dtvdrv_assemble_channel_filename(&ffblk->name[0], sizeof(ffblk->name), dtis->ff_protocol_data, dtis->ff_frequency_data);
					snprintf(sout, sizeof(sout), "%s channel %dMhz", prot_name, dtis->ff_frequency_data->frequency / 1000000);
				}
				else
				{
					mpxplay_dtvdrv_assemble_scan_filename(&ffblk->name[0], sizeof(ffblk->name), dtis->ff_protocol_data);
					snprintf(sout, sizeof(sout), "Scan %s device from %dMhz to %dMhz", prot_name, dtis->ff_protocol_data->scan_freq_begin / 1000000, dtis->ff_protocol_data->scan_freq_end / 1000000);
				}
				dtis->psifn_callback(dtis->psifn_cb_data, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_I3I_ARTIST, (void *)&ffblk->name[0], (void *)&sout[0]);
				break;
			}
		}
	}
}

static unsigned int dtvdrive_findfirst(void *drivehand_data, char *pathname, unsigned int attrib, struct pds_find_t *ffblk)
{
	struct dtvdrive_info_s *dtis = (struct dtvdrive_info_s *)drivehand_data;
	if(!dtis)
		return 1;
	if(!ffblk)
		return 1;
	dtis->ff_protocol_data = NULL;
	dtis->ff_frequency_data = NULL;
	dtis->ff_program_data = NULL;
	dtis->ff_subdirelem = 0;
	return dtvdrive_drive_prepare_find_next_filename(dtis, ffblk);
}

static unsigned int dtvdrive_findnext(void *drivehand_data, struct pds_find_t *ffblk)
{
	struct dtvdrive_info_s *dtis = (struct dtvdrive_info_s *)drivehand_data;
	dtvdrive_drive_ff_update_entry_metadata(dtis, ffblk);
	return dtvdrive_drive_prepare_find_next_filename(dtis, ffblk);
}

static void dtvdrive_findclose(void *drivehand_data, struct pds_find_t *ffblk)
{
	struct dtvdrive_info_s *dtis = (struct dtvdrive_info_s *)drivehand_data;
	dtvdrive_drive_ff_update_entry_metadata(dtis, ffblk);
}

static char *dtvdrive_getcwd(void *drivehand_data, char *buf, unsigned int buflen)
{
	struct dtvdrive_info_s *dtis = (struct dtvdrive_info_s *)drivehand_data;
	if(!buf || !buflen)
		return buf;
	snprintf(buf,buflen,"%s", dtvdrv_path_names[dtis->currdir_dirtype]);
	return buf;
}

static int dtvdrive_chdir(void *drivehand_data, char *path)
{
	struct dtvdrive_info_s *dtis = (struct dtvdrive_info_s *)drivehand_data;
	int i;

	i = dtvdrive_get_virtual_directory_type(path);
	if(i >= 0)
	{
		dtis->currdir_dirtype = i;
		return 0;
	}

	return -1;
}

static int dtvdrive_mkdir(void *drivehand_data, char *path)
{
 return -1;
}

static int dtvdrive_rmdir(void *drivehand_data, char *path)
{
 return -1;
}

static int dtvdrive_rename(void *drivehand_data, char *oldfilename,char *newfilename)
{
 return -1;
}

static int dtvdrive_unlink(void *drivehand_data, char *filename)
{
 return -1;
}

// update all program names and events in playlist editor of application, once in a minute
static void dtvdrive_drive_database_update_all_programs_timed(void *drivehand_data)
{
	struct pds_find_t ffblk;

	struct dtvdrive_info_s *dtis = (struct dtvdrive_info_s *)drivehand_data;
	if(!dtis)
		return;

	pds_memset(&ffblk, 0, sizeof(ffblk));

	if(dtvdrive_findfirst(drivehand_data, NULL, 0, &ffblk)) // TODO: not protected to parallel execution (probably there's no such one currently)
		return;

	while(!dtvdrive_findnext(drivehand_data, &ffblk)) {}

	dtvdrive_findclose(drivehand_data, &ffblk);
}

//---------------------------------------------------------------------
static long dtvdrv_file_epg_callback_func(void *file_data, unsigned long funcnum, void *argp1, void *argp2)
{
	struct dtvfile_info_s *dtvfi = (struct dtvfile_info_s *)file_data;
	dvb_device_t *dtv_device = dtvfi->dtv_device;
	struct mpxplay_dvb_epg_frequency_data_s *freq_data;
	struct mpxplay_dvb_epg_program_data_s *prog_data;
	struct mpxplay_dvb_epg_event_data_s *event_data_in, *event_data_new;
	int retval = MPXPLAY_DISKDRIV_CFGERROR_CANNOT_DO, len, program_id;
	mpxp_ptrsize_t duration_ms;
	char strtmp[MAX_PATHNAMELEN];

	if(!dtv_device || !argp1)
		return retval;

	program_id = *((mpxp_uint32_t *)argp1);
	if(!program_id)
		return retval;

	switch(funcnum)
	{
		case MPXPLAY_DISKFILE_CFGFUNCNUM_SET_PROGRAMNAME:
			prog_data = mpxplay_dtvdrv_database_program_new(dtv_device->protocol_number_id, dtv_device->frequency, program_id, (char *)argp2);
			if(!prog_data)
				return retval;
			freq_data = mpxplay_dtvdrv_database_frequency_search(dtv_device->protocol_number_id, dtv_device->frequency); // FIXME: duplicated search
			if(!freq_data)
				return retval;
			funcbit_enable(freq_data->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_SDT);
			freq_data->freq_signal_quality = dtv_device->signal_quality;
			if(dtvfi->psifn_callback)
			{
				mpxplay_dtvdrv_assemble_filename(&strtmp[0], sizeof(strtmp), dtv_device->protocol_number_id, dtv_device->frequency, prog_data->program_id);
				//dtvfi->psifn_callback(dtvfi->psifn_cb_data, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_ADDDIRENTRY, (void *)&strtmp[0], NULL); // not available from here
				retval = dtvfi->psifn_callback(dtvfi->psifn_cb_data, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_I3I_ARTIST, (void *)&strtmp[0], (char *)argp2);
			}
			else
			{
				retval = MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
			}
			break;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_SET_PROGRAMCHANID:
			if(!argp2)
				return retval;
			prog_data = mpxplay_dtvdrv_database_program_search(dtv_device->protocol_number_id, dtv_device->frequency, program_id);
			if(!prog_data)
				return retval;
			freq_data = mpxplay_dtvdrv_database_frequency_search(dtv_device->protocol_number_id, dtv_device->frequency); // FIXME: duplicated search
			if(!freq_data)
				return retval;
			funcbit_enable(freq_data->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_NIT);
			prog_data->channel_id = *((mpxp_uint32_t *)argp2);
			retval = MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
			break;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_SET_PROGRAMCTRLFLGS:
			if(!argp2)
				return retval;
			prog_data = mpxplay_dtvdrv_database_program_search(dtv_device->protocol_number_id, dtv_device->frequency, program_id);
			if(!prog_data)
				return retval;
			prog_data->ctrl_flags = *((mpxp_uint32_t *)argp2);
			retval = MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
			break;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_SET_PROGEVENTCHANGE:
			if(!dtvfi->psifn_callback)
				break;
			prog_data = mpxplay_dtvdrv_database_program_search(dtv_device->protocol_number_id, dtv_device->frequency, program_id);
			if(prog_data)
			{
				struct mpxplay_dvb_epg_event_data_s *curr_event = mpxplay_dtvdrv_database_get_current_event(prog_data);
				mpxp_ptrsize_t duration_ms = 1000;
				mpxplay_dtvdrv_assemble_filename(&strtmp[0], sizeof(strtmp), dtv_device->protocol_number_id, dtv_device->frequency, prog_data->program_id);
				if(curr_event)
				{
					duration_ms *= pds_timeval_to_seconds(curr_event->event_duration_time);
					dtvfi->psifn_callback(dtvfi->psifn_cb_data, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_I3I_TITLE, (void *)&strtmp[0], (void *)curr_event->event_shortdesc_name);
				}
				dtvfi->psifn_callback(dtvfi->psifn_cb_data, MPXPLAY_DISKDRIVE_CFGFUNCNUM_CBEVENT_MODDIRENTRY_TIMELENMS, (void *)&strtmp[0], (void *)duration_ms);
				retval = MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
			}
			break;
		case MPXPLAY_DISKFILE_CFGFUNCNUM_ADD_PROGEVENTENTRY:
			if(!argp2)
				return retval;
			event_data_in = (struct mpxplay_dvb_epg_event_data_s *)argp2;
			if(!event_data_in->event_id && (!event_data_in->event_begin_date_time || !event_data_in->event_duration_time))
				return retval;
			prog_data = mpxplay_dtvdrv_database_program_new(dtv_device->protocol_number_id, dtv_device->frequency, program_id, NULL);
			if(!prog_data)
				return retval;
			event_data_new = mpxplay_dtvdrv_database_event_add(prog_data, event_data_in->event_id, event_data_in->event_begin_date_time, event_data_in->event_duration_time, TRUE);
			if(event_data_new)
			{
				mpxplay_dtvdrv_database_event_add_short_description(event_data_new, event_data_in->event_shortdesc_name, event_data_in->event_shortdesc_details);
				if(event_data_new->extdesc_counter < event_data_in->extdesc_counter)
				{   // extended event descriptor is always updated once in a program session (extdesc_counter is not saved to XML)
					mpxplay_dtvdrv_database_event_add_extended_description(event_data_new, event_data_in->event_extended_desc, FALSE);
					event_data_new->extdesc_counter = event_data_in->extdesc_counter;
					//mpxplay_debugf(MPXPLAY_DEBUGOUT_PROGEV, "%3d %5d %.200s", prog_data->program_id, event_data_in->event_id, event_data_new->event_extended_desc);
				}
				event_data_new->parental_rating = event_data_in->parental_rating;
				event_data_new->event_flags = event_data_in->event_flags;
			}
			retval = MPXPLAY_DISKDRIV_CFGERROR_SET_OK;
			break;
	}

	return retval;
}

static int dtvdrv_file_bda_wait_for_signal(dvb_device_t *d, unsigned int timeout_ms)
{
    int retry = timeout_ms / 50 + 1;
    int retval = MPXPLAY_ERROR_DISKDRIV_ERROR;
    char sout[128];

    do
    {
    	//mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "dtvdrv_file_bda_wait_for_signal r:%d", retry);

    	if(dtv_bda_device_is_signal(d))
    	{
    		retval = MPXPLAY_ERROR_DISKDRIV_OK;
    		break;
    	}

        if(funcbit_test(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_DISKDRIVTERM) || (!mpxplay_control_keyboard_get_topfunc() && pds_wipeout_by_extkey(KEY_ESC)))
        {
        	funcbit_enable(d->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE);
        	break;
        }

        snprintf(sout, sizeof(sout), "DTV: opening %d Mhz, signal: %d%% (retry: %2d)", d->frequency / 1000000, 0, retry / 20);
        display_timed_message(sout);

        pds_threads_sleep(50);
    }
    while((--retry) && !funcbit_test(d->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE));

    return retval;
}

static unsigned int dtvdrive_file_check(void *drivehand_data,char *filename)
{
	return (mpxplay_drvdtv_get_protocol_id(filename) >= 0)? 1 : 0;
}

static void *dtvdrive_file_open(void *drivehand_data,char *filename,unsigned long openmode,int *err_code)
{
	struct dtvfile_info_s *dtvfi = NULL;
	int prot_number, lock_result = MPXPLAY_ERROR_MUTEX_ARGS, retVal = MPXPLAY_ERROR_DISKDRIV_ERROR;
	char sout[256];

	if(openmode&(O_RDWR|O_WRONLY|O_CREAT))
		goto err_out_openr;

	if(!drvdtv_mutexhnd_open)
		pds_threads_mutex_new(&drvdtv_mutexhnd_open);
	lock_result = PDS_THREADS_MUTEX_LOCK(&drvdtv_mutexhnd_open, MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_MUTEX);

	prot_number = mpxplay_drvdtv_get_protocol_id(filename);
	if(prot_number < 0)
	{
		retVal = MPXPLAY_ERROR_DISKDRIV_PROTOCOL;
		goto err_out_openr;
	}

	dtvfi = pds_calloc(1,sizeof(*dtvfi));
	if(!dtvfi)
		goto err_out_openr;

	dtvfi->filesize = MPXPLAY_DISKDRIV_FILESIZE_UNKNOWN;
	dtvfi->openmode = openmode;
	dtvfi->dtvdi = (dtvdrive_info_s *)drivehand_data;

	dtvfi->dtv_device = dtv_bda_device_alloc();
	if(!dtvfi->dtv_device)
		goto err_out_openr;

	mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "dtvdrive_file_open fh:%8.8X om:%8.8X %s", (mpxp_ptrsize_t)dtvfi, openmode, filename);

	dtvfi->dtv_device->protocol_number_id = prot_number;
	dtvfi->dtv_device->epg_infos.control_cb = dtvdrv_file_epg_callback_func;
	dtvfi->dtv_device->epg_infos.ccb_data = (void *)dtvfi;
	in_ffmp_epg_mpegts_clearbuf_epginfo(&dtvfi->dtv_device->epg_infos);

	if(dtvdrive_parse_options(filename, &dtvfi->filename_store, dtvfi->dtv_device) == FALSE) // new error code for this?
	{
		retVal = MPXPLAY_ERROR_DISKDRIV_PROTOCOL;
		goto err_out_openr;
	}

	if(!funcbit_test(openmode, MPXPLAY_INFILE_OPENMODE_LOAD_PLAY) && funcbit_test(openmode, MPXPLAY_INFILE_OPENMODE_CHECK))
	{
		retVal = MPXPLAY_ERROR_DISKDRIV_OPENMODE;
		goto err_out_openr;
	} // else open for play, or simple copy, or scan

	if(dtvfi->dtv_device->scan_freq_begin || dtvfi->dtv_device->scan_freq_end || dtvfi->dtv_device->scan_do_epgload)
	{
		if(mpxplay_drvdtv_scan_start(dtvfi->dtv_device) != MPXPLAY_ERROR_DISKDRIV_OK)
			goto err_out_openr;
	}
	else
	{
		mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "----------------- LOAD_PLAY fh:%8.8X dv:%8.8X %s", (mpxp_ptrsize_t)dtvfi, (mpxp_ptrsize_t)dtvfi->dtv_device, filename);
		if(funcbit_test(openmode, MPXPLAY_INFILE_OPENMODE_LOAD_PLAY))
		{
			funcbit_enable(dtvfi->dtv_device->flags, MPXPLAY_DRVDTV_FLAG_OPENMODE_PLAY);
			snprintf(sout, sizeof(sout), "DTV: opening %d Mhz", dtvfi->dtv_device->frequency / 1000000);
			display_timed_message(sout);
			dtvfi->dtv_device->timeout_open_showsignalstrength = pds_gettimem();
			mpxplay_drvdtv_database_load_from_file();
		}

		if(dtv_bda_resource_assign(dtvfi->dtv_device) != MPXPLAY_ERROR_DISKDRIV_OK)
			goto err_out_openr;
		if(dtvdrv_file_bda_wait_for_signal(dtvfi->dtv_device, MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_SIGNAL) != MPXPLAY_ERROR_DISKDRIV_OK)
			goto err_out_openr;

		if(!dtvfi->dtv_device->program_number_id)
			snprintf(dtvfi->content_artist, sizeof(dtvfi->content_artist), "DTV-%dMhz-%6.6d", dtvfi->dtv_device->frequency / 1000000, pds_gettime());

		mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "----------------- LOAD_PLAY END fh:%8.8X dv:%8.8X %s", (mpxp_ptrsize_t)dtvfi, (mpxp_ptrsize_t)dtvfi->dtv_device, filename);

		if(err_code)
			*err_code = MPXPLAY_ERROR_DISKDRIV_OK;
	}

	if(lock_result == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&drvdtv_mutexhnd_open);

	return dtvfi;

err_out_openr:
	mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "ERR_out_openr fh:%8.8X di:%8.8X %s", (mpxp_ptrsize_t)dtvfi, (mpxp_ptrsize_t)dtvfi->dtv_device, filename);
	if(dtvfi && dtvfi->dtv_device && (retVal != MPXPLAY_ERROR_DISKDRIV_OPENMODE))
	{
		snprintf(sout, sizeof(sout), "DTV: init %d Mhz failed!", dtvfi->dtv_device->frequency / 1000000);
		display_timed_message(sout);
	}
	if(lock_result == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&drvdtv_mutexhnd_open);
	dtvdrive_file_close(dtvfi);
	if(err_code)
		*err_code = retVal;
	return NULL;
}

static void dtvdrive_file_close(void *filehand_data)
{
	const int lock_result = PDS_THREADS_MUTEX_LOCK(&drvdtv_mutexhnd_open, MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_MUTEX);
	struct dtvfile_info_s *dtvfi = (struct dtvfile_info_s *)filehand_data;
	mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "dtvdrive_file_close fh:%8.8X di:%8.8X bda:%8.8X",
			filehand_data, ((dtvfi)? (mpxp_ptrsize_t)dtvfi->dtv_device : 0),
			((dtvfi && dtvfi->dtv_device)? (mpxp_ptrsize_t)dtvfi->dtv_device->bdagraph_module : 0));
	if(dtvfi)
	{
		dvb_device_t *d = dtvfi->dtv_device;
		if(d)
		{
			funcbit_enable(d->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE);
			mpxplay_drvdtv_scan_stop(d);
			dtv_bda_resource_remove(d);
			dtv_bda_device_free(d);
		}
		if(dtvfi->filename_store)
			pds_free(dtvfi->filename_store);
		pds_free(dtvfi);
	}
	if(lock_result == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&drvdtv_mutexhnd_open);
}

static long dtvdrive_file_read(void *filehand_data,char *ptr,unsigned int num)
{
	struct dtvfile_info_s *dtvfi = (struct dtvfile_info_s *)filehand_data;
	long bytes = -1, read_error_count = 0;
	dvb_device_t *d;
	char retry_string[64], sout[256];

	if(!dtvfi)
		return MPXPLAY_ERROR_DISKDRIV_ERROR;
	if(funcbit_test(dtvfi->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE))
		return MPXPLAY_ERROR_DISKDRIV_CONNECT_TERMINATED;
	if(!dtvfi->dtv_device || !ptr || !num)
		return bytes;
	if(funcbit_test(dtvfi->dtv_device->flags, (MPXPLAY_DRVDTV_FLAG_OPENMODE_SCAN | MPXPLAY_DRVDTV_FLAG_TERMINATE)))
	{
		funcbit_enable(dtvfi->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE); // FIXME: ???
		return MPXPLAY_ERROR_DISKDRIV_CONNECT_TERMINATED;
	}
	if(PDS_THREADS_MUTEX_LOCK(&drvdtv_mutexhnd_open, 0) != MPXPLAY_ERROR_OK)
		return bytes;

//	mpxplay_debugf(MPXPLAY_DEBUGOUT_READ, "dtvdrive_file_read fh:%8.8X n:%d", (mpxp_uint32_t)dtvfi, num);
	d = dtvfi->dtv_device;
	do{
		bytes = dtv_bda_device_read(d, ptr, num);
		if(bytes > 0)
		{
			dtvfi->filepos += bytes;
			read_error_count = 0;
		}
		else if(bytes == MPXPLAY_ERROR_DISKDRIV_ERROR)
		{
			bytes = MPXPLAY_ERROR_DISKDRIV_CONNECT_TERMINATED;
			break;
		}
		else if((bytes == MPXPLAY_ERROR_DISKDRIV_TIMEOUT_READ) && (dtvfi->filepos < MPXPLAY_DRVDTV_INITIAL_READ_DATASIZE))
		{
			mpxplay_debugf(MPXPLAY_DEBUGOUT_ERROR, "dtvdrive_file_read dtv_bda_device_read fh:%8.8X b:%d c:%d", (mpxp_ptrsize_t)dtvfi, bytes, read_error_count);
			if(read_error_count >= MPXPLAY_DRVDTV_INITIAL_READERROR_LIMIT)
			{
				funcbit_enable(dtvfi->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE);
				bytes = MPXPLAY_ERROR_DISKDRIV_CONNECT_TERMINATED;
				snprintf(sout, sizeof(sout), "DTV: opening %d Mhz failed (read timeout)!", d->frequency / 1000000);
				display_timed_message(sout);
				break;
			}
		}

		if(d->timeout_open_showsignalstrength)
		{
			mpxp_uint64_t currtime_ms = pds_gettimem();
			if((currtime_ms >= (d->timeout_open_showsignalstrength + MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_OPENSHOWSIGSTR_60)) && (bytes > 0))
			{
				d->timeout_open_showsignalstrength = 0;
			}
			else
			{
				int signal_strength = dtv_bda_device_get_signal_strength(d);
				if((signal_strength <= 60) || (currtime_ms < (d->timeout_open_showsignalstrength + MPXPLAY_DRVDTV_DEFAULT_TIMEOUTMS_OPENSHOWSIGSTR_100)))
				{
					if(bytes == MPXPLAY_ERROR_DISKDRIV_TIMEOUT_READ)
					{
						snprintf(retry_string, sizeof(retry_string), ", read timeout, retry %d/%d ", (read_error_count + 1), MPXPLAY_DRVDTV_INITIAL_READERROR_LIMIT);
					}
					else
					{
						retry_string[0] = 0;
					}
					if(signal_strength < 0)
						signal_strength = 0;
					snprintf(sout, sizeof(sout), "DTV: opening %d Mhz, signal: %d%%%s", d->frequency / 1000000, signal_strength, retry_string);
					display_timed_message(sout);
				}
				d->signal_quality = signal_strength;
			}
		}
	}while((bytes <= 0) && (++read_error_count <= MPXPLAY_DRVDTV_INITIAL_READERROR_LIMIT) && !funcbit_test(dtvfi->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE));

	PDS_THREADS_MUTEX_UNLOCK(&drvdtv_mutexhnd_open);

	return bytes;
}

static mpxp_filesize_t dtvdrive_file_seek(void *filehand_data,mpxp_filesize_t pos,int fromwhere)
{
	struct dtvfile_info_s *dtvfi = (struct dtvfile_info_s *)filehand_data;
	return (funcbit_test(dtvfi->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE))? (mpxp_filesize_t)-1 : dtvfi->filepos;
}

static mpxp_filesize_t dtvdrive_file_tell(void *filehand_data)
{
	struct dtvfile_info_s *dtvfi = (struct dtvfile_info_s *)filehand_data;
	return dtvfi->filepos;
}

static mpxp_filesize_t dtvdrive_file_length(void *filehand_data)
{
	struct dtvfile_info_s *dtvfi = (struct dtvfile_info_s *)filehand_data;
	return dtvfi->filesize;
}

static int dtvdrive_file_eof(void *filehand_data)
{
	struct dtvfile_info_s *dtvfi = (struct dtvfile_info_s *)filehand_data;
	return (funcbit_test(dtvfi->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE))? 1 : 0;
}

//---------------------------------------------------------------------------------------------------------------------------------

struct mpxplay_drivehand_func_s DTVDRIVE_drivehand_funcs={
 "DTVDRIVE",
 MPXPLAY_DRIVEHANDFUNC_INFOBIT_DISCONNECT | MPXPLAY_DRIVEHANDFUNC_INFOBIT_READONLY | MPXPLAY_DRIVEHANDFUNC_INFOBIT_LISTTYPEDIR | MPXPLAY_DRIVEHANDFUNC_INFOBIT_NOAUTOSKIP,
 &dtvdrive_drive_config,
 &dtvdrive_drive_check,
 &dtvdrive_drive_mount,
 &dtvdrive_drive_unmount,
 &dtvdrive_findfirst,
 &dtvdrive_findnext,
 &dtvdrive_findclose,
 &dtvdrive_getcwd,
 &dtvdrive_chdir,
 &dtvdrive_mkdir,
 &dtvdrive_rmdir,
 &dtvdrive_rename,
 &dtvdrive_unlink,
 NULL, // r15
 NULL, // r16
 NULL, // r17
 NULL, // r18
 NULL, // r19
 NULL, // r20

 &dtvdrive_file_check,
 &dtvdrive_file_open,
 &dtvdrive_file_close,
 &dtvdrive_file_read,
 NULL, // file_write
 &dtvdrive_file_seek,
 &dtvdrive_file_tell,
 &dtvdrive_file_length,
 &dtvdrive_file_eof,
 NULL, // file_chsize
 NULL  // r31
};

#endif // defined(MPXPLAY_LINK_INFILE_FF_MPEG) && defined(MPXPLAY_WIN32)
