//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
////function: DVB scan and Electronic Program Guide dialog window

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUGOUT_EPG stdout
#define DISPQT_DEBUGOUT_SCAN stdout

#include <QtGui>
#include <QtWidgets>
#include <QtAlgorithms>
#include "disp_qt.h"
#include "moc_dvb_epg.h"
#include "moc_mainwindow.h"

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

extern "C" {
 extern unsigned long mpxplay_config_dvbepg_control_flags, mpxplay_config_dvbepg_pastevents_hours;
}

DispQtEPGDialog::DispQtEPGDialog(MainWindow *mainwindow, QWidget *parent, bool open_from_contextmenu) : DispQtDialogElemWindow(mainwindow, parent, (Qt::Dialog | Qt::WindowMaximizeButtonHint | Qt::WindowCloseButtonHint), DISPQT_DIALOG_WINDOWTYPE_DVBEPG)
{
	const int screen_number = (open_from_contextmenu && (QApplication::desktop()->screenCount() > 1))? mainwindow->video_get_fullscreen_display_number() : mainwindow->mainwin_get_display_number();
	const QRect screenGeometry = QApplication::desktop()->screenGeometry(screen_number);
	const int screen_pos_x0 = screenGeometry.x(), screen_pos_y0 = screenGeometry.y();
	const int screen_max_x = screenGeometry.width(), screen_max_y = screenGeometry.height();
	struct mmc_dispqt_config_s *gcfg = mainwindow->gui_config;

	this->main_window = mainwindow;
	this->setContentsMargins(0, 0, 0, 0);

	if(mainwindow->qt_video_player)
		mainwindow->qt_video_player->video_mouse_cursor_forbid_hide(true); // FIXME

	if( (gcfg->epgdialogwin_pos_x >= 0) && (gcfg->epgdialogwin_pos_x < (screen_max_x - 20)) && (gcfg->epgdialogwin_pos_y >= 0) && (gcfg->epgdialogwin_pos_y < (screen_max_y - 20))
	 && (gcfg->epgdialogwin_size_x > 0) && (gcfg->epgdialogwin_size_x <= screen_max_x) && (gcfg->epgdialogwin_size_y > 0) && (gcfg->epgdialogwin_size_y <= screen_max_y)
	){
		this->move(screen_pos_x0 + gcfg->epgdialogwin_pos_x, screen_pos_y0 + gcfg->epgdialogwin_pos_y);
		this->resize(gcfg->epgdialogwin_size_x, gcfg->epgdialogwin_size_y);
	}
	else
	{
		this->resize(520, 480);
	}
	this->epgtab_widget = new DispQtEPGTabWidget(mainwindow, this);
	this->setWindowTitle("DVB/EPG control");
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->setMargin(0);
	mainLayout->addWidget(this->epgtab_widget, 1);
	setLayout(mainLayout);
	this->timer_epgdialog_geometryupdate.setInterval(DISPQT_WINDOW_ACTIVATE_GEOMETRYUPDATE_DELAY);
	this->timer_epgdialog_geometryupdate.setSingleShot(true);
	connect(this, SIGNAL(signal_epgdialog_geometry_update()), this, SLOT(epgdialog_geometry_start_update()));
	connect(this, SIGNAL(signal_epgdialog_config_style_apply(bool)), this, SLOT(epgdialog_config_style_apply(bool)));
	connect(this, SIGNAL(signal_epgdialog_send_dtvdbase_epginfos(void *, char *)), this, SLOT(epgdialog_send_dtvdbase_epginfos(void *, char *)));
	connect(this, SIGNAL(signal_epgdialog_send_scaninfos(void *)), this, SLOT(epgdialog_send_scaninfos(void *)));
	connect(this, SIGNAL(signal_epgdialog_send_localfile_epginfos(void *, char *)), this, SLOT(epgdialog_send_localfile_epginfos(void *, char *)));
	connect(this->epgtab_widget->dvbepg_widget->button_close, SIGNAL(pressed()), this, SLOT(close()));
	connect(this->epgtab_widget->epglocalfile_widget->button_close, SIGNAL(pressed()), this, SLOT(close()));
	connect(&this->timer_epgdialog_geometryupdate, SIGNAL(timeout()), this, SLOT(epgdialog_geometry_update()));
	this->show();
	this->epgdialog_open();
}

DispQtEPGDialog::~DispQtEPGDialog()
{
	this->epgdialog_close();
}

void DispQtEPGDialog::closeEvent(QCloseEvent *e)
{
	this->epgdialog_close();
	QDialog::closeEvent(e);
}

bool DispQtEPGDialog::event(QEvent *event)
{
	switch(event->type())
	{
		case QEvent::Enter:
			if(this->main_window->qt_video_player)
				this->main_window->qt_video_player->video_mouse_cursor_forbid_hide(true); // FIXME: ???
			break;
		case QEvent::WindowActivate:
			if( (this->main_window->ms_windows_version >= MPXPLAY_MSWIN_VERSIONID_WIN80)
			 && (this->main_window->ms_windows_version < MPXPLAY_MSWIN_VERSIONID_WIN11)
			 && !this->main_window->mainwin_is_on_video() && this->main_window->mainwin_guilayout_is_translucent_enabled()
			){
				this->epgdialog_geometry_start_update();
			}
			break;
		default:
			break;
	}
	return QDialog::event(event);
}

// prepare a move event (to update Win10 transparent background of mainwindow)
void DispQtEPGDialog::epgdialog_geometry_update()
{
	if(this->windowState() & (Qt::WindowMaximized | Qt::WindowMinimized))
		return;
	int x = this->x(), y = this->y();
	this->move(x + 1, y);
	this->move(x, y);
}

void DispQtEPGDialog::epgdialog_geometry_start_update()
{
	this->timer_epgdialog_geometryupdate.start();
}

static void dvbepg_dialog_send_full_epg_data(unsigned int prot_flags)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_data;

	mpxplay_drvdtv_database_load_from_file();

	prot_data = mpxplay_dtvdrv_database_protocol_all_info(mpxplay_config_dvbepg_control_flags);
	if(prot_data)
	{
		funcbit_enable(prot_data->prot_flags, prot_flags);
		if(!mpxplay_dispqt_epgdialog_send_dtvdabse_epginfos((void *)prot_data))
			mpxplay_dtvdrv_database_clear_protocols(prot_data); // dialog is closed (should not happen here)
	}
	mpxplay_debugf(DISPQT_DEBUGOUT_SCAN, "dvbepg_dialog_send_full_epg_data");
}

static void dvbepg_dialog_send_full_epg_timer_data(void)
{
	dvbepg_dialog_send_full_epg_data(MPXPLAY_DVBEPG_PROTOCOL_FLAG_FULLDATA);
}

static void dvbepg_dialog_send_full_epg_newfile_data(void)
{
	dvbepg_dialog_send_full_epg_data(MPXPLAY_DVBEPG_PROTOCOL_FLAG_FULLDATA | MPXPLAY_DVBEPG_PROTOCOL_FLAG_NEWFILE);
}

void DispQtEPGDialog::epgdialog_open(void)
{
	mpxplay_timer_addfunc((void *)&dvbepg_dialog_send_full_epg_timer_data,   NULL, MPXPLAY_TIMERTYPE_REPEAT, mpxplay_timer_secs_to_counternum(DISPQT_EPG_REFRESH_INTERVAL_SEC));
	mpxplay_timer_addfunc((void *)&dvbepg_dialog_send_full_epg_newfile_data, NULL, MPXPLAY_TIMERTYPE_SIGNAL | MPXPLAY_TIMERTYPE_REPEAT | MPXPLAY_TIMERFLAG_MULTIPLY, MPXPLAY_SIGNALTYPE_NEWFILE);
	mpxplay_timer_addfunc((void *)&dvbepg_dialog_send_full_epg_timer_data,   NULL, MPXPLAY_TIMERTYPE_WAKEUP | MPXPLAY_TIMERFLAG_MULTIPLY, 0); // FIXME: first call of timed functions
	this->epgtab_widget->dvbscan_widget->dvb_loadchannels_button_handler();
}

void DispQtEPGDialog::epgdialog_close(void)
{
	this->main_window->dvbepg_dialog = NULL;
	mpxplay_timer_deletefunc((void *)&dvbepg_dialog_send_full_epg_timer_data, NULL);
	mpxplay_timer_deletefunc((void *)&dvbepg_dialog_send_full_epg_newfile_data, NULL);
	mpxplay_timer_deletefunc((void *)&dvbepg_dialog_send_full_epg_timer_data, NULL); // TODO: deleting of all instances of a function from the timer
	mpxplay_timer_deletefunc((void *)&dvbepg_dialog_send_full_epg_timer_data, NULL);
	if(!(this->windowState() & ((unsigned int)Qt::WindowMaximized)))
	{
		const int screen_number = QApplication::desktop()->screenNumber(this->epgtab_widget);
		const QRect screenGeometry = QApplication::desktop()->screenGeometry(screen_number);
		const int screen_pos_x0 = screenGeometry.x(), screen_pos_y0 = screenGeometry.y();
		struct mmc_dispqt_config_s *gcfg = this->main_window->gui_config;
		gcfg->epgdialogwin_pos_x = this->x() - screen_pos_x0;
		gcfg->epgdialogwin_pos_y = this->y() - screen_pos_y0;
		gcfg->epgdialogwin_size_x = this->width();
		gcfg->epgdialogwin_size_y = this->height();
	}
	this->epgtab_widget->dvbscan_widget->dvb_stopscan_init();
	if(this->main_window->qt_video_player)
		this->main_window->qt_video_player->video_mouse_cursor_forbid_hide(false);
	QApplication::setOverrideCursor(Qt::ArrowCursor);
}

void DispQtEPGDialog::epgdialog_config_style_apply(bool initial)
{
	this->dialogwin_config_style_apply(initial);
	this->epgtab_widget->epgtabwidget_config_style_apply(initial);
}

// callback for dtvdbase.c
void DispQtEPGDialog::epgdialog_send_dtvdbase_epginfos(void *protocol_data, char *currfilename)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = (struct mpxplay_dvb_epg_protocol_data_s *)protocol_data;
	if(this->mutex_epg_update.tryLock())
	{
		if(this->epgtab_widget->dvbepg_widget->dvbepgwidget_send_epginfos(prot_data, currfilename)) // if a DVB program has started manually under the scan process
			emit this->epgtab_widget->dvbscan_widget->dvb_stopscan_button_handler();                // then close the scanning
		this->mutex_epg_update.unlock();
	}
	mpxplay_dtvdrv_database_clear_protocols(prot_data);
}

// callback for dtvscan.c
void DispQtEPGDialog::epgdialog_send_scaninfos(void *protocol_data)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = (struct mpxplay_dvb_epg_protocol_data_s *)protocol_data;
	mpxplay_debugf(DISPQT_DEBUGOUT_SCAN, "GOT SCAN INFOS");
	if(this->mutex_scan_update.tryLock())
	{
		if(funcbit_test(prot_data->prot_flags, MPXPLAY_DVBEPG_PROTOCOL_FLAG_SCANFINISH))
			this->epgtab_widget->dvbepg_widget->dvbepg_reload_epginfos_full();
		this->epgtab_widget->dvbscan_widget->dvbscanwidget_send_epginfos(prot_data);
		this->mutex_scan_update.unlock();
	}
	mpxplay_dtvdrv_database_clear_protocols(prot_data);
}

// callback for ffmpepg.c
void DispQtEPGDialog::epgdialog_send_localfile_epginfos(void *protocol_data, char *currfilename)
{
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = (struct mpxplay_dvb_epg_protocol_data_s *)protocol_data;
	if(this->mutex_epg_localfile_update.tryLock())
	{
		this->epgtab_widget->epglocalfile_widget->dvbepgwidget_send_epginfos(prot_data, currfilename);
		this->mutex_epg_localfile_update.unlock();
	}
	mpxplay_dtvdrv_database_clear_protocols(prot_data);
}

//-------------------------------------------------------------------------------------------

DispQtEPGTabWidget::DispQtEPGTabWidget(MainWindow *mainwindow, QWidget *parent) : DispQtDialogElemTabWidget(mainwindow, parent)
{
	this->main_window = mainwindow;
	this->setContentsMargins(0, 0, 0, 0);
	this->setMinimumSize(210, 160);
	this->setFocusPolicy(Qt::NoFocus);
	this->dvbepg_widget = new DispQtDVBEPGMainWidget(main_window, this, false);
	this->dvbscan_widget = new DispQtDVBScanGWidget(main_window, this);
	this->epglocalfile_widget = new DispQtEPGLocalFileWidget(main_window, this);
	int index = this->addTab(this->dvbepg_widget, "      DVB EPG      ");
	this->setTabToolTip(index, "Display EPG data of all DVB devices");
	this->addTab(this->dvbscan_widget, "      DVB Scan     ");
	index = this->addTab(this->epglocalfile_widget, "      File EPG      ");
	this->setTabToolTip(index, "Display EPG data of a local MPEG-TS file");
//	mpxplay_debugf(DISPQT_DEBUGOUT_EPG, "w:%d h:%d", this->width(), this->height());
}

void DispQtEPGTabWidget::epgtabwidget_config_style_apply(bool initial)
{
	this->tabwidget_config_style_apply(initial);
	this->dvbepg_widget->dvbepg_config_style_apply(initial);
	this->dvbscan_widget->dvbscan_config_style_apply(initial);
	this->epglocalfile_widget->dvbepg_config_style_apply(initial);
}

//=========================================================================================================================

DispQtDVBEPGMainWidget::DispQtDVBEPGMainWidget(MainWindow *mainwindow, DispQtEPGTabWidget *parent, bool local_file_epg_handling) : QWidget(parent)
{
	this->main_window = mainwindow;
	this->parent_tabwidget = parent;
	this->is_local_file_epg_handler = local_file_epg_handling;
	this->setContentsMargins(0, 0, 0, 0);

	this->dvbepg_stacked_widget = new QStackedWidget(this);
	this->dvbepg_eventsview_widget = new DispQtDVBEPGEventsViewWidget(mainwindow, this, local_file_epg_handling);
	this->dvbepg_ruler_widget = new DispQtDVBEPGRulerWidget(this->dvbepg_eventsview_widget, this);
	this->dvbepg_programs_widget = new DispQtDVBEPGProgramsWidget(this->dvbepg_eventsview_widget, this);

	this->button_refresh = new DispQtDialogElemPushButton(mainwindow, this, tr("Refresh"));
	this->button_refresh->setFocusPolicy(Qt::NoFocus);
	this->button_refresh->setToolTip("Events are automatically updated in every 10 secs.\n(but without clearing the wrong/changed events)\nThis button completely reloads all events.");
	this->button_close = new DispQtDialogElemPushButton(mainwindow, this, tr("Close"));
	this->button_close->setFocusPolicy(Qt::NoFocus);

	this->dvbepgwidget_set_zoom(1);

	QLabel *noepgLabel = new QLabel( tr("EPG data is not available!"), this);
	noepgLabel->setAlignment(Qt::AlignCenter);
	this->dvbepg_stacked_widget->insertWidget(DISPQT_DVBEPG_NOEPG_WIDGET, noepgLabel);

	QWidget *buttonsWidget = new QWidget(this);
	QVBoxLayout *buttonsLayout = new QVBoxLayout(buttonsWidget);
	buttonsLayout->addWidget(this->button_refresh);
	buttonsLayout->addWidget(this->button_close);

	QWidget *containerWidget = new QWidget(this);
	QGridLayout* containerLayout = new QGridLayout(containerWidget);
	containerLayout->setMargin(0);
	containerLayout->setSpacing(0);

	containerLayout->addWidget(this->dvbepg_ruler_widget, 0, 1);
	containerLayout->addWidget(this->dvbepg_programs_widget, 1, 0);
	containerLayout->addWidget(this->dvbepg_eventsview_widget, 1, 1);
	containerLayout->addWidget(buttonsWidget, 2, 0);
	containerLayout->setRowStretch(1, 1);

	QWidget *descriptionWidget = new QWidget(this);
	QVBoxLayout *descriptionLayout = new QVBoxLayout(descriptionWidget);
	descriptionLayout->setContentsMargins(0, 0, 0, 5);
	descriptionLayout->setSpacing(0);
	QLabel *desc_head = new QLabel(tr("Description"), this);
	descriptionLayout->addWidget(desc_head);
	this->dvbepg_description = new DispQtDialogElemTextEdit(mainwindow, this);
	this->dvbepg_description->setReadOnly(true);
	this->dvbepg_description->setFrameStyle(QFrame::Sunken | QFrame::StyledPanel);
	this->dvbepg_description->setAlignment(Qt::AlignLeft | Qt::AlignTop);
	this->dvbepg_description->setFixedHeight(DISPQT_EPG_DESCRIPTION_HEIGHT);
	this->dvbepg_description->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff); // because we use hover event to show description of a program item
	descriptionLayout->addWidget(this->dvbepg_description);
	descriptionLayout->addStretch(0);
	containerLayout->addWidget(descriptionWidget, 2, 1);
	containerLayout->setRowStretch(2, 0);

	this->dvbepg_stacked_widget->insertWidget(DISPQT_DVBEPG_EPGVIEW_WIDGET, containerWidget);

	this->dvbepg_stacked_widget->setCurrentIndex(DISPQT_DVBEPG_NOEPG_WIDGET);
	QGridLayout *mainEpgLayout = new QGridLayout(this);
	mainEpgLayout->addWidget(this->dvbepg_stacked_widget);
	setLayout(mainEpgLayout);

	connect(this->dvbepg_eventsview_widget, SIGNAL(signal_eventsviewwidget_program_added(const DispQtDVBEPGProgram *)), this->dvbepg_programs_widget, SLOT(programswidget_add_program(const DispQtDVBEPGProgram *)));
	connect(this->dvbepg_eventsview_widget, SIGNAL(signal_eventsviewwidget_time_start_changed(const QDateTime &)), this->dvbepg_ruler_widget, SLOT(rulerwidget_set_starttime(const QDateTime &)));
	connect(this->dvbepg_eventsview_widget, SIGNAL(signal_eventsviewwidget_item_focused(DispQtDVBEPGItem*)), this, SLOT(dvbepgwidget_display_event_description(DispQtDVBEPGItem*)));
	connect(this->dvbepg_eventsview_widget->horizontalScrollBar(), SIGNAL(valueChanged(int)), this->dvbepg_ruler_widget, SLOT(rulerwidget_set_offset(int)));
	connect(this->dvbepg_eventsview_widget->verticalScrollBar(), SIGNAL(valueChanged(int)),this->dvbepg_programs_widget, SLOT(programswidget_set_offset(int)));
	connect(this->button_refresh, SIGNAL(pressed()), this, SLOT(dvbepg_reload_epginfos_full()));
}

void DispQtDVBEPGMainWidget::dvbepgwidget_display_event_description(DispQtDVBEPGItem *epgItem)
{
	if(!epgItem)
	{
		this->dvbepg_description->clear();
		return;
	}

	QString ev_start = (epgItem->get_event_start().daysTo(QDateTime::currentDateTime()) == 0)? epgItem->get_event_start().time().toString("hh:mm") : epgItem->get_event_start().toString(Qt::SystemLocaleLongDate);
	QString ev_end = epgItem->get_event_start().addSecs(epgItem->get_event_duration()).time().toString("hh:mm");

	this->dvbepg_description->setText( QString("%1 - %2 : %3%4 %5%6")
		.arg(ev_start)
		.arg(ev_end)
		.arg(epgItem->get_event_name())
		.arg(epgItem->get_event_parental_rating()? tr(" (%1+ rated)").arg(epgItem->get_event_parental_rating()) : QString())
		.arg(epgItem->get_event_description())
		.arg(epgItem->get_event_extdescription())
	);
}

void DispQtDVBEPGMainWidget::keyPressEvent(QKeyEvent *event)
{
	if(event->key() != Qt::Key_Escape)
		this->main_window->handle_keypressevent(event);
	else
		QWidget::keyPressEvent(event);
}

void DispQtDVBEPGMainWidget::dvbepg_config_style_apply(bool initial)
{
	if(!initial)
	{
		this->dvbepgwidget_clear();
		this->dvbepg_description->textedit_set_style(initial);
		this->dvbepg_ruler_widget->rulerwidget_config_style_apply(initial);
		this->dvbepg_programs_widget->programswidget_config_style_apply(initial);
		this->dvbepg_eventsview_widget->eventsviewwidget_config_style_apply(initial);
		this->button_refresh->pushbutton_set_style(initial);
		this->button_close->pushbutton_set_style(initial);
		mpxplay_timer_addfunc((void *)&dvbepg_dialog_send_full_epg_timer_data, NULL, MPXPLAY_TIMERTYPE_WAKEUP | MPXPLAY_TIMERFLAG_MULTIPLY, 0);
	}
}

void DispQtDVBEPGMainWidget::dvbepg_reload_epginfos_full(void)
{
	this->dvbepgwidget_clear();
	if(!this->is_local_file_epg_handler)
	{
		this->parent_tabwidget->dvbscan_widget->dvb_scan_apply_settings();
		mpxplay_timer_addfunc((void *)&dvbepg_dialog_send_full_epg_timer_data, NULL, MPXPLAY_TIMERTYPE_WAKEUP | MPXPLAY_TIMERFLAG_MULTIPLY, 0);
		mpxplay_timer_addfunc((void *)&mpxplay_playlist_loaddir_dtvdrive_refresh, NULL, MPXPLAY_TIMERTYPE_WAKEUP, 0);
	}
}

bool DispQtDVBEPGMainWidget::dvbepgwidget_send_epginfos(struct mpxplay_dvb_epg_protocol_data_s *prot_data, char *currfilename)
{
	QDateTime epg_curr_time = this->dvbepg_eventsview_widget->get_epg_current_time();
	bool is_dvb_program_change;
	if(this->is_local_file_epg_handler)
	{
		if(pds_strcmp(this->current_local_filename, currfilename) != 0)
		{
			pds_strcpy(this->current_local_filename, currfilename);
			this->dvbepgwidget_clear();
			epg_curr_time = QDateTime();
		}
		if(prot_data->stream_datetime_val)
		{
			if(!epg_curr_time.isValid())
				this->dvbepg_eventsview_widget->set_eventsview_begin_time(epg_curr_time); // !!! resets min time, when stream time has arrived
			this->dvbepg_eventsview_widget->datetimeval_to_qdatetime(&epg_curr_time, prot_data->stream_datetime_val);
		}
	}
	else
	{
		epg_curr_time = QDateTime::currentDateTime();
	}
	is_dvb_program_change = this->dvbepg_eventsview_widget->eventsviewwidget_set_new_playfile(prot_data, currfilename, this->is_local_file_epg_handler);
	this->dvbepg_eventsview_widget->set_epg_current_time(epg_curr_time);
	if(this->dvbepg_eventsview_widget->eventsviewwidget_update_event_items(prot_data))
		this->dvbepg_stacked_widget->setCurrentIndex(DISPQT_DVBEPG_EPGVIEW_WIDGET);
	this->dvbepg_eventsview_widget->eventsviewwidget_consolidate_items();
	return is_dvb_program_change;
}

void DispQtDVBEPGMainWidget::dvbepgwidget_clear(void)
{
	this->dvbepg_stacked_widget->setCurrentIndex(DISPQT_DVBEPG_NOEPG_WIDGET);
	this->dvbepg_programs_widget->programswidget_clear();
	this->dvbepg_eventsview_widget->eventsviewwidget_clear();
	this->dvbepgwidget_set_zoom(1);
	emit this->dvbepg_eventsview_widget->signal_eventsviewwidget_item_focused(NULL);
}

void DispQtDVBEPGMainWidget::dvbepgwidget_set_zoom(int level)
{
	double scale = (double)level / 20;
	this->dvbepg_eventsview_widget->eventsviewwidget_set_widgetscale(scale);
	this->dvbepg_ruler_widget->rulerwidget_set_widgetscale(scale);
}

//-------------------------------------------------------------------------------------------------------------------------------

DispQtDVBEPGItem::DispQtDVBEPGItem(DispQtDVBEPGEventsViewWidget *view, DispQtDVBEPGProgram *program, struct mpxplay_dvb_epg_event_data_s *event_data) : QGraphicsItem()
{
	this->eventsview_widget = view;
	this->program_of_event = program;
	this->event_duration_secs = this->event_duration_gui = 0;
	this->event_is_fake = false;
	this->event_parental_rating = 0;
	m_boundingRect.setHeight(DISPQT_EPG_PROGRAM_HEIGHT);
	//setFlags(QGraphicsItem::ItemIsSelectable | QGraphicsItem::ItemIsFocusable);
	setAcceptHoverEvents(true);
	this->epgitem_set_eventdata(event_data);
}

QRectF DispQtDVBEPGItem::boundingRect() const
{
	return m_boundingRect;
}

void DispQtDVBEPGItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
	bool is_dark_background_used = this->eventsview_widget->main_window->mainwin_guilayout_is_dark_background();
	bool is_current_eventitem = (this->program_of_event->get_current_eventitem() == this);
	bool is_past_eventitem = (!is_current_eventitem && this->epgitem_is_past_event(this->eventsview_widget->get_epg_current_time()));
	bool is_playing_eventitem = (is_current_eventitem && this->eventsview_widget->is_played_program(this->program_of_event->get_protocol_id(), this->program_of_event->get_frequency_hz(), this->program_of_event->get_program_id()));
	QColor gradientcolor_start, gradientcolor_stop;
	QPen pen;

	// Draw in view's coordinates
	painter->setWorldMatrixEnabled(false);

	// Get the transformations required to map the text on the viewport
	QTransform viewPortTransform = this->eventsview_widget->viewportTransform();
	QRectF mapped = deviceTransform(viewPortTransform).mapRect(boundingRect());
	QLinearGradient gradient(mapped.topLeft(), mapped.bottomLeft());

	if(is_dark_background_used)
	{
		if(is_playing_eventitem)
		{
			gradientcolor_start.setRgb(0, 240, 255, 110);
			gradientcolor_stop.setRgb(0, 170, 180, 110);
		}
		else if(is_current_eventitem)
		{
			gradientcolor_start.setRgb(0, 180, 255, 110);
			gradientcolor_stop.setRgb(0, 130, 180, 110);
		}
		else if(is_past_eventitem)
		{
			gradientcolor_start.setRgb(0, 120, 202, 50);
			gradientcolor_stop.setRgb(0, 60, 100, 50);
		}
		else
		{
			gradientcolor_start.setRgb(0, 120, 202, 110);
			gradientcolor_stop.setRgb(0, 60, 100, 110);
		}
	}
	else
	{
		if(is_playing_eventitem)
			gradientcolor_stop.setRgb(100, 180, 200);
		else if(is_current_eventitem)
			gradientcolor_stop.setRgb(100, 150, 200);
		else
			gradientcolor_stop.setRgb(200, 220, 240);
		gradientcolor_start = gradientcolor_stop.lighter(120);
	}

	gradient.setColorAt(0.0, gradientcolor_start);
	gradient.setColorAt(1.0, gradientcolor_stop);

	pen.setColor(((option->state & QStyle::State_MouseOver) || hasFocus())? QColor(0, 0, 0) : ((is_dark_background_used)? QColor(64,96,128) : QColor(192, 192, 192)));
	pen.setStyle(((option->state & QStyle::State_MouseOver) && !hasFocus())? Qt::DashLine : Qt::SolidLine);

	painter->setBrush(QBrush(gradient));
	painter->setPen(pen);
	mapped.adjust(1, 2, -1, -2);
	painter->drawRoundedRect(mapped, 5, 5);

	// Setup the font and text color
	QFont f = painter->font();
	f.setPixelSize(11);
	f.setBold(true);
	painter->setFont(f);
	QFontMetrics fm = painter->fontMetrics();
	painter->setPen((is_dark_background_used)? Qt::white : Qt::black);

	// Draw the title
	mapped.adjust( 6, 3, -6, -3 );
	painter->drawText(mapped, Qt::AlignTop | Qt::AlignLeft, fm.elidedText( this->event_name, Qt::ElideRight, mapped.width()));

	// Draw the hours
	mapped.adjust(0, 15, 0, 0);
	f.setBold(false);
	painter->setFont(f);
	painter->drawText(mapped, Qt::AlignTop | Qt::AlignLeft,
						fm.elidedText( this->event_start_time.toString( "hh:mm" ) + " - " +
						this->event_start_time.addSecs(this->event_duration_secs).toString( "hh:mm" ),
						Qt::ElideRight, mapped.width() ) );
}

bool DispQtDVBEPGItem::epgitem_set_eventdata(struct mpxplay_dvb_epg_event_data_s *event_data)
{
	QDateTime start_time_event, start_time_gui;
	int32_t duration_event = pds_timeval_to_seconds(event_data->event_duration_time);
	int32_t duration_gui = duration_event;
	QString newname = QString::fromUtf8(event_data->event_shortdesc_name);

	this->eventsview_widget->datetimeval_to_qdatetime(&start_time_event, event_data->event_begin_date_time);

	start_time_gui = start_time_event;

	if(!this->epgitem_update_itemsize(start_time_gui, duration_gui))
		return false;

	if((start_time_gui != this->event_start_gui) || (duration_gui != this->event_duration_gui) || (newname != this->event_name)
	 || (event_data->event_shortdesc_details && this->event_shortDescription.isEmpty())
	 || (event_data->event_extended_desc && this->event_extDescription.isEmpty())
	){
		QString newshortdesc = QString::fromUtf8(event_data->event_shortdesc_details);
		QString newextdesc = QString::fromUtf8(event_data->event_extended_desc);
		this->event_start_time = start_time_event;
		this->event_start_gui = start_time_gui;
		this->event_duration_secs = duration_event;
		this->event_duration_gui = duration_gui;
		this->event_name = newname;
		this->event_shortDescription = newshortdesc;
		this->event_extDescription = newextdesc;
		this->event_parental_rating = event_data->parental_rating;
		if(funcbit_test(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_EPGWINDOW_ITEMS_TOOLTIP))
		{
			if( (this->eventsview_widget->main_window->ms_windows_version < MPXPLAY_MSWIN_VERSIONID_WIN80)
			 && this->eventsview_widget->main_window->mainwin_guilayout_is_nonclassic()
			 && !this->eventsview_widget->main_window->mainwin_guibkg_is_transparent()
			){  // QGraphicsItem ToolTip background is black in this case
				this->event_rich_hame = "<font color=\"white\">" + this->event_name + "</font>";
				this->setToolTip(this->event_rich_hame);
			}
			else
			{
				this->setToolTip(this->event_name);
			}
		}
		this->m_boundingRect.setWidth(duration_gui);
		this->epgitem_update_itempos();
		this->prepareGeometryChange();
		return true;
	}
	return false;
}

bool DispQtDVBEPGItem::epgitem_update_itemsize(QDateTime &start_time_gui, int32_t &duration_gui)
{
	QDateTime overallmin = this->eventsview_widget->get_eventsview_begintime();
	if(overallmin.isValid() && (start_time_gui < overallmin))
	{
		duration_gui -= start_time_gui.secsTo(overallmin);
		if(duration_gui <= 1)
			return false;
		start_time_gui = overallmin;
	}
	return true;
}

void DispQtDVBEPGItem::epgitem_update_itempos(void)
{
	QDateTime overallmin = this->eventsview_widget->get_eventsview_begintime();
	if(overallmin.isValid())
		this->setPos((qreal)overallmin.secsTo(this->event_start_gui), (qreal)(this->program_of_event->getPosition() * DISPQT_EPG_PROGRAM_HEIGHT));
}

void DispQtDVBEPGItem::epgitem_update_itemgeometry()
{
	this->event_start_gui = this->event_start_time;
	this->event_duration_gui = this->event_duration_secs;
	if(this->epgitem_update_itemsize(this->event_start_gui, this->event_duration_gui))
		this->m_boundingRect.setWidth(this->event_duration_gui);
	this->epgitem_update_itempos();
}

bool DispQtDVBEPGItem::epgitem_is_past_event(const QDateTime &ref_time) const
{
	return (this->event_start_time.addSecs(this->event_duration_secs) < ref_time);
}

bool DispQtDVBEPGItem::epgitem_is_curr_event(const QDateTime &ref_time) const
{
	return ((this->event_start_time <= ref_time) && !this->epgitem_is_past_event(ref_time));
}

void DispQtDVBEPGItem::hoverEnterEvent(QGraphicsSceneHoverEvent *event)
{
	event->accept();
	this->eventsview_widget->focusItem(this);
	scene()->update();
}

void DispQtDVBEPGItem::hoverLeaveEvent(QGraphicsSceneHoverEvent *event)
{
	hoverEnterEvent( event );
}

void DispQtDVBEPGItem::mouseDoubleClickEvent(QGraphicsSceneMouseEvent *event)
{
	this->program_of_event->program_activate();
}

//void DispQtDVBEPGItem::focusInEvent(QFocusEvent *event)
//{
//	event->accept();
//}

//-------------------------------------------------------------------------------------------------------------------------------

DispQtDVBEPGProgram::DispQtDVBEPGProgram(DispQtDVBEPGEventsViewWidget *view, struct mpxplay_dvb_epg_protocol_data_s *prot_data, struct mpxplay_dvb_epg_frequency_data_s *freq_data, struct mpxplay_dvb_epg_program_data_s *prog_data)
{
	this->eventsview_widget = view;
	this->protocol_id = prot_data->protocol_id;
	this->frequency_hz = freq_data->frequency;
	this->channel_id = prog_data->channel_id;
	this->program_id = prog_data->program_id;
	this->program_ctrl_flags = prog_data->ctrl_flags;
	this->program_name = QString::fromUtf8(prog_data->program_name);
	this->current_item = NULL;
	this->pos = 0;
	this->events_by_begintime.reserve(DISPQT_EPG_ITEMNUM_INITIAL_ALLOC);
	this->program_has_fake_events = false;
}

DispQtDVBEPGProgram::~DispQtDVBEPGProgram()
{
	this->events_by_begintime.clear();
}

void DispQtDVBEPGProgram::program_update_eventitems(void)
{
    foreach(const DispQtDVBEPGItem *item, this->events_by_begintime)
        ((DispQtDVBEPGItem *)item)->epgitem_update_itemgeometry();
}

void DispQtDVBEPGProgram::program_activate(void)
{
	this->eventsview_widget->eventsviewwidget_program_activate(this);
}

void DispQtDVBEPGProgram::program_update_events(struct mpxplay_dvb_epg_event_data_s *event_data,
		const mpxp_uint64_t limit_datetime_min_secs, const mpxp_uint64_t limit_datetime_max_secs,
		mpxp_uint64_t *min_datetime_val)
{
	if(event_data)
	{
		if(this->program_has_fake_events) // clear fake events, because valid has arrived
			this->program_clear_items();

		do
		{
			QHash<uint64_t, DispQtDVBEPGItem *>::iterator it = this->events_by_begintime.find(event_data->event_begin_date_time);
			DispQtDVBEPGItem *epgItem;

			if((it != this->events_by_begintime.end()) && it.value()) // update existent entry
			{
				epgItem = it.value();
				epgItem->epgitem_set_eventdata(event_data);
			}
			else
			{
				mpxp_uint64_t event_end_secs = pds_datetimeval_to_seconds(event_data->event_begin_date_time) + pds_timeval_to_seconds(event_data->event_duration_time);
				if((event_end_secs <= limit_datetime_min_secs) || (event_end_secs >= limit_datetime_max_secs)) // drop entry (it's out of valid time range)
				{
					event_data = event_data->next_event;
					continue;
				}
				else // add a new entry
				{
					epgItem = new DispQtDVBEPGItem(this->eventsview_widget, this, event_data);
					this->events_by_begintime.insert(event_data->event_begin_date_time, epgItem);
					this->eventsview_widget->scene()->addItem(epgItem);
				}
			}

			if(epgItem->epgitem_is_curr_event(this->eventsview_widget->get_epg_current_time()))
				current_item = epgItem;

			if(!(*min_datetime_val) || (event_data->event_begin_date_time < *min_datetime_val))
				*min_datetime_val = event_data->event_begin_date_time;
			event_data = event_data->next_event;
		}while(event_data);
	}
	else // create fake epgitem (to fill program line with something), if there's no epgitem (yet) for this program
	{
		DispQtDVBEPGItem *epgItem;
		struct mpxplay_dvb_epg_event_data_s fake_event;
		const char str_noepg[64] = "No EPG data";

		if((this->events_by_begintime.size() > 0) || !this->eventsview_widget->get_eventsview_begintime().isValid())
			return;

		pds_memset(&fake_event, 0, sizeof(fake_event));
		fake_event.event_begin_date_time = this->eventsview_widget->qdatetime_to_datetimeval(this->eventsview_widget->get_eventsview_begintime());
		fake_event.event_begin_date_time &= 0xFFFFFFFFFFFF0000UL; // round down to hours (keep date and hours only, clear minutes and seconds)
		fake_event.event_duration_time = 0X00180000 * 2;          // 2 days long (2 * 24 hours)
		fake_event.event_shortdesc_name = (char *)&str_noepg[0];

		epgItem = new DispQtDVBEPGItem(this->eventsview_widget, this, &fake_event);
		this->events_by_begintime.insert(fake_event.event_begin_date_time, epgItem);
		this->eventsview_widget->scene()->addItem(epgItem);

		this->program_has_fake_events = true;
	}
}

void DispQtDVBEPGProgram::program_clear_items(void)
{
	QHash<uint64_t, DispQtDVBEPGItem *>::iterator it = this->events_by_begintime.begin();

	while(it != this->events_by_begintime.end())
	{
		const DispQtDVBEPGItem *item = it.value();
		if(item)
		{
			this->eventsview_widget->scene()->removeItem((QGraphicsItem *)item);
			delete item;
		}
		it++;
	}
	this->events_by_begintime.clear();
	this->program_has_fake_events = false;
}

void DispQtDVBEPGProgram::program_remove_expired_events(const QDateTime &ref_datetime)
{
	QHash<uint64_t, DispQtDVBEPGItem *>::iterator it = this->events_by_begintime.begin();

    while(it != this->events_by_begintime.end())
    {
        const DispQtDVBEPGItem *item = it.value();
        if(item)
        {
        	if(item->epgitem_is_past_event(ref_datetime))
        	{
        		this->eventsview_widget->scene()->removeItem((QGraphicsItem *)item);
        		delete item;
        		it = this->events_by_begintime.erase(it);
        	}
        	else
        	{
        		it++;
        	}
        }
        else
        {
        	it = this->events_by_begintime.erase(it);
        }
    }
}

//-------------------------------------------------------------------------------------------------------------------------------

DispQtDVBEPGRulerWidget::DispQtDVBEPGRulerWidget(DispQtDVBEPGEventsViewWidget *view, QWidget* parent) : QWidget(parent)
{
	this->setContentsMargins(0, 0, 0, 0);
	this->setFixedHeight(DISPQT_EPG_RULER_HEIGHT);
	this->eventsview_widget = view;
	this->ruler_view_offset = 0;
	this->ruler_view_scale = 1.0;
	this->rulerwidget_config_style_apply(true);
}

void DispQtDVBEPGRulerWidget::rulerwidget_config_style_apply(bool initial)
{
	if(this->eventsview_widget->main_window->mainwin_guilayout_is_dark_background())
	{
		this->setStyleSheet("background: transparent; color: white");
	}
	else if(!initial)
	{
		this->setStyleSheet(QString());
	}
}

void DispQtDVBEPGRulerWidget::rulerwidget_set_widgetscale(double scale)
{
	this->ruler_view_scale = scale;
	update();
}

void DispQtDVBEPGRulerWidget::rulerwidget_set_starttime(const QDateTime& startTime)
{
	this->ruler_startTime = startTime;
	update();
}

void DispQtDVBEPGRulerWidget::rulerwidget_set_offset(int offset)
{
	if(offset < 0)
		offset = 0;
	this->ruler_view_offset = offset;
	update();
}

void DispQtDVBEPGRulerWidget::paintEvent(QPaintEvent *event)
{
	if(!this->ruler_startTime.isValid())
	{
		QWidget::paintEvent(event);
		return;
	}

	QPainter p(this);

	QFont f = p.font();
	f.setPixelSize(12);
	f.setBold(true);
	p.setFont(f);

	QDateTime localStartTime = this->ruler_startTime.addSecs(this->ruler_view_offset / this->ruler_view_scale);
	QDateTime localSTimeHour(localStartTime.date(), QTime(localStartTime.time().hour(), 0, 0, 0));
	const int secondsToHour = localStartTime.secsTo(localSTimeHour);

	// draw hour blocks with vertical line separation and hour text
	QDateTime currentTimePos(localStartTime.addSecs(secondsToHour - 3600));
	QPoint currentBlockPos(secondsToHour * this->ruler_view_scale, 0);
	QPoint previousBlockPos(-1, 0);
	const int spacing = this->ruler_view_scale * 3600;

	while(currentBlockPos.rx() < (width() + spacing))
	{
		QRect blockArea(QPoint(previousBlockPos.x() + 1, 0), QPoint(currentBlockPos.x(), DISPQT_EPG_RULER_HEIGHT));
		QString timeString = currentTimePos.toString("hh'h'");
		QColor fillColor;

		if (currentTimePos.time().hour() == 0) // Show Day
			timeString += currentTimePos.date().toString(" ddd dd");

		if (this->ruler_startTime.date().daysTo(currentTimePos.date()) & 1) // draw each second day with gray background
			fillColor = QColor(180, 180, 180, 160);
		else
			fillColor = palette().color(QPalette::Window);

		p.fillRect(blockArea, fillColor);
		p.drawLine(blockArea.topRight(), blockArea.bottomRight());
		p.drawText(blockArea, Qt::AlignLeft, timeString);

		previousBlockPos = currentBlockPos;
		currentBlockPos.rx() += spacing;
		currentTimePos = currentTimePos.addSecs(3600);
	}

	// draw current time position (a vertical red line)
	if(this->eventsview_widget->get_epg_current_time().isValid())
	{
		currentBlockPos.rx() = localStartTime.secsTo(this->eventsview_widget->get_epg_current_time()) * this->ruler_view_scale;
		if ((currentBlockPos.x() >= 0) && (currentBlockPos.x() <= width()))
		{
			p.setPen(QPen(Qt::red));
			p.drawLine(currentBlockPos, QPoint( currentBlockPos.x(), currentBlockPos.y() + DISPQT_EPG_RULER_HEIGHT));
		}
	}
}

//-------------------------------------------------------------------------------------------------------------------------------
DispQtDVBEPGProgramsWidget::DispQtDVBEPGProgramsWidget(DispQtDVBEPGEventsViewWidget *view, QWidget *parent) : QWidget(parent)
{
	setContentsMargins(0, 0, 0, 0);
	setFixedWidth(DISPQT_EPG_PROGRAMS_WIDTH);
	this->epgview_widget = view;
	this->programs_view_offset = 0;
	this->programswidget_config_style_apply(true);
}

void DispQtDVBEPGProgramsWidget::programswidget_config_style_apply(bool initial)
{
	if(this->epgview_widget->main_window->mainwin_guilayout_is_dark_background())
	{
		this->setStyleSheet("background: transparent; color: white");
	}
	else if(!initial)
	{
		this->setStyleSheet(QString());
	}
}

void DispQtDVBEPGProgramsWidget::programswidget_add_program(const DispQtDVBEPGProgram *program)
{
    if(!program_list.contains(program))
    {
        this->program_list.append(program);
        update();
    }
}

void DispQtDVBEPGProgramsWidget::programswidget_set_offset(int offset)
{
	this->programs_view_offset = offset;
	this->update();
}

void DispQtDVBEPGProgramsWidget::programswidget_clear(void)
{
	this->program_list.clear();
	this->update();
}

void DispQtDVBEPGProgramsWidget::paintEvent(QPaintEvent *event)
{
	QPainter p(this);

	QFont f = p.font();
	f.setPixelSize(12);
	f.setBold(true);
	p.setFont(f);

	unsigned int i = 0;
	foreach(const DispQtDVBEPGProgram *program, program_list)
	{
		p.drawText(0, - this->programs_view_offset + (i++ + 0.5) * DISPQT_EPG_PROGRAM_HEIGHT - 4, width(), height(), Qt::AlignLeft, program->get_program_name());
	}
}

//-------------------------------------------------------------------------------------------------------------------------------
DispQtDVBEPGGraphicsScene::DispQtDVBEPGGraphicsScene(QObject *parent) : QGraphicsScene(parent)
{

}

DispQtDVBEPGEventsViewWidget::DispQtDVBEPGEventsViewWidget(MainWindow *mainwindow, QWidget *parent, bool local_file_epg_handling) : QGraphicsView(parent)
{
	this->main_window = mainwindow;
	this->is_local_file_epg_handler = local_file_epg_handling;
	this->current_filename[0] = 0;
	this->current_protocol_id = 0;
	this->current_frequency_hz = 0;
	this->current_program_id = 0;

	setContentsMargins(0, 0, 0, 0);
	setAlignment( Qt::AlignLeft | Qt::AlignTop );
	setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	this->eventsviewwidget_config_style_apply(true);

	this->eventsviewwidget_initvals();

	this->vscrollbar = new DispQtDialogElemScrollBar(mainwindow, Qt::Vertical, this);
	this->setVerticalScrollBar(this->vscrollbar);
	this->hscrollbar = new DispQtDialogElemScrollBar(mainwindow, Qt::Horizontal, this);
	this->setHorizontalScrollBar(this->hscrollbar);

	this->epg_scene = new DispQtDVBEPGGraphicsScene(this);
	setScene(this->epg_scene);
}

DispQtDVBEPGEventsViewWidget::~DispQtDVBEPGEventsViewWidget()
{
	this->eventsviewwidget_clear();
}

void DispQtDVBEPGEventsViewWidget::eventsviewwidget_config_style_apply(bool initial)
{
	if(this->main_window->mainwin_guilayout_is_dark_background())
	{
		this->setStyleSheet("background: transparent");
		setFrameStyle(QFrame::StyledPanel);
	}
	else
	{
		if(!initial)
		{
			this->setStyleSheet(QString());
		}
		setFrameStyle(QFrame::Box);
	}
	if(!initial)
	{
		this->vscrollbar->scrollbar_set_style(initial);
		this->hscrollbar->scrollbar_set_style(initial);
	}
}

bool DispQtDVBEPGEventsViewWidget::eventsviewwidget_set_new_playfile(struct mpxplay_dvb_epg_protocol_data_s *prot_data, char *filename, bool is_local_file)
{
	bool is_dvb_program_change = false;
	if(is_local_file)
	{
		this->current_protocol_id = prot_data->protocol_id;
		this->current_frequency_hz = prot_data->frequency_data_chain->frequency;
		this->current_program_id = prot_data->curr_program_id;
	}
	else
	{
		this->current_program_id = mpxplay_dtvdrv_parse_filename(filename, &this->current_protocol_id, &this->current_frequency_hz);
		if((this->current_program_id > 0) && funcbit_test(prot_data->prot_flags, MPXPLAY_DVBEPG_PROTOCOL_FLAG_NEWFILE))
			is_dvb_program_change = true;
	}
	pds_strcpy(this->current_filename, filename);
	QApplication::setOverrideCursor(Qt::ArrowCursor);
	return is_dvb_program_change;
}

bool DispQtDVBEPGEventsViewWidget::is_played_program(int protocol_id, mpxp_int32_t frequency_hz, mpxp_int32_t program_id)
{
	if((program_id == this->current_program_id) && (frequency_hz == this->current_frequency_hz) && (protocol_id == this->current_protocol_id))
		return true;
	return false;
}

void DispQtDVBEPGEventsViewWidget::eventsviewwidget_set_widgetscale(double scaleFactor)
{
	this->eventsview_zoom_scalefactor = scaleFactor;
	QMatrix matrix;
	matrix.scale(scaleFactor, 1);
	setMatrix(matrix);
}

void DispQtDVBEPGEventsViewWidget::datetimeval_to_qdatetime(QDateTime *qdatetime, mpxp_uint64_t datetime_val)
{
	if(datetime_val)
	{
		int year = (datetime_val >> 48);
		int month = (datetime_val >> 40) & 0xFF;
		int mday = (datetime_val >> 32) & 0xFF;
		int hour = (datetime_val >> 16) & 0xFF;
		int minute = (datetime_val >> 8) & 0xFF;
		int seconds = datetime_val & 0xFF;
		*qdatetime = QDateTime(QDate(year, month, mday), QTime(hour, minute, seconds));
	} // keep QDateTime invalid
}

mpxp_uint64_t DispQtDVBEPGEventsViewWidget::qdatetime_to_datetimeval(QDateTime qdatetime)
{
	mpxp_uint64_t datetime_val;
	if(!qdatetime.isValid())
		return 0ULL;
	datetime_val = ((mpxp_uint64_t)qdatetime.date().year() << 48);
	datetime_val |= ((mpxp_uint64_t)qdatetime.date().month() << 40);
	datetime_val |= ((mpxp_uint64_t)qdatetime.date().day() << 32);
	datetime_val |= ((mpxp_uint64_t)qdatetime.time().hour() << 16);
	datetime_val |= ((mpxp_uint64_t)qdatetime.time().minute() << 8);
	datetime_val |= (mpxp_uint64_t)qdatetime.time().second();
	return datetime_val;
}

bool DispQtDVBEPGEventsViewWidget::eventsviewwidget_update_event_items(struct mpxplay_dvb_epg_protocol_data_s *prot_data)
{
	const bool is_local_file = (this->is_local_file_epg_handler && funcbit_test(prot_data->prot_flags, MPXPLAY_DVBEPG_PROTOCOL_FLAG_LOCALFILE));
	const QDateTime curr_epg_qdatetime = this->get_epg_current_time();
	const mpxp_uint64_t curr_datetime_val = (curr_epg_qdatetime.isValid())? this->qdatetime_to_datetimeval(curr_epg_qdatetime) : 0ULL;
	const mpxp_uint64_t curr_datetime_secs = pds_datetimeval_to_seconds(curr_datetime_val);
	const mpxp_uint64_t limit_datetime_min_secs = (curr_datetime_secs > 3600)? (curr_datetime_secs - (mpxplay_config_dvbepg_pastevents_hours * 60 * 60)) : 10ULL;
	const mpxp_uint64_t limit_datetime_max_secs = (curr_datetime_secs)? (curr_datetime_secs + (MPXPLAY_DVBEPG_EVENTS_LIMIT_MAX_HOURS * 60 * 60)) : INT64_MAX;
	mpxp_uint64_t min_datetime_val = curr_datetime_val;
	bool found_event = false, get_min_time;
	QDateTime qmintime;

	if(is_local_file)
	{
		get_min_time = true;
	}
	else
	{
		if(curr_epg_qdatetime.isValid())
			this->eventsview_begin_time = curr_epg_qdatetime.addSecs(-(qint64)(mpxplay_config_dvbepg_pastevents_hours * 60 * 60));
		get_min_time = !this->eventsview_begin_time.isValid(); // FIXME: probably min_time belongs to local file only (else eventsview_begin_time shall be valid here)
	}

	while(prot_data)
	{
		struct mpxplay_dvb_epg_frequency_data_s *freq_data = prot_data->frequency_data_chain;
		const uint32_t prot_id_s28 = prot_data->protocol_id << 28; // protocol_id is 4 bits max (15) (BDADEVTYPE_NUM is 11, MPXPLAY_DRVDTV_BDADEVTYPE_LOCALFILE is 15)

		while(freq_data)
		{
			struct mpxplay_dvb_epg_program_data_s *prog_data = freq_data->program_data_chain;
			const uint32_t freq_uid_number_s16 = prot_id_s28 | ((freq_data->frequency / 1000000) << 16); // freq_mhz is 12 bits max (4095)

			while(prog_data)
			{
				mpxp_uint32_t prog_uid_number = freq_uid_number_s16 | prog_data->program_id; // program_id is 16 bits max (65535)
		        QHash<uint32_t, DispQtDVBEPGProgram*>::iterator it = this->program_datas.find(prog_uid_number);
		        mpxp_uint64_t prog_datetime_min_val = curr_datetime_val;
		        DispQtDVBEPGProgram *program;

		        if(it != this->program_datas.end())
		        {
		            program = *it;
		            if(program->get_program_name().isEmpty() && prog_data->program_name)
		            	program->set_program_name(prog_data->program_name);
		        }
		        else
		        {
		            program = new DispQtDVBEPGProgram(this, prot_data, freq_data, prog_data);
		            program->setPosition(this->program_datas.count());
		            this->program_datas.insert(prog_uid_number, program);
		            emit signal_eventsviewwidget_program_added(program);
		        }

		        program->program_update_events(prog_data->epg_program_event_chain, limit_datetime_min_secs, limit_datetime_max_secs, &prog_datetime_min_val);

		        if(get_min_time && (!min_datetime_val || (prog_datetime_min_val < min_datetime_val)) && (pds_datetimeval_to_seconds(prog_datetime_min_val) >= limit_datetime_min_secs))
		        {
		        	min_datetime_val = prog_datetime_min_val;
		        }

				prog_data = prog_data->next_program;
				found_event = true;
			}
			freq_data = freq_data->next_frequency;
		}
		prot_data = prot_data->next_protocol;
	}

    if(get_min_time && min_datetime_val)
    {
    	this->datetimeval_to_qdatetime(&qmintime, min_datetime_val);
    	if(!this->eventsview_begin_time.isValid() || (qmintime < this->eventsview_begin_time))
    		this->eventsview_begin_time = qmintime;
    }

    this->update();

    return found_event;
}

void DispQtDVBEPGEventsViewWidget::eventsviewwidget_initvals(void)
{
	this->eventsview_begin_time = this->eventsview_prev_begin_time = QDateTime();
	this->eventsview_zoom_scalefactor = 1.0;
}

void DispQtDVBEPGEventsViewWidget::eventsviewwidget_clear()
{
	qDeleteAll(this->program_datas.values());
	this->program_datas.clear();
	this->scene()->clear();
	this->setScene(NULL);
	this->eventsviewwidget_initvals();
	delete this->epg_scene;
	this->epg_scene = new DispQtDVBEPGGraphicsScene(this); // FIXME: bullshit (I couldn't find better way to reset/reduce scene size)
	setScene(this->epg_scene);
}

void mpxplay_dispqt_dvbepg_program_start(struct dispqt_dvbepg_startstop_info_s *dvbinfo)
{
	if(dvbinfo)
	{
		mpxplay_playlist_loaddir_dvbprogram_start(dvbinfo->dvb_filename, dvbinfo->local_filename, dvbinfo->is_local_file);
		pds_free(dvbinfo);
	}
}

void DispQtDVBEPGEventsViewWidget::eventsviewwidget_program_activate(DispQtDVBEPGProgram *program)
{
	struct dispqt_dvbepg_startstop_info_s *dvbinfo = (struct dispqt_dvbepg_startstop_info_s *)pds_calloc(1, sizeof(struct dispqt_dvbepg_startstop_info_s));
	if(!dvbinfo)
		return;
	QApplication::setOverrideCursor(Qt::BusyCursor);
	dvbinfo->is_local_file = this->is_local_file_epg_handler;
	mpxplay_dtvdrv_assemble_filename(dvbinfo->dvb_filename, sizeof(dvbinfo->dvb_filename), program->get_protocol_id(), program->get_frequency_hz(), program->get_program_id());
	pds_strcpy(dvbinfo->local_filename, this->current_filename);
	mpxplay_dispqt_mainthread_callback_init((void *)mpxplay_dispqt_dvbepg_program_start, (void *)dvbinfo, 0);
}

void DispQtDVBEPGEventsViewWidget::focusItem(DispQtDVBEPGItem *item)
{
	emit signal_eventsviewwidget_item_focused(item);
}

void DispQtDVBEPGEventsViewWidget::eventsviewwidget_consolidate_items(void)
{
	if(this->eventsview_begin_time.isValid())
	{
		if(this->eventsview_begin_time != this->eventsview_prev_begin_time)
		{
			if(!this->is_local_file_epg_handler)
			{
				foreach(DispQtDVBEPGProgram *program, this->program_datas)
				{
					program->program_remove_expired_events(this->eventsview_begin_time);
				}
			}
			foreach(DispQtDVBEPGProgram *program, this->program_datas)
			{
				program->program_update_eventitems();
			}

			emit signal_eventsviewwidget_time_start_changed(this->eventsview_begin_time);

			this->eventsview_prev_begin_time = this->eventsview_begin_time;
		}
		else if(this->is_local_file_epg_handler) // hack to update ruler widget's current timepos
		{
			emit signal_eventsviewwidget_time_start_changed(this->eventsview_begin_time);
		}
	}
}

//------------------------------------------------------------------------------------------------------------------------------
DispQtEPGLocalFileWidget::DispQtEPGLocalFileWidget(MainWindow *mainwindow, DispQtEPGTabWidget *parent) : DispQtDVBEPGMainWidget(mainwindow, parent, true)
{
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG
