//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: Seekbar and time-pos in mainwindow

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_OUTPUT NULL // stdout
#define DISPQT_DEBUGOUT_PAINT stdout

#include <QtGui>
#include <QtWidgets>
#include "disp_qt.h"
#include "moc_seekbar.h"
#include "moc_mainwindow.h"
#include "moc_video_ctrlbar.h"
#include "display/display.h"

#define DISPQT_SEEKBAR_RESOLUTION  1000
#define DISPQT_SEEKBAR_SLIDER_BUTTON_SIZE_X  6
#define DISPQT_SEEKBAR_SLIDER_TOOLTIP_SHOWTIME  60000

extern "C" {
 extern struct mainvars mvps;
 extern unsigned int refdisp, timemode;
 extern mpxp_int64_t mpxplay_infmpeg_ffmpdmux_seekpreview_seek;
 extern mpxp_bool_t mpxplay_infmpeg_ffmpdmux_seekpreview_reset;
 extern keyconfig kb[];
}

extern struct dispqt_video_filter_format_s mpxplay_videoworkdec_seekpreview_format;

DispQtMainwinSeekbar::DispQtMainwinSeekbar(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	this->main_window = main_window;
	this->time_len_ms = 0;
	this->time_display_mode = timemode;
	this->setFocusPolicy(Qt::NoFocus);

	layout_seekbar = new QHBoxLayout;
	layout_seekbar->setSpacing(2);

	this->mainwinseekbar_config_style_apply(true);

	this->label_songnums = new QLabel;
	this->seek_slider = new DispQtMainwinSeekslider(main_window, this, false);
	this->label_times = new QLabel;

	layout_seekbar->addWidget(this->label_songnums, 0);
	layout_seekbar->addWidget(this->seek_slider, 1);
	layout_seekbar->addWidget(this->label_times, 0);
	this->setLayout(layout_seekbar);

	this->mainwinseekbar_set_timepos(0);

	connect(this, SIGNAL(signal_mainwinseekbar_set_timepos(int)), this, SLOT(mainwinseekbar_set_timepos(int)));
	connect(this, SIGNAL(signal_mainwinseekbar_option_timemode_change(unsigned int)), this, SLOT(mainwinseekbar_option_timemode_change(unsigned int)));
	//connect(this->label_songnums, SIGNAL(pressed()), this, SLOT(mainwinseekbar_timelabel_change_timemode()));
	//connect(this->label_times, SIGNAL(pressed()), this, SLOT(mainwinseekbar_timelabel_change_timemode()));

	refdisp |= RDT_HEADER;
}

void DispQtMainwinSeekbar::mainwinseekbar_config_style_apply(bool initial)
{
	this->mutex_seekbar.tryLock(DISPQT_MUTEX_TIMEOUT);

	if(this->main_window->mainwin_guilayout_is_dark_background())
		setStyleSheet("QWidget { background: transparent; color: white}");
	else
		setStyleSheet(QString());

	if(this->main_window->mainwin_guilayout_is_translucent_enabled() && (this->main_window->ms_windows_version < MPXPLAY_MSWIN_VERSIONID_WIN11))
		layout_seekbar->setMargin(0);
	else
		layout_seekbar->setContentsMargins(4,0,4,0);

	if(!initial)
		this->seek_slider->mainwin_seekslider_config_style_apply(initial);

	this->mutex_seekbar.unlock();
}

void DispQtMainwinSeekbar::mainwinseekbar_set_currfileinfos(struct dispqt_currfileinfo_s *cur)
{
	char sout[128];
	this->mutex_seekbar.tryLock(DISPQT_MUTEX_TIMEOUT);
	this->time_len_ms = cur->currtimelen;
	if(this->time_len_ms < 0)
		this->time_len_ms = 0;
	this->time_len_ms += 500;
	this->seek_slider->mainwin_seekslider_set_timelen(this->time_len_ms);
	snprintf(sout, sizeof(sout), "%2d / %d", cur->currsongnum, cur->allsongnum);
	this->label_songnums->setText(QString::fromLatin1(sout));
	this->mutex_seekbar.unlock();
}

void DispQtMainwinSeekbar::mainwinseekbar_set_timepos(int new_timepos_ms)
{
	char sout[128];
	this->mutex_seekbar.tryLock(DISPQT_MUTEX_TIMEOUT);
	this->seek_slider->mainwin_seekslider_set_timepos(new_timepos_ms);
	if(this->time_display_mode & 1)
		new_timepos_ms = this->time_len_ms - new_timepos_ms;
	else
		new_timepos_ms += 500;
	if(new_timepos_ms < 0)
		new_timepos_ms = 0;
	if(this->time_len_ms < 3600000)
		snprintf(sout, sizeof(sout), "%d:%2.2d / %d:%2.2d", (new_timepos_ms / 60000), ((new_timepos_ms % 60000) / 1000), (this->time_len_ms / 60000), ((this->time_len_ms % 60000) / 1000));
	else
		snprintf(sout, sizeof(sout), "%d:%2.2d:%2.2d / %d:%2.2d:%2.2d", (new_timepos_ms / 3600000), (new_timepos_ms / 60000) % 60, ((new_timepos_ms % 60000) / 1000),
									(this->time_len_ms / 3600000), (this->time_len_ms / 60000) % 60, ((this->time_len_ms % 60000) / 1000));
	this->label_times->setText(QString::fromUtf8(sout));
	this->mutex_seekbar.unlock();
}

void DispQtMainwinSeekbar::mainwinseekbar_option_timemode_change(unsigned int new_timemode)
{
	this->mutex_seekbar.tryLock(DISPQT_MUTEX_TIMEOUT);
	this->time_display_mode = timemode;
	this->mutex_seekbar.unlock();
}

void DispQtMainwinSeekbar::mainwinseekbar_timelabel_change_timemode(void)
{
	this->mutex_seekbar.tryLock(DISPQT_MUTEX_TIMEOUT);
	timemode = (timemode + 1) & 3;
	this->mutex_seekbar.unlock();
	emit this->signal_mainwinseekbar_option_timemode_change(timemode);
}

//=========================================================================================================================

DispQtMainwinSeekslider::DispQtMainwinSeekslider(MainWindow *main_window, QWidget *parent, bool control_bar) : QSlider(Qt::Horizontal, parent)
{
	this->main_window = main_window;
	this->parent_widget = parent;
	this->is_control_bar = control_bar;
	this->mouse_is_pressed = mouse_is_moved_while_pressed = false;
	this->time_len_ms = 0;
	this->last_mousecursor_globalpos_x = this->last_mousecursor_widgetpos_x = -1;
	this->setFocusPolicy(Qt::NoFocus);
	this->setTickPosition(QSlider::NoTicks);
	this->setRange(0, DISPQT_SEEKBAR_RESOLUTION);
	this->setSliderPosition(0);
	this->setMouseTracking(true);
	this->tooltip_time_text[0] = 0;
	this->seek_preview = NULL;
	this->mainwin_seekslider_config_style_apply(true);
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	if(this->is_control_bar)
	{
		this->seek_preview = new DispQtSeekPreviewWidget(main_window, parent);
	}
#endif
}

DispQtMainwinSeekslider::~DispQtMainwinSeekslider()
{
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	if(this->seek_preview)
	{
		DispQtSeekPreviewWidget *spw = this->seek_preview;
		this->seek_preview = NULL;
		delete spw;
	}
#endif
}

void DispQtMainwinSeekslider::mainwin_seekslider_config_style_apply(bool initial)
{
	switch(this->main_window->mainwin_guilayout_get_type())
	{
		case DISPQT_GUILAYOUT_OSDEF_CLASSIC:
			if(!initial)
				setStyleSheet(QString());
			break;
		case DISPQT_GUILAYOUT_BLUE_CLASSIC:
		case DISPQT_GUILAYOUT_BLUE_TRANSPARENT:
			this->setStyleSheet(
				"QSlider::groove:horizontal { background: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1,"
					"stop: 0 transparent, stop: 0.25 rgba(0,60,110,130), stop: 0.75 rgba(0,60,110,130), stop: 1 transparent);"
					"height: 8px; border-radius: 2px; }"
				"QSlider::groove:horizontal:hover { background: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1,"
					"stop: 0 transparent, stop: 0.10 rgba(0,60,110,130), stop: 0.90 rgba(0,60,110,130), stop: 1 transparent);"
					"height: 9px; border-radius: 2px; }"
				"QSlider::sub-page:horizontal { background: qlineargradient(x1: 0, y1: 0.2, x2: 0, y2: 0.8, stop: 0 #66e, stop: 1 #bbf);"
					"background: qlineargradient(x1: 0, y1: 0.2, x2: 1, y2: 0.8, stop: 0 rgba(0,60,110,130), stop: 1 rgba(0,150,255,200));"
					"height: 8px; border-radius: 2px;}"
//				"QSlider::sub-page:horizontal {"
//					"background: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 rgba(0,82,144,110), stop: 0.5 rgba(0,72,120,110), stop: 0.6 rgba(0,60,100,110), stop:1 rgba(0,90,152,110));"
//					"border: 1px solid rgb(64,96,128);"
//					"height: 9px; border-radius: 2px;}"
				"QSlider::handle:horizontal { background: rgba(0,255,255,200); width: 8px; border-radius: 4px; }"
				"QSlider::handle:horizontal:hover { background: qlineargradient(x1:0, y1:0, x2:1, y2:1, stop:0 #fff, stop:1 #ddd);"
					"width: 9px; border-radius: 4px; }"
//				"QSlider::sub-page:horizontal:disabled { background: #bbb; border-color: #999; }"
//				"QSlider::add-page:horizontal:disabled { background: #eee; border-color: #999; }"
//				"QSlider::handle:horizontal:disabled { background: #eee; border: 1px solid #aaa; border-radius: 4px; }"
			);
			break;
	}
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	if(this->is_control_bar)
	{
		this->tooltip_palette = QToolTip::palette();
		this->tooltip_palette.setColor(QPalette::All, QPalette::ToolTipText, Qt::black);
		this->tooltip_palette.setColor(QPalette::All, QPalette::ToolTipBase, Qt::white);
		QToolTip::setPalette(this->tooltip_palette);
	}
#endif
}

void DispQtMainwinSeekslider::mainwin_seekslider_set_timelen(int new_timelen_ms)
{
	this->time_len_ms = new_timelen_ms;
	this->mouse_is_pressed = false;
	this->mouse_is_moved_while_pressed = false;

	if((this->last_mousecursor_globalpos_x >= 0) && (this->last_mousecursor_widgetpos_x >= 0))
		this->seekbar_show_timepos_and_seekpreview(this->last_mousecursor_globalpos_x, this->last_mousecursor_widgetpos_x);
}

void DispQtMainwinSeekslider::mainwin_seekslider_set_timepos(int new_timepos_ms)
{
	int newpos;
	if(this->mouse_is_pressed)
		return;
	if(this->time_len_ms){
		newpos = (int)((float)DISPQT_SEEKBAR_RESOLUTION * (float) new_timepos_ms / (float) this->time_len_ms);
		if(newpos > DISPQT_SEEKBAR_RESOLUTION)
			newpos = DISPQT_SEEKBAR_RESOLUTION;
		else if(newpos < 0)
			newpos = 0;
	} else {
		newpos = 0;
	}
	this->setSliderPosition(newpos);
}

void DispQtMainwinSeekslider::seekbar_show_timepos_and_seekpreview(int cursor_globalpos_x, int cursor_widgetpos_x)
{
	int seekbar_width = this->width() - DISPQT_SEEKBAR_SLIDER_BUTTON_SIZE_X;
	int time_pos_ms = (int)((float) cursor_widgetpos_x * (float)this->time_len_ms / (float)seekbar_width);
	int parent_widget_globalpos_y = this->parent_widget->y();
	int tooltip_y_pos = parent_widget_globalpos_y + ((this->is_control_bar)? -35 : (this->main_window->isMaximized()? 0 : 15));

	if(this->time_len_ms < 3600000)
		snprintf(this->tooltip_time_text, sizeof(this->tooltip_time_text), "%d:%2.2d", (time_pos_ms / 60000), ((time_pos_ms % 60000) / 1000));
	else
		snprintf(this->tooltip_time_text, sizeof(this->tooltip_time_text), "%d:%2.2d:%2.2d", (time_pos_ms / 3600000), (time_pos_ms / 60000) % 60, ((time_pos_ms % 60000) / 1000));

	QToolTip::showText(QPoint(cursor_globalpos_x - 18, tooltip_y_pos), QString::fromLatin1(this->tooltip_time_text), this, rect(), DISPQT_SEEKBAR_SLIDER_TOOLTIP_SHOWTIME);

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	if(this->is_control_bar && funcbit_test(this->main_window->gui_config->video_control, DISPQT_CONFIG_VIDEOCTRL_SEEKPREVIEW_ENABLE))
	{
		this->seek_preview->show_window(cursor_globalpos_x, cursor_widgetpos_x, parent_widget_globalpos_y, seekbar_width);
		//QToolTip::setPalette(this->tooltip_palette);
	}
#endif
}

void DispQtMainwinSeekslider::wheelEvent(QWheelEvent *event)
{
    this->main_window->mainwin_handle_keypresscode(kb[(event->delta() < 0)? 0 : 2].c);  // seek rewind / forward with wheel
	event->accept();
}

void DispQtMainwinSeekslider::mousePressEvent(QMouseEvent *event)
{
	this->mouse_is_moved_while_pressed = false;
    if(event->button() & Qt::LeftButton){
        this->mouse_is_pressed = true;
        QSlider::mousePressEvent(event);
    } else {
        this->main_window->mainwin_handle_mousepress_skip_or_seek_event(event);
    }
}

static mpxp_int64_t DispQtMainwinSeekslider_calc_file_seek(int x, int w)
{
	mpxp_int64_t seek_frame;
	if(x <= DISPQT_SEEKBAR_SLIDER_BUTTON_SIZE_X)
		seek_frame = 1;
	else
		seek_frame = (mpxp_int64_t) ((float)mvps.fr_primary->index_len * (float)x / (float)w) + (mpxp_int64_t)mvps.fr_primary->index_start + 1;
	return seek_frame;
}

void DispQtMainwinSeekslider::mouseReleaseEvent(QMouseEvent *event)
{
    if(this->mouse_is_pressed) {
        this->mouse_is_pressed = false;
        int x = event->x(), w = this->width() - DISPQT_SEEKBAR_SLIDER_BUTTON_SIZE_X;
        if(!this->mouse_is_moved_while_pressed) { // to avoid duplicated seek
        	mvps.seek_absolute = (long)DispQtMainwinSeekslider_calc_file_seek(x, w);
        	mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "mouseReleaseEvent x:%d w:%d sa:%d", x, w, mvps.seek_absolute);
        }
        QSlider::mouseReleaseEvent(event);
        this->main_window->mainwindow_set_focus(); // FIXME: bullshit
    }else{
    	this->main_window->mainwin_handle_mousepress_skip_or_seek_event(event);
    }
}

void DispQtMainwinSeekslider::mouseMoveEvent(QMouseEvent *event)
{
	int cursor_globalpos_x = event->globalPos().x();
	int cursor_widgetpos_x = event->x();

	this->last_mousecursor_globalpos_x = cursor_globalpos_x;
	this->last_mousecursor_widgetpos_x = cursor_widgetpos_x;

	this->seekbar_show_timepos_and_seekpreview(cursor_globalpos_x, cursor_widgetpos_x);

	if(this->mouse_is_pressed) {
		int seekbar_width = this->width() - DISPQT_SEEKBAR_SLIDER_BUTTON_SIZE_X;
		int slider_pos = (int)((float) cursor_widgetpos_x * (float)DISPQT_SEEKBAR_RESOLUTION / (float)seekbar_width);
		this->setSliderPosition(slider_pos);
		mvps.seek_absolute = (long)DispQtMainwinSeekslider_calc_file_seek(cursor_widgetpos_x, seekbar_width);
		this->mouse_is_moved_while_pressed = true;
	}
}

bool DispQtMainwinSeekslider::event(QEvent *event)
{
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	switch(event->type()){
		case QEvent::Close:
			if(this->seek_preview)
				this->seek_preview->hide_window();
			break;
		case QEvent::Leave:
			this->last_mousecursor_globalpos_x = -1;
			if(this->seek_preview)
				this->seek_preview->fadeout_window_start();
			//QToolTip::hideText(); // it seems this is not needed
			break;
	}
#endif
	return QSlider::event(event);
}

void DispQtMainwinSeekslider::keyPressEvent(QKeyEvent *event)
{
    this->main_window->handle_keypressevent(event);
}

void DispQtMainwinSeekslider::seekslider_preview_send(void *preview_frame)
{
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	if(this->seek_preview)
		this->seek_preview->paint_window(preview_frame);
#endif
}

void DispQtMainwinSeekslider::seekslider_preview_hide(void)
{
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	if(this->seek_preview)
		this->seek_preview->hide_window();
#endif
}

//=========================================================================================================================
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

#define DISPQT_SEEKPREVIEW_BORDER_SIZE             1 // in pixels
#define DISPQT_SEEKPREVIEW_ASPECT_RATIO_MAX        4 // max 4 : 1 aspect ratio (wide window)
#define DISPQT_SEEKPREVIEW_TIMEOUT_INTERVAL      200 // msec in timer to avoid seek jamming

DispQtSeekPreviewWidget::DispQtSeekPreviewWidget(MainWindow *main_window, QWidget *parent) : QWidget(NULL, Qt::FramelessWindowHint | Qt::WindowDoesNotAcceptFocus | Qt::CustomizeWindowHint | Qt::WindowStaysOnTopHint)
{
	this->main_window = main_window;
	this->parent_window = parent;
	this->decoded_video_frame = this->output_video_frame = NULL;
	this->switch_on_window = false;
	this->clear_window = true;
	this->initiated_seekpos = this->requested_seekpos = -1;
	this->set_window_size();
	this->timer_timeout.setSingleShot(true);
	this->timer_interval.setSingleShot(true);
	this->timer_fadeout.setSingleShot(false);
	connect(&timer_timeout, SIGNAL(timeout()), this, SLOT(timeout_seek_activate()));
	connect(&timer_fadeout, SIGNAL(timeout()), this, SLOT(fadeout_window_timerfunc()));
	setStyleSheet("background-color: black");
}

DispQtSeekPreviewWidget::~DispQtSeekPreviewWidget()
{
	this->hide();
	this->reset_window();
	if(this->decoded_video_frame)
		av_frame_free(&this->decoded_video_frame);
}

void DispQtSeekPreviewWidget::reset_window(void)
{
	this->output_video_frame = NULL;
	this->switch_on_window = false;
	this->clear_window = true;
	this->requested_seekpos = this->initiated_seekpos = -1;
	this->opacity_curr = DISPQT_VIDEOCTRLBAR_OPACITY_MAX;
	this->timer_timeout.stop();
	this->timer_interval.stop();
	this->timer_fadeout.stop();
	mpxplay_infmpeg_ffmpdmux_seekpreview_reset = TRUE;
}

unsigned long DispQtSeekPreviewWidget::get_aspectratio_type(void)
{
	unsigned long ar_type = this->main_window->gui_config->videoseekpreview_aspect_ratio_type;
	if(ar_type == DISPQT_VIDEOFULLSCREENARTYPE_NONE)
		ar_type =  this->main_window->gui_config->video_fullscreen_ar_type;
	return ar_type;
}

void DispQtSeekPreviewWidget::set_window_size(void)
{
	const int screen_number = this->main_window->video_get_fullscreen_display_number();
	const QRect screenGeometry = QApplication::desktop()->screenGeometry(screen_number);
	int screen_max_y = screenGeometry.height();
	this->screen_max_x = screenGeometry.width();
	this->screen_pos_x0 = screenGeometry.x();
	this->window_size_y = screen_max_y * this->main_window->gui_config->videoseekpreview_size_percent / 100;
	this->window_size_y = ((this->window_size_y + 3) & ~3) + (2 * DISPQT_SEEKPREVIEW_BORDER_SIZE);
	switch(get_aspectratio_type())
	{
		case DISPQT_VIDEOFULLSCREENARTYPE_FILL: this->window_size_x = this->window_size_y * this->screen_max_x / screen_max_y; break;
		case DISPQT_VIDEOFULLSCREENARTYPE_4_3: this->window_size_x = this->window_size_y * 4 / 3; break;
		case DISPQT_VIDEOFULLSCREENARTYPE_235_1: this->window_size_x = this->window_size_y * 235 / 100; break;
		case DISPQT_VIDEOFULLSCREENARTYPE_1_1: this->window_size_x = this->window_size_y; break;
		case DISPQT_VIDEOFULLSCREENARTYPE_CUSTOM: this->window_size_x = this->window_size_y * mpxplay_dispqt_videotrans_videooutput_get_custom_ar_value(this->main_window->gui_config->video_ar_custom, 1000, 1000) / 1000; break;
		default: this->window_size_x = this->window_size_y * 16 / 9; break; // default is 16:9
	}
	this->window_size_x = ((this->window_size_x + 4) & ~7) + (2 * DISPQT_SEEKPREVIEW_BORDER_SIZE);
	this->clear_window = true;
	this->setFixedSize(this->window_size_x, this->window_size_y);
}

void DispQtSeekPreviewWidget::timeout_seek_activate(void)
{
	if(this->requested_seekpos >= 0)
	{
		mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "timeout_seek_activate rsp:%d", this->requested_seekpos);
		mpxplay_infmpeg_ffmpdmux_seekpreview_seek = this->initiated_seekpos = this->requested_seekpos;
		this->requested_seekpos = -1;
		unsigned int refresh_interval_ms = this->main_window->gui_config->videoseekpreview_refresh_interval_ms;
		unsigned int timeout_ms = max(DISPQT_SEEKPREVIEW_TIMEOUT_INTERVAL, refresh_interval_ms);
		this->timer_timeout.start(timeout_ms);
		if(refresh_interval_ms)
			this->timer_interval.start(refresh_interval_ms);
	}
	else
	{
		this->initiated_seekpos = -1;
	}
}

void DispQtSeekPreviewWidget::show_window(int globalpos_x, int eventpos_x, int sliderpos_y, int seekbar_width)
{
	if(!this->switch_on_window && this->isHidden())
	{
		this->reset_window();
		this->switch_on_window = true;
		this->set_window_size();
		unsigned long ar_type = this->get_aspectratio_type();
		if(ar_type == DISPQT_VIDEOFULLSCREENARTYPE_VIDEO)
			mpxplay_videoworkdec_seekpreview_format.width = 0; // !!! this signals video aspect ratio for seekpreview
		else if(ar_type == DISPQT_VIDEOFULLSCREENARTYPE_PIXEL)
			mpxplay_videoworkdec_seekpreview_format.width = 1; // !!! this signals pixel aspect ratio for seekpreview
		else
			mpxplay_videoworkdec_seekpreview_format.width = this->window_size_x - (2 * DISPQT_SEEKPREVIEW_BORDER_SIZE);
		mpxplay_videoworkdec_seekpreview_format.height = this->window_size_y - (2 * DISPQT_SEEKPREVIEW_BORDER_SIZE);
		mpxplay_videoworkdec_seekpreview_format.pixelformat = DISPQT_VIDEO_QIMAGE_OUTPUT_AVFORMAT;
	}

	int winsize_x = this->width(), move_x = globalpos_x - (winsize_x / 2);
	if(move_x < this->screen_pos_x0)
		move_x = this->screen_pos_x0;
	else if(move_x > (this->screen_pos_x0 + this->screen_max_x - winsize_x))
		move_x = this->screen_pos_x0 + this->screen_max_x - winsize_x;
	int move_y = sliderpos_y - this->height() - 18;
	this->move(move_x, move_y);

	this->requested_seekpos = DispQtMainwinSeekslider_calc_file_seek(eventpos_x, seekbar_width) - 1;
	mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "show_window gx:%d x:%d w:%d rsp:%d isp:%d", globalpos_x, eventpos_x, seekbar_width, this->requested_seekpos, this->initiated_seekpos);
	if(this->timer_interval.isActive())
		this->timer_timeout.start((this->initiated_seekpos < 0)? min(this->timer_interval.remainingTime(), DISPQT_SEEKPREVIEW_TIMEOUT_INTERVAL) : DISPQT_SEEKPREVIEW_TIMEOUT_INTERVAL);
	else if(this->initiated_seekpos < 0)
		this->timeout_seek_activate();
}

void DispQtSeekPreviewWidget::refresh_window(int globalpos_x, int eventpos_x, int sliderpos_y, int seekbar_width)
{
	if(!this->isHidden())
		this->show_window(globalpos_x, eventpos_x, sliderpos_y, seekbar_width);
}

void DispQtSeekPreviewWidget::fadeout_window_start(void)
{
	if(this->timer_fadeout.isActive())
	{
		return;
	}
	if(this->isHidden())
	{
		this->reset_window();
		return;
	}
	if(funcbit_test(this->main_window->gui_config->video_control, DISPQT_CONFIG_VIDEOCTRL_CONTROLBAR_FADEOUT))
	{
		this->opacity_curr = DISPQT_VIDEOCTRLBAR_OPACITY_MAX;
		this->timer_fadeout.start(DISPQT_VIDEOCTRLBAR_FADEOUT_TIMERINTERVAL);
	}
	else
	{
		this->hide_window();
	}
}

void DispQtSeekPreviewWidget::fadeout_window_timerfunc(void)
{
	if(this->opacity_curr > DISPQT_VIDEOCTRLBAR_OPACITY_MIN)
	{
		this->setWindowOpacity((qreal)this->opacity_curr / (qreal)DISPQT_VIDEOCTRLBAR_OPACITY_MAX);
		this->opacity_curr -= DISPQT_VIDEOCTRLBAR_FADEOUT_OPACITYSTEP;
	}
	else
	{
		this->hide_window();
	}
}

void DispQtSeekPreviewWidget::hide_window(void)
{
	DispQtVideoControlbar *videoCtrlbar = (DispQtVideoControlbar *)this->parent_window;
	QPoint qp = QCursor::pos();
	this->hide();
	this->reset_window();
	if(videoCtrlbar->videoctrlbar_check_show(qp.x(), qp.y())) // FIXME: this is a hack to handle wrong/missing leave event of video control bar
		this->parent_window->raise(); // to drop always on top to parent window
}

void DispQtSeekPreviewWidget::paint_window(void *preview_frame_p)
{
	AVFrame *av_decoded_frame = (AVFrame *)preview_frame_p;

	this->output_video_frame = NULL;

	mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "paint_window isp:%d swo:%d %8.8X", this->initiated_seekpos, this->switch_on_window, (uint32_t)preview_frame_p);

	unsigned long ar_type = this->get_aspectratio_type();
	if(av_decoded_frame && ((ar_type == DISPQT_VIDEOFULLSCREENARTYPE_VIDEO) || (ar_type == DISPQT_VIDEOFULLSCREENARTYPE_PIXEL)))
	{
		int new_windowsize_x = av_decoded_frame->width + (2 * DISPQT_SEEKPREVIEW_BORDER_SIZE);
		if((this->window_size_x != new_windowsize_x) && (new_windowsize_x >= 10) && (new_windowsize_x < this->screen_max_x) && (new_windowsize_x <= (this->window_size_y * DISPQT_SEEKPREVIEW_ASPECT_RATIO_MAX))) // detection of possible input frame width error
		{
			this->window_size_x = new_windowsize_x;
			this->setFixedSize(this->window_size_x, this->window_size_y);
			this->clear_window = true;
		}
	}

	if(this->decoded_video_frame)
		av_frame_free(&this->decoded_video_frame);

	this->decoded_video_frame = av_decoded_frame;
	if(this->decoded_video_frame && (this->decoded_video_frame->format != DISPQT_VIDEO_QIMAGE_OUTPUT_AVFORMAT)) // detection of possible input frame format error (normally this should not happen)
		av_frame_free(&this->decoded_video_frame);

	this->output_video_frame = this->decoded_video_frame;

	if(this->switch_on_window || !this->isHidden())
	{
		this->repaint();
		if(this->switch_on_window)
		{
			this->setWindowOpacity((qreal)this->opacity_curr / (qreal)DISPQT_VIDEOCTRLBAR_OPACITY_MAX);
			this->show();
			this->switch_on_window = false;
		}
		this->raise(); // to keep always on top
		if(!this->timer_interval.isActive())
			timeout_seek_activate();
	}
	this->initiated_seekpos = -1;
}

void DispQtSeekPreviewWidget::paintEvent(QPaintEvent *event)
{
	AVFrame *output_frame = this->output_video_frame;
	QPainter p;

	p.begin(this);
	if(this->clear_window || !output_frame)
	{
		this->clear_window = false;
		p.eraseRect(0, 0, this->width(), this->height());
		mpxplay_debugf(DISPQT_DEBUGOUT_PAINT, "paintEvent BLACK w:%d h:%d", this->width(), this->height());
	}
	if(output_frame)
	{
		QImage image(output_frame->data[0], output_frame->width, output_frame->height, output_frame->linesize[0], DISPQT_VIDEO_QIMAGE_OUTPUT_QFORMAT);
		p.drawImage(QRectF(DISPQT_SEEKPREVIEW_BORDER_SIZE, DISPQT_SEEKPREVIEW_BORDER_SIZE, this->width() - (2 * DISPQT_SEEKPREVIEW_BORDER_SIZE), this->height() - (2 * DISPQT_SEEKPREVIEW_BORDER_SIZE)), image);
		mpxplay_debugf(DISPQT_DEBUGOUT_PAINT, "paintEvent IMG w:%d h:%d vw:%d vh:%d", this->width(), this->height(), preview_frame->width, preview_frame->height);
	}
	p.end();
}

bool DispQtSeekPreviewWidget::event(QEvent *event)
{
	switch(event->type()){
		case QEvent::Enter: break;
		case QEvent::Close: this->hide_window(); break;
		case QEvent::Leave: this->fadeout_window_start(); break;
		case QEvent::Move: break;
		default: goto err_out_event;
	}

err_out_event:
	return QWidget::event(event);
}

void DispQtSeekPreviewWidget::keyPressEvent(QKeyEvent *event)
{
    this->main_window->handle_keypressevent(event);
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG
