//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: (bottom) video control bar

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_OUTPUT stdout

#include <QtGui>
#include <QtWidgets>
#include <QtWinExtras/QtWinExtras>
#include "disp_qt.h"
#include "moc_video_ctrlbar.h"
#include "moc_seekbar.h"
#include "moc_mainwindow.h"
#include "moc_textwin.h"
#include "moc_video_qt.h"

#define DISPQT_VIDEOCTRLBAR_SIZE_Y_AERO         80  // Win7 Aero window cannot be smaller than this!
#define DISPQT_VIDEOCTRLBAR_SIZE_Y_WIN10       100  // Win10/11 transparent/blur window size (with header)
#define DISPQT_VIDEOCTRLBAR_SIZE_Y_HEADERLESS   70  // simple black transparent window (headerless)

#define DISPQT_VIDEOCTRLBAR_OPACITY_MIN             0 //
#define DISPQT_VIDEOCTRLBAR_OPACITY_MAX           100 //
#define DISPQT_VIDEOCTRLBAR_FADEOUT_TIMERINTERVAL  18 // fadeout timer interval
#define DISPQT_VIDEOCTRLBAR_FADEOUT_OPACITYSTEP    10 // fadeout opacity steps -> 20*10 = 200ms to fade out

#define DISPQT_VIDEOCTRLBAR_REFESH_DELAY           10 // delayed windows resize refresh (to apply properly changes)

extern "C" {
 //extern unsigned int timemode;
 extern keyconfig kb[];
}

DispQtVideoControlbar::DispQtVideoControlbar(MainWindow *main_window, QWidget *parent) : DispQtDialogElemWindow(main_window, parent, Qt::CoverWindow | Qt::WindowDoesNotAcceptFocus, DISPQT_DIALOG_WINDOWTYPE_VIDEOCTRLBAR)
{
	this->main_window = main_window;
	this->is_show_enabled = false;
	this->is_headerless_window = false;
	this->is_window_automoved = false;
	this->is_window_manualmoved = false;
	this->calculated_winpos_y = 0;
	this->calculated_winsize_y = DISPQT_VIDEOCTRLBAR_SIZE_Y_WIN10;
	this->opacity_max = this->opacity_curr = DISPQT_VIDEOCTRLBAR_OPACITY_MAX;
	this->timer_fadeout.setSingleShot(false);
	this->timer_refresh.setSingleShot(true);

	this->setFocusPolicy(Qt::NoFocus);

	optionToolBar = new QToolBar(tr("VidOpt"), this);
	playToolBar = new QToolBar(tr("VidPlay"), this);

	this->videoctrlbar_config_style_apply(true);

	this->seek_slider = new DispQtMainwinSeekslider(main_window, this, true);

	this->number = new QLabel;
	this->number->setContentsMargins(0,0,0,0);
	QFont font = this->number->font();
	font.setPointSize(12);
	this->number->setFont(font);

	optionToolBar->setContentsMargins(0,0,0,0);
	optionToolBar->addAction(main_window->action_view_videofs);
	optionToolBar->addAction(main_window->action_list_replay);
	optionToolBar->addAction(main_window->action_play_highlight);
	optionToolBar->addAction(main_window->action_view_videowall);

	playToolBar->setContentsMargins(0,0,0,0);
	playToolBar->addAction(main_window->action_play_skipback);
	playToolBar->addAction(main_window->action_play_seekback);
	playToolBar->addAction(main_window->action_play_start);
	playToolBar->addAction(main_window->action_play_stop);
	playToolBar->addAction(main_window->action_play_seekforward);
	playToolBar->addAction(main_window->action_play_skipforward);

	QToolButton *button_rewind = dynamic_cast<QToolButton*>(playToolBar->widgetForAction(main_window->action_play_seekback));
	button_rewind->setAutoRepeat(true);
	button_rewind->setAutoRepeatDelay(100);
	button_rewind->setAutoRepeatInterval(1000/30);
	QToolButton* button_forward = dynamic_cast<QToolButton*>(playToolBar->widgetForAction(main_window->action_play_seekforward));
	button_forward->setAutoRepeat(true);
	button_forward->setAutoRepeatDelay(100);
	button_forward->setAutoRepeatInterval(1000/30);

	QVBoxLayout *layout_ctrlbar = new QVBoxLayout;
	layout_ctrlbar->setMargin(0);
	layout_ctrlbar->setSpacing(0);
	layout_ctrlbar->addWidget(this->seek_slider, 1);

	QWidget *widget_play = new QWidget;
	QHBoxLayout *layout_play = new QHBoxLayout;
	widget_play->setContentsMargins(0,0,0,0);
	layout_play->setMargin(0);
	layout_play->setSpacing(0);
	layout_play->addWidget(optionToolBar, 1, Qt::AlignLeft | Qt::AlignVCenter);
	layout_play->addWidget(playToolBar, 1, Qt::AlignCenter);
	layout_play->addWidget(this->number, 1, Qt::AlignRight | Qt::AlignVCenter);
	widget_play->setLayout(layout_play);
	layout_ctrlbar->addWidget(widget_play, 0);

	this->setLayout(layout_ctrlbar);

	connect(this, SIGNAL(signal_videoctrlbar_config_style_apply(bool)), this, SLOT(videoctrlbar_config_style_apply(bool)));
	connect(this, SIGNAL(signal_videoctrlbar_set_show(bool)), this, SLOT(videoctrlbar_set_show(bool)));
	connect(this, SIGNAL(signal_videoctrlbar_set_timepos(int)), this, SLOT(videoctrlbar_set_timepos(int)));
	connect(this, SIGNAL(signal_videoctrlbar_option_timemode_change(unsigned int)), this, SLOT(videoctrlbar_option_timemode_change(unsigned int)));
	connect(this, SIGNAL(signal_videoctrlbar_seekbar_send_seekpreview(void *)), this, SLOT(videoctrlbar_seekbar_send_seekpreview(void *)));
	connect(&timer_fadeout, SIGNAL(timeout()), this, SLOT(videoctrlbar_fadeout_window_timerfunc()));
	connect(&timer_refresh, SIGNAL(timeout()), this, SLOT(videoctrlbar_set_size_and_pos()));

	this->hide();
}

DispQtVideoControlbar::~DispQtVideoControlbar()
{
	if(this->seek_slider)
	{
		DispQtMainwinSeekslider *mss = this->seek_slider;
		this->seek_slider = NULL;
		delete mss;
	}
}

bool DispQtVideoControlbar::event(QEvent *event)
{
	bool cursor_hide = false;
	switch(event->type()){
		case QEvent::Enter: this->videoctrlbar_show_window(); break;
		case QEvent::Leave: if(this->is_headerless_window) // else the 'leave' is checked by videoctrlbar_check_show from DispQtSeekPreviewWidget::hide_window // TODO: a better solution
		                        this->videoctrlbar_fadeout_window_start();
		                    mpxplay_debugf(DISPQT_DEBUG_OUTPUT,"LEAVE: um:%d", (int)this->underMouse());
		                    cursor_hide = true;
		                    break;
		case QEvent::Move:  if(!this->is_window_automoved)
		                    {
		                    	this->is_window_manualmoved = true;
		                    	this->videoctrlbar_set_size_and_pos();
		                    }
		                    else
		                    {
		                    	this->is_window_automoved = false;
		                    }
		                    break;
		default: goto err_out_event;
	}
	if(this->main_window->mainwin_is_on_video())
		emit this->main_window->signal_video_mouse_cursor_show(cursor_hide);

err_out_event:
	return QDialog::event(event);
}

void DispQtVideoControlbar::keyPressEvent(QKeyEvent *event)
{
	this->main_window->handle_keypressevent(event);
}

void DispQtVideoControlbar::wheelEvent(QWheelEvent *event)
{
	this->main_window->mainwin_handle_keypresscode(kb[(event->delta() < 0)? 0 : 2].c);  // KeyRewind1 / KeyForward1
	event->accept();
}

void DispQtVideoControlbar::mousePressEvent(QMouseEvent *event)
{
	Qt::MouseButton ebutton = event->button();
	if((ebutton & Qt::MidButton) || (ebutton & Qt::RightButton))
		this->main_window->mainwin_handle_mousepress_skip_or_seek_event(event);
	else
		QDialog::mousePressEvent(event);
}

void DispQtVideoControlbar::mouseReleaseEvent(QMouseEvent *event)
{
	Qt::MouseButton ebutton = event->button();
	if((ebutton & Qt::MidButton) || (ebutton & Qt::RightButton))
		this->main_window->mainwin_handle_mousepress_skip_or_seek_event(event);  // to handle press or click
	else
		QDialog::mouseReleaseEvent(event);
}

void DispQtVideoControlbar::videoctrlbar_set_size_and_pos(void)
{
	const unsigned int winVer = this->main_window->ms_windows_version;
	const bool is_gui_nonclassic = ((winVer >= MPXPLAY_MSWIN_VERSIONID_WIN80) || this->main_window->mainwin_guilayout_is_nonclassic())? true : false;
	const int screen_number = this->main_window->video_get_fullscreen_display_number();
	const QRect screenGeometry = QApplication::desktop()->screenGeometry(screen_number);
	const int screenpos_x = screenGeometry.x();
	const int screenpos_y = screenGeometry.y();
	const int max_x = screenGeometry.width();
	const int max_y = screenGeometry.height();
	int winsize_x = max_x * this->main_window->gui_config->videocontrolbar_size_percent / 100;
	const int newpos_x = screenpos_x + (max_x - winsize_x) / 2;
	int winsize_y = (this->is_headerless_window)? DISPQT_VIDEOCTRLBAR_SIZE_Y_HEADERLESS : ((winVer < MPXPLAY_MSWIN_VERSIONID_WIN80)? DISPQT_VIDEOCTRLBAR_SIZE_Y_AERO : DISPQT_VIDEOCTRLBAR_SIZE_Y_WIN10);
	int newpos_y = screenpos_y + max_y - winsize_y - ((this->is_headerless_window)? 0 : 4);

	this->calculated_winpos_y = newpos_y;
	this->calculated_winsize_y = winsize_y;

	if(this->is_headerless_window)
	{
		this->setContentsMargins(0, 0, 2, 10);
	}
	else if( is_gui_nonclassic
	     && ((winVer < MPXPLAY_MSWIN_VERSIONID_WIN80) || (winVer >= MPXPLAY_MSWIN_VERSIONID_WIN11)) // ??? related to Win7 Aero and Win11 only?
	     && ( this->is_window_manualmoved
	      || (this->main_window->dialog_handler && this->main_window->dialog_handler->was_dialog_used()))
	){  // FIXME: this is a workaround (for a Qt bug?) to set correct window size with and without opening a dialog (F1..F8) or after manually trying to move the video control bar
		this->setContentsMargins(4, 0, 4, 4);
		winsize_x -= 16;
		winsize_y = DISPQT_VIDEOCTRLBAR_SIZE_Y_HEADERLESS - 6;
		newpos_y = screenpos_y + max_y - winsize_y - 40;
	}
	else
	{
#ifdef MPXPLAY_ARCH_X64
		const int margin_right = (is_gui_nonclassic)? 20 : 14;
		const int margin_bottom = (winVer < MPXPLAY_MSWIN_VERSIONID_WIN80)? 24 : 38;
#else
		const int margin_right = 10;
		const int margin_bottom = 24;
#endif
		this->setContentsMargins(4, 0, margin_right, margin_bottom);
	}

	this->is_window_automoved = true;
	this->move(newpos_x, newpos_y);
	this->setFixedSize(winsize_x, winsize_y);
}

void DispQtVideoControlbar::videoctrlbar_config_style_apply(bool initial)
{
	struct mmc_dispqt_config_s *gcfg = this->main_window->gui_config;
	const bool isHeaderlessCtrlBar = (funcbit_test(gcfg->video_control, DISPQT_CONFIG_VIDEOCTRL_CONTROLBAR_HEADERLESS))? true : false;
	const unsigned int winVer = this->main_window->ms_windows_version;

	this->opacity_max = DISPQT_VIDEOCTRLBAR_OPACITY_MAX;

	if(initial || (isHeaderlessCtrlBar != this->is_headerless_window))
	{
		if(isHeaderlessCtrlBar)
		{
			if(!initial)
			{
#ifdef Q_OS_WIN
				if(winVer >= MPXPLAY_MSWIN_VERSIONID_WIN80)
				{
					mpxplay_dispqt_dialog_elems_reset_windowcomposition_win10((HWND)this->window()->winId());
				}
#endif
			}
			setWindowFlags((windowFlags() & ~Qt::CoverWindow) | Qt::Dialog | Qt::FramelessWindowHint | Qt::WindowDoesNotAcceptFocus);
			if(isHeaderlessCtrlBar != this->is_headerless_window)
			{
				gcfg->videocontrolbar_size_percent = DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_DEF_FULL;
			}
			setAttribute(Qt::WA_TranslucentBackground, true);
			setAttribute(Qt::WA_NoSystemBackground, false);
			setStyleSheet("QDialog { background: rgba(0,0,0,20); } QLabel { color:white; }");
		}
		else if(!initial)
		{
			setWindowFlags((windowFlags() & ~(Qt::Dialog | Qt::FramelessWindowHint)) | Qt::CoverWindow | Qt::WindowDoesNotAcceptFocus);
			if(isHeaderlessCtrlBar != this->is_headerless_window)
			{
				gcfg->videocontrolbar_size_percent = DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_DEF_HALF;
			}
		}
		this->is_headerless_window = isHeaderlessCtrlBar;
		this->videoctrlbar_set_size_and_pos();
	}

	if(!isHeaderlessCtrlBar)
	{
		if(this->main_window->mainwin_guibkg_is_transparent())
		{
#ifdef Q_OS_WIN
			if(winVer >= MPXPLAY_MSWIN_VERSIONID_WIN80)
			{
				if(this->main_window->mainwin_guilayout_is_translucent_enabled())
				{
					this->dialogwin_config_get_colors();
					mpxplay_dispqt_dialog_elems_setwindowcomposition_win10((HWND)this->window()->winId(), this->color_background_hexa, winVer);
				}
				else if(!initial)
				{
					mpxplay_dispqt_dialog_elems_reset_windowcomposition_win10((HWND)this->window()->winId());
				}
				QtWin::enableBlurBehindWindow(this); // to set full transparent without blur effect too
			}
#endif
			QtWin::extendFrameIntoClientArea(this, -1, -1, -1, -1);
			setAttribute(Qt::WA_TranslucentBackground, true);
			setAttribute(Qt::WA_NoSystemBackground, false);
		}
		else if(this->main_window->mainwin_guibkg_setting_is_transparent())
		{
			this->opacity_max = DISPQT_CONFIG_MAINWIN_OPACITY_DIALOG_DEFAULT;
		}
		else if(!initial)
		{
#ifdef Q_OS_WIN
			if(winVer >= MPXPLAY_MSWIN_VERSIONID_WIN80)
			{
				mpxplay_dispqt_dialog_elems_reset_windowcomposition_win10((HWND)this->window()->winId());
			}
#endif
			QtWin::resetExtendedFrame(this);
			setAttribute(Qt::WA_TranslucentBackground, false);
		}

		if(this->main_window->mainwin_guilayout_is_dark_background())
		{
			setStyleSheet("QLabel { color:white; }");
		}
		else if(!initial)
		{
			setStyleSheet(QString());
		}

		this->dialogwin_config_style_color();
	}

	this->setWindowTitle(this->current_title);

	if(!initial)
	{
		this->seek_slider->mainwin_seekslider_config_style_apply(initial);
	}

	this->main_window->mainwin_guilayout_toolbutton_setstyle(this->optionToolBar, initial);
	this->main_window->mainwin_guilayout_toolbutton_setstyle(this->playToolBar, initial);

	this->videoctrlbar_set_size_and_pos();
}

void DispQtVideoControlbar::videoctrlbar_set_show(bool enable)
{
	if(enable)
		this->videoctrlbar_set_size_and_pos();
	else if(this->is_show_enabled)
		this->videoctrlbar_hide_window();
	this->is_show_enabled = enable;
}

bool DispQtVideoControlbar::videoctrlbar_check_show(int x, int y)
{
	if(!this->is_show_enabled || !funcbit_test(this->main_window->gui_config->video_control, DISPQT_CONFIG_VIDEOCTRL_ENABLE_CONTROLBAR))
		return false;
	int pos_x = this->pos().x(), pos_y = this->calculated_winpos_y, size_x = this->width(), size_y = this->calculated_winsize_y;
	if((x >= pos_x) && (x < (pos_x + size_x)) && (y >= pos_y) && (y < (pos_y + size_y))) {
		this->videoctrlbar_show_window();
		return true;
	} else
		this->videoctrlbar_fadeout_window_start();
	return false;
}

void DispQtVideoControlbar::videoctrlbar_show_window(void)
{
	if(!this->is_show_enabled && !funcbit_test(this->main_window->gui_config->video_control, DISPQT_CONFIG_VIDEOCTRL_ENABLE_CONTROLBAR))
		return;
	this->opacity_curr = this->opacity_max;
	this->setWindowOpacity((qreal)this->opacity_curr / (qreal)DISPQT_VIDEOCTRLBAR_OPACITY_MAX);
	this->timer_fadeout.stop();
	if(this->isHidden())
	{
		this->videoctrlbar_set_size_and_pos();
		this->show();
		this->timer_refresh.start(DISPQT_VIDEOCTRLBAR_REFESH_DELAY);
	}
}

void DispQtVideoControlbar::videoctrlbar_hide_window(void)
{
	this->seek_slider->seekslider_preview_hide();
	this->timer_fadeout.stop();
	this->hide();
	this->main_window->raise(); // to drop always on top to parent window
}

void DispQtVideoControlbar::videoctrlbar_fadeout_window_start(void)
{
	if(this->isHidden() || this->timer_fadeout.isActive())
		return;
	if(funcbit_test(this->main_window->gui_config->video_control, DISPQT_CONFIG_VIDEOCTRL_CONTROLBAR_FADEOUT))
	{
		this->opacity_curr = this->opacity_max;
		this->timer_fadeout.start(DISPQT_VIDEOCTRLBAR_FADEOUT_TIMERINTERVAL);
	}
	else
	{
		this->videoctrlbar_hide_window();
	}
}

void DispQtVideoControlbar::videoctrlbar_fadeout_window_timerfunc(void)
{
	if(this->opacity_curr > DISPQT_VIDEOCTRLBAR_OPACITY_MIN)
	{
		this->setWindowOpacity((qreal)this->opacity_curr / (qreal)DISPQT_VIDEOCTRLBAR_OPACITY_MAX);
		this->opacity_curr -= DISPQT_VIDEOCTRLBAR_FADEOUT_OPACITYSTEP;
	}
	else
	{
		this->videoctrlbar_hide_window();
	}
}

void DispQtVideoControlbar::videoctrlbar_set_currfileinfos(struct dispqt_currfileinfo_s *cur)
{
	this->time_len_ms = cur->currtimelen;
	this->seek_slider->mainwin_seekslider_set_timelen(cur->currtimelen);
	this->current_title = QString::fromUtf8(cur->curr_title);
	this->setWindowTitle(this->current_title);
	mpxplay_debugf(DISPQT_DEBUG_OUTPUT,"title: %s", cur->curr_title);
}

void DispQtVideoControlbar::videoctrlbar_set_timepos(int new_timepos_ms)
{
	char sout[128];
	emit this->seek_slider->mainwin_seekslider_set_timepos(new_timepos_ms);
	if(this->time_len_ms < 3600000) {
		snprintf(sout, sizeof(sout), "%d:%2.2d / %d:%2.2d", (new_timepos_ms / 60000), ((new_timepos_ms % 60000) / 1000),
					(this->time_len_ms / 60000), ((this->time_len_ms % 60000) / 1000) );
	} else {
		snprintf(sout, sizeof(sout), "%d:%2.2d:%2.2d / %d:%2.2d:%2.2d", (new_timepos_ms / 3600000), (new_timepos_ms / 60000) % 60, ((new_timepos_ms % 60000) / 1000),
					(this->time_len_ms / 3600000), (this->time_len_ms / 60000) % 60, ((this->time_len_ms % 60000) / 1000) );
	}
	this->number->setText(QString::fromLatin1(sout));
}

void DispQtVideoControlbar::videoctrlbar_option_timemode_change(unsigned int mode)
{
	//emit this->seek_bar->signal_mainwinseekbar_option_timemode_change(mode);
}

void DispQtVideoControlbar::videoctrlbar_seekbar_send_seekpreview(void *preview_frame_p)
{
	if(!this->isHidden() && !this->timer_fadeout.isActive())
		this->seek_slider->seekslider_preview_send(preview_frame_p);
}
