//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: video software based transformations (crop, pad, eq)

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_OUTPUT NULL //stdout
#define DISPQT_DEBUG_ERROR stderr
#define DISPQT_DEBUGOUT_FFMPEG NULL
#define DISPQT_DEBUGOUT_RENDER NULL // stdout
#define DISPQT_DEBUGOUT_CROP NULL // stdout
#define DISPQT_DEBUGOUT_ACROP stdout

#include "moc_video_qt.h"
#include "moc_mainwindow.h"

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

FFMpegVideoSoftwareTransform::FFMpegVideoSoftwareTransform(MainWindow *mainwindow, dispqt_video_source_info_s *video_source_infos, dispqt_video_surface_info_s *video_surface_infos)
{
	this->main_window = mainwindow;
	this->video_source_infos = video_source_infos;
	this->video_surface_infos = video_surface_infos;

	video_surface_infos->video_filter_infos = &this->video_filter_infos;

	mpxplay_dispqt_videotrans_autocrop_clear(video_source_infos);
	this->video_source_infos->last_video_crop_type = mainwindow->gui_config->video_crop_type;

	memset(&this->video_filter_infos, 0, sizeof(this->video_filter_infos));

	this->videowidget_ffmpeg_videoctx_init(&this->video_filter_infos);
	this->videowidget_ffmpeg_swsctx_clear(&this->video_filter_infos);
}

FFMpegVideoSoftwareTransform::~FFMpegVideoSoftwareTransform()
{
	videowidget_ffmpeg_swsctx_close();
	videowidget_ffmpeg_videoctx_close(&this->video_filter_infos);
}

// --------------------------------------------------------------------------------------------------------------------

int FFMpegVideoSoftwareTransform::videowidget_ffmpeg_videoctx_init(struct dispqt_video_filter_info_s *video_filter_infos)
{
	int retcode = MPXPLAY_ERROR_OK;
	video_filter_infos->ffmpeg_filtered_frame = av_frame_alloc();
	this->videowidget_ffmpeg_videoctx_clear();
	return retcode;
}

void FFMpegVideoSoftwareTransform::videowidget_ffmpeg_videoctx_close(struct dispqt_video_filter_info_s *video_filter_infos)
{
	this->videowidget_ffmpeg_videoctx_clear();
	av_frame_free(&video_filter_infos->ffmpeg_transformed_frame);
	av_frame_free(&video_filter_infos->ffmpeg_filtered_frame);
}

void FFMpegVideoSoftwareTransform::videowidget_ffmpeg_videoctx_clear(void)
{
	av_frame_free(&this->video_source_infos->ffmpeg_decoded_frame);
	this->video_surface_infos->ffmpeg_output_frame = NULL;
	this->videowidget_ffmpeg_videoctx_reconfig();
}

void FFMpegVideoSoftwareTransform::videowidget_ffmpeg_videoctx_reconfig(void)
{
	struct dispqt_video_filter_info_s *video_filter_infos = &this->video_filter_infos;
	video_filter_infos->ffmpeg_filtered_frame->format = this->video_surface_infos->videoout_surface_pix_fmt;
}

//-------------------------------------------------------------------------------------------------------------------------------------
bool mpxplay_dispqt_videotrans_crop_change_check(struct dispqt_video_source_info_s *video_source_infos, struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct mmc_dispqt_config_s *gcfg = video_surface_infos->gui_config;
	bool check_autocrop = false;

	if((video_source_infos->last_video_crop_type != gcfg->video_crop_type) || (video_source_infos->last_video_autocrop_limit != gcfg->video_autocrop_limit))
	{
		if(!video_source_infos->ffmpeg_swsframe_cropreset)
			video_source_infos->ffmpeg_swsframe_cropreset = DISPQT_VIDEO_AUTOCROP_COUNTER_REFRESH;
		video_source_infos->sws_autoxcrop_framecounter = 0;
		video_source_infos->last_video_crop_type = gcfg->video_crop_type;
		video_source_infos->last_video_autocrop_limit = gcfg->video_autocrop_limit;
		video_surface_infos->videoout_surface_reset = true;
		check_autocrop = true;
		mpxplay_debugf(DISPQT_DEBUGOUT_ACROP, "crop_change_check");
	}

	return check_autocrop;
}

void FFMpegVideoSoftwareTransform::videowidget_ffmpeg_filtered_crop_update(bool check_autocrop)
{
	struct dispqt_video_source_info_s *video_source_infos = this->video_source_infos;
	struct dispqt_video_filter_info_s *video_filter_infos = &this->video_filter_infos;
	struct mmc_dispqt_config_s *gcfg = this->main_window->gui_config;

	if(mpxplay_dispqt_videotrans_crop_change_check(video_source_infos, this->video_surface_infos))
		check_autocrop = true;

	if((gcfg->video_crop_type == DISPQT_VIDEOSCRCROPTYPE_AUTO) && check_autocrop)
	{
		AVFrame *autocrop_input_frame = ((video_filter_infos->ffmpeg_swsframe_hasdata)? video_filter_infos->ffmpeg_sws_frame : ((video_filter_infos->ffmpeg_transformed_frame)? video_filter_infos->ffmpeg_transformed_frame : this->video_source_infos->ffmpeg_decoded_frame)); // if the video is RGB, we use the original frame
		if( autocrop_input_frame && (autocrop_input_frame->height <= gcfg->video_autocrop_limit)
		 && (autocrop_input_frame->format != mpxplay_video_render_infos.hwdevice_avpixfmt) && !mpxplay_video_render_infos.render_function_poolbufelem_validity_check(autocrop_input_frame)
		){
			if(mpxplay_dispqt_videotrans_autocrop_process(autocrop_input_frame->data[0], autocrop_input_frame->linesize[0], autocrop_input_frame->format,
					autocrop_input_frame->width, autocrop_input_frame->height, autocrop_input_frame->linesize[0] / autocrop_input_frame->width,
					video_source_infos, &this->video_surface_infos->videoout_surface_clearpic))
			{
				this->video_surface_infos->videoout_surface_recalc = true;
			}
		}
	}
}

void FFMpegVideoSoftwareTransform::videowidget_ffmpeg_filtered_trans_update(void)
{
	struct dispqt_video_filter_info_s *video_filter_infos = &this->video_filter_infos;

	av_frame_free(&video_filter_infos->ffmpeg_transformed_frame);
	if(this->video_source_infos->ffmpeg_decoded_frame)
	{
		DispQtVideoFrameFilter *vfilter_process_trans = this->main_window->qt_video_player->video_frame_filter_transform_process;
		if(!mpxplay_dispqt_videotrans_videoframe_hw_rotate_check(this->video_surface_infos, this->video_source_infos->ffmpeg_decoded_frame))
		{
			if(vfilter_process_trans->video_filter_check(this->video_source_infos->ffmpeg_decoded_frame, AV_PIX_FMT_NONE) != AV_PIX_FMT_NONE)
			{
				DispQtVideoFrameFilter *vfilter_process_eq = this->main_window->qt_video_player->video_frame_filter_eq_process;
				int frameout_format = (vfilter_process_eq->video_filter_check(this->video_source_infos->ffmpeg_decoded_frame, AV_PIX_FMT_NONE) == AV_PIX_FMT_NONE)? this->video_surface_infos->videoout_surface_pix_fmt : DISPQT_VIDEO_FILTERPROCESS_AVFORMAT;
				vfilter_process_trans->video_filter_apply(this->video_source_infos->ffmpeg_decoded_frame, &video_filter_infos->ffmpeg_transformed_frame, frameout_format);
			}
		}
		video_filter_infos->ffmpeg_swsframe_hasdata = videowidget_ffmpeg_swsctx_conv_frame(video_filter_infos);
		this->videowidget_ffmpeg_filtered_crop_update(true);
	}
	this->videowidget_ffmpeg_filtered_eq_update();
}

void FFMpegVideoSoftwareTransform::videowidget_ffmpeg_filtered_eq_update(void)
{
	struct dispqt_video_filter_info_s *video_filter_infos = &this->video_filter_infos;
	AVFrame *filter_in_frame;

	this->video_surface_infos->ffmpeg_output_frame = NULL;
	av_frame_unref(video_filter_infos->ffmpeg_filtered_frame);
	filter_in_frame = (video_filter_infos->ffmpeg_transformed_frame)? video_filter_infos->ffmpeg_transformed_frame : this->video_source_infos->ffmpeg_decoded_frame;
	if(filter_in_frame)
	{
		DispQtVideoFrameFilter *vfilter_process_eq = this->main_window->qt_video_player->video_frame_filter_eq_process;
		bool swsframe_for_outsurface = videowidget_ffmpeg_swsctx_check_swsframe_for_outsurface(video_filter_infos);
		if( (!swsframe_for_outsurface && (filter_in_frame->format != this->video_surface_infos->videoout_surface_pix_fmt))
		 || (!(this->video_surface_infos->video_render_infos->renderer_capabilities & DISPQT_VIDEORENDERER_CAPFLAG_COLORMIXER) && (vfilter_process_eq->video_filter_check(filter_in_frame, AV_PIX_FMT_NONE) != AV_PIX_FMT_NONE))
		){
			int retVal = vfilter_process_eq->video_filter_apply(filter_in_frame, &video_filter_infos->ffmpeg_filtered_frame, this->video_surface_infos->videoout_surface_pix_fmt);
			if(retVal & DISPQT_VIDEOFILTER_APPLYRETBIT_OUTPUTFRAME_PRODUCED)
				this->video_surface_infos->ffmpeg_output_frame = video_filter_infos->ffmpeg_filtered_frame;
			mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "eq_update r:%d sw:%d sh:%d dw:%d dh:%d", retVal, filter_in_frame->width, filter_in_frame->height, video_surface_infos->ffmpeg_output_frame->width, video_surface_infos->ffmpeg_output_frame->height);
		}
		else
		{
			this->video_surface_infos->ffmpeg_output_frame = (swsframe_for_outsurface)? video_filter_infos->ffmpeg_sws_frame : filter_in_frame;
			mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "eq_update pass");
		}
	}
}

//-------------------------------------------------------------------------------------------------------------------------------------
#define DISPQT_VIDEO_AUTOCROP_FASTMODE 0  // less precision, more speed on YUV planar types

#if DISPQT_VIDEO_AUTOCROP_FASTMODE
#define DISPQT_VIDEO_AUTOCROP_LIMITMASK_YUV42X   0xe0e0e0e0 // YUV 8 bits
#define DISPQT_VIDEO_AUTOCROP_LIMITMASK_YUV42X10 0xe000e000 // YUV 10/16 bits
#else
#define DISPQT_VIDEO_AUTOCROP_LIMITMASK_YUV42X   0x000000e0 // YUV 8 bits
#define DISPQT_VIDEO_AUTOCROP_LIMITMASK_YUV42X10 0x0000e000 // YUV 10/16 bits
#endif
#define DISPQT_VIDEO_AUTOCROP_LIMITMASK_YUYV422  0x00e000e0 // YUY2
#define DISPQT_VIDEO_AUTOCROP_LIMITMASK_RGB32    0x00e0e0e0 // RGB32
#define DISPQT_VIDEO_AUTOCROP_LIMITMASK_RGB64    0xe000e000 // RGB64
// TODO: some rare color formats are not supported natively by autocrop (eg: AV_PIX_FMT_BGR565LE)

bool mpxplay_dispqt_videotrans_autocrop_process(uint8_t *data_begin, const int data_linesize, const int data_format,
		const int video_width, const int video_height, const int data_elemsize,
		dispqt_video_source_info_s *video_source_infos, mpxp_bool_t *videooutsurface_clearpic)
{
	const int skip_border_y = DISPQT_CONFIG_VIDEO_AUTOCROP_SKIP_BORDER;
	const int check_range = 100 / DISPQT_CONFIG_VIDEO_AUTOCROP_CHECK_RANGE;
	int x, y, check_len, check_step = 1, skip_border_x, counter, crop_x = 0, crop_y = 0, crop_width = 0, crop_height = 0;
	uint8_t *data_ptr = (uint8_t *)data_begin;
	uint32_t limit_mask;
	bool retval = false;

	if(video_source_infos->ffmpeg_swsframe_cropreset == DISPQT_VIDEO_AUTOCROP_COUNTER_INIT){
		video_source_infos->sws_srccrop_x = 0;
		video_source_infos->sws_srccrop_y = 0;
		video_source_infos->sws_srccrop_width = video_width;
		video_source_infos->sws_srccrop_height = video_height;
		video_source_infos->ffmpeg_swsframe_cropreset --;
		video_source_infos->sws_autoxcrop_framecounter = 0;
		*videooutsurface_clearpic = true;
		retval = true;
		mpxplay_debugf(DISPQT_DEBUGOUT_ACROP, "AUTOCROP0 x:%d y:%d w:%d h:%d sw:%d sh:%d", crop_x, crop_y, crop_width, crop_height, video_width, video_height);
	}
	if(!data_ptr || (video_width < 32) || (video_height < 32))
		return false;
	if(video_source_infos->sws_autoxcrop_framecounter > 0){
		video_source_infos->sws_autoxcrop_framecounter--;
		return false;
	}

	switch(data_elemsize)
	{
		case 1: limit_mask = DISPQT_VIDEO_AUTOCROP_LIMITMASK_YUV42X;
#if DISPQT_VIDEO_AUTOCROP_FASTMODE
				check_step = 4;
#endif
				break;
		case 2: limit_mask = DISPQT_VIDEO_AUTOCROP_LIMITMASK_YUV42X10;
#if DISPQT_VIDEO_AUTOCROP_FASTMODE
				check_step = 2;
#endif
				break;
		case 8: limit_mask = DISPQT_VIDEO_AUTOCROP_LIMITMASK_RGB64; break;
		default: limit_mask = (data_format == AV_PIX_FMT_YUYV422)? DISPQT_VIDEO_AUTOCROP_LIMITMASK_YUYV422 : DISPQT_VIDEO_AUTOCROP_LIMITMASK_RGB32; break;
	}

	skip_border_x = DISPQT_CONFIG_VIDEO_AUTOCROP_SKIP_BORDER - (check_step / 2);

	// check top side
	x = skip_border_x;
	y = skip_border_y;
	data_ptr = data_begin + (y * data_linesize) + (x * data_elemsize);
	check_len = video_height / check_range;
	do{
		if(PDS_GETB_LEU32(data_ptr) & limit_mask){
			//mpxplay_debugf(DISPQT_DEBUGOUT_CROP, "auto1 data:%8.8X m:%8.8X x:%d y:%d", *data32, limit_mask, x,y);
			break;
		}
		data_ptr += data_elemsize;
		x += check_step;
		if(x >= (video_width - skip_border_x)){
			x = skip_border_x; y++;
			data_ptr = data_begin + (y * data_linesize) + (x * data_elemsize);
		}
	}while(y < check_len);

	if(y >= check_len)  // probably full (or partially) black picture
		goto err_out_autocrop_black;
	if(y > skip_border_y)
		crop_y = y;

	// check bottom side
	x = skip_border_x;
	y = skip_border_y;
	data_ptr = data_begin + ((video_height - y) * data_linesize) + (x * data_elemsize);
	check_len = video_height / check_range;
	do{
		if(PDS_GETB_LEU32(data_ptr) & limit_mask){
			//mpxplay_debugf(DISPQT_DEBUGOUT_CROP, "auto1 data:%8.8X m:%8.8X c:%d", *data32, limit_mask, counter);
			break;
		}
		data_ptr += data_elemsize;
		x += check_step;
		if(x >= (video_width - skip_border_x)){
			x = skip_border_x; y++;
			data_ptr = data_begin + ((video_height - y) * data_linesize) + (x * data_elemsize);
		}
	}while(y < check_len);

	if(y >= check_len)  // partially black picture
		goto err_out_autocrop_black;

	if(video_source_infos->ffmpeg_swsframe_cropreset){ // top and bottom side crop shall be the same initially
		if(y > crop_y)
			y = crop_y;
		else if(crop_y > y)
			crop_y = y;
	}

	crop_height = video_height - crop_y;
	if((y > skip_border_y) && (y < check_len))
		crop_height -= y;

	//check left side
	x = skip_border_x;
	y = skip_border_y;
	data_ptr = data_begin + (y * data_linesize) + (x * data_elemsize);
	check_len = video_width / check_range;
	do{
		if(PDS_GETB_LEU32(data_ptr) & limit_mask){
			//mpxplay_debugf(DISPQT_DEBUGOUT_CROP, "auto3 data:%8.8X m:%8.8X x:%d", *data32, limit_mask, x);
			break;
		}
		data_ptr += data_linesize;
		if(++y >= (video_height - skip_border_y)){
			x += check_step; y = skip_border_y;
			data_ptr = data_begin + (y * data_linesize) + (x * data_elemsize);
		}
	}while(x < check_len);

	if(x >= check_len) // partially black picture
		goto err_out_autocrop_black;
	if((x > skip_border_x) && (x < check_len))
		crop_x = x - 1;

	//check right side
	x = skip_border_x;
	y = skip_border_y;
	data_ptr = data_begin + (y * data_linesize) + ((video_width - x) * data_elemsize);
	check_len = video_width / check_range;
	do{
		if(PDS_GETB_LEU32(data_ptr) & limit_mask)
			break;
		data_ptr += data_linesize;
		if(++y >= (video_height - skip_border_y)){
			x += check_step; y = skip_border_y;
			data_ptr = data_begin + (y * data_linesize) + ((video_width - x) * data_elemsize);
		}
	}while(x < check_len);

	if(x >= check_len) // partially black picture
		goto err_out_autocrop_black;

	if(video_source_infos->ffmpeg_swsframe_cropreset){ // left and right side crop shall be the same initially
		if(x > crop_x)
			x = crop_x;
		else if(crop_x > x)
			crop_x = x;
	}

	crop_width = video_width - crop_x;
	if((x > skip_border_x) && (x < check_len))
		crop_width -= x - 1;

	if(video_source_infos->ffmpeg_swsframe_cropreset){
		video_source_infos->sws_srccrop_x = crop_x;
		video_source_infos->sws_srccrop_y = crop_y;
		video_source_infos->sws_srccrop_width = crop_width;
		video_source_infos->sws_srccrop_height = crop_height;
		video_source_infos->ffmpeg_swsframe_cropreset --;
		*videooutsurface_clearpic = true;
		mpxplay_debugf(DISPQT_DEBUGOUT_ACROP, "AUTOCROP1 c:%d x:%d y:%d w:%d h:%d sw:%d sh:%d",
			video_source_infos->ffmpeg_swsframe_cropreset, crop_x, crop_y, crop_width, crop_height, video_width, video_height);
		retval = true;
	}else{
		const int diff_change = DISPQT_CONFIG_VIDEO_AUTOCROP_SENSITIVITY;
		const int change_step_x = DISPQT_CONFIG_VIDEO_AUTOCROP_STEP, change_step_y = DISPQT_CONFIG_VIDEO_AUTOCROP_STEP;
		int change_step_w, change_step_h;

		change_step_w = change_step_x * 2;
		change_step_h = change_step_y * 2;

		if((crop_x + diff_change ) < video_source_infos->sws_srccrop_x){
			video_source_infos->sws_srccrop_x -= change_step_x;  retval = true;
		}else if(crop_x > (video_source_infos->sws_srccrop_x + diff_change)){
			video_source_infos->sws_srccrop_x += change_step_x;  retval = true;
		}else{
			change_step_w = change_step_x;
		}
		if((crop_y + diff_change ) < video_source_infos->sws_srccrop_y){
			video_source_infos->sws_srccrop_y -= change_step_y;  retval = true;
		}else if(crop_y > (video_source_infos->sws_srccrop_y + diff_change)){
			video_source_infos->sws_srccrop_y += change_step_y;  retval = true;
		}else{
			change_step_h = change_step_y;
		}
		if((crop_width + diff_change ) < video_source_infos->sws_srccrop_width){
			video_source_infos->sws_srccrop_width -= change_step_w;  retval = true;
		}else if(crop_width > (video_source_infos->sws_srccrop_width + diff_change)){
			video_source_infos->sws_srccrop_width += change_step_w;  retval = true;
		}
		if((crop_height + diff_change ) < video_source_infos->sws_srccrop_height){
			video_source_infos->sws_srccrop_height -= change_step_h;  retval = true;
		}else if(crop_height > (video_source_infos->sws_srccrop_height + diff_change)){
			video_source_infos->sws_srccrop_height += change_step_h;  retval = true;
		}
		if(video_source_infos->sws_srccrop_width > video_width)
			video_source_infos->sws_srccrop_width = video_width;
		if(video_source_infos->sws_srccrop_height > video_height)
			video_source_infos->sws_srccrop_height = video_height;
		if((video_source_infos->sws_srccrop_x + video_source_infos->sws_srccrop_width) > video_width)
			video_source_infos->sws_srccrop_x = video_width - video_source_infos->sws_srccrop_width;
		if((video_source_infos->sws_srccrop_y + video_source_infos->sws_srccrop_height) > video_height)
			video_source_infos->sws_srccrop_y = video_height - video_source_infos->sws_srccrop_height;

		mpxplay_debugf(DISPQT_DEBUGOUT_ACROP, "AUTOCROP2 r:%d CR:%d x:%3d y:%2d w:%4d h:%3d iw:%d ih:%d cx:%3d cy:%2d cw:%4d ch:%3d stx:%d sty:%d",
				(int)retval, video_source_infos->ffmpeg_swsframe_cropreset, crop_x, crop_y, crop_width, crop_height, video_width, video_height,
				video_source_infos->sws_srccrop_x, video_source_infos->sws_srccrop_y, video_source_infos->sws_srccrop_width, video_source_infos->sws_srccrop_height,
				change_step_x, change_step_y);
		video_source_infos->sws_autoxcrop_framecounter = (retval)? DISPQT_CONFIG_VIDEO_AUTOCROP_SHORTDELAY : DISPQT_CONFIG_VIDEO_AUTOCROP_LONGDELAY;
	}

	return retval;

err_out_autocrop_black:
	video_source_infos->sws_autoxcrop_framecounter = DISPQT_CONFIG_VIDEO_AUTOCROP_SKIPDELAY;
	return retval;
}

void mpxplay_dispqt_videotrans_autocrop_reset(struct dispqt_video_source_info_s *video_source_infos)
{
	if(!video_source_infos->ffmpeg_swsframe_cropreset)
		video_source_infos->ffmpeg_swsframe_cropreset = DISPQT_VIDEO_AUTOCROP_COUNTER_REFRESH;
	video_source_infos->sws_autoxcrop_framecounter = 0;
}

void mpxplay_dispqt_videotrans_autocrop_clear(struct dispqt_video_source_info_s *video_source_infos)
{
	video_source_infos->ffmpeg_swsframe_cropreset = DISPQT_VIDEO_AUTOCROP_COUNTER_INIT;
	video_source_infos->sws_autoxcrop_framecounter = 0;
}

//-------------------------------------------------------------------------------------------------------------------------------------

static int videotrans_ffmpeg_color_range_correction(int pix_fmt, int *pixfmt_corr)
{
	int color_range = AVCOL_RANGE_JPEG;
	int pixfmt_new = pix_fmt;
	switch (pix_fmt)
	{
		case AV_PIX_FMT_YUVJ420P: pixfmt_new = AV_PIX_FMT_YUV420P; break;
		case AV_PIX_FMT_YUVJ422P: pixfmt_new = AV_PIX_FMT_YUV422P; break;
		case AV_PIX_FMT_YUVJ444P: pixfmt_new = AV_PIX_FMT_YUV444P; break;
		case AV_PIX_FMT_GRAY8:
		case AV_PIX_FMT_GRAY16LE:
		case AV_PIX_FMT_RGB555LE:
		case AV_PIX_FMT_BGR24:
		case AV_PIX_FMT_BGRA:
		case AV_PIX_FMT_BGR48LE:
		case AV_PIX_FMT_RGB48BE:
		case AV_PIX_FMT_BGRA64LE:
		case AV_PIX_FMT_RGBA64BE:
			break;
		default: color_range = AVCOL_RANGE_MPEG; break;
	}
	if(pixfmt_corr)
		*pixfmt_corr = pixfmt_new;
	return color_range;
}

int mpxplay_dispqt_videotrans_ffmpeg_swsctx_init(struct dispqt_video_filter_info_s *video_filter_infos, AVFrame *input_video_frame, struct dispqt_video_filter_format_s *output_format)
{
	int src_range, src_pix_fmt, dst_range, dst_pix_fmt, retcode;
	int output_flags = SWS_POINT;// | SWS_FULL_CHR_H_INP | SWS_ACCURATE_RND;
	const int *coefficients_input = NULL, *coefficients_output = NULL;
	enum AVColorSpace color_space_input;
	struct SwsContext *swsctx = NULL;
	AVFrame *swsframe = NULL;

	mpxplay_dispqt_videotrans_ffmpeg_swsctx_close(video_filter_infos);

	if(!input_video_frame || (input_video_frame->width <= 0) || (input_video_frame->height <= 0) || (input_video_frame->format <= AV_PIX_FMT_NONE)
	  || (output_format->width <= 0) || (output_format->height <= 0) || ((int)output_format->pixelformat <= AV_PIX_FMT_NONE)
	){
		return -1;
	}

	//mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "videowidget_ffmpeg_swsctx_init BEGIN (close)");

	src_pix_fmt = (enum AVPixelFormat)input_video_frame->format;
	src_range = (int)input_video_frame->color_range;
	color_space_input = input_video_frame->colorspace;

	if(src_range <= AVCOL_RANGE_UNSPECIFIED)
		src_range = videotrans_ffmpeg_color_range_correction(src_pix_fmt, &src_pix_fmt);

	dst_range = videotrans_ffmpeg_color_range_correction(output_format->pixelformat, &dst_pix_fmt);
	/* if context is YUV full range */
	if((dst_range < src_range) && (videotrans_ffmpeg_color_range_correction(src_pix_fmt, NULL) == videotrans_ffmpeg_color_range_correction(output_format->pixelformat, NULL)))
		dst_range = src_range;

	/* SWS_FULL_CHR_H_INT is correctly supported only for RGB formats */
	//if ((dst_pix_fmt == AV_PIX_FMT_BGR24) || (dst_pix_fmt == AV_PIX_FMT_BGRA))
	//    output_flags |= SWS_FULL_CHR_H_INT;

	swsctx = sws_alloc_context();
	if (!swsctx)
		goto err_out_init;

	av_opt_set_int(swsctx, "sws_flags",  output_flags, 0);

	av_opt_set_int(swsctx, "srcw",       input_video_frame->width,   0);
	av_opt_set_int(swsctx, "srch",       input_video_frame->height,  0);
	av_opt_set_int(swsctx, "src_format", src_pix_fmt, 0);
	av_opt_set_int(swsctx, "src_range",  ((src_range == AVCOL_RANGE_JPEG)? 1 : 0),   0);

	av_opt_set_int(swsctx, "dstw",       output_format->width,   0);
	av_opt_set_int(swsctx, "dsth",       output_format->height,  0);
	av_opt_set_int(swsctx, "dst_format", dst_pix_fmt, 0);
	av_opt_set_int(swsctx, "dst_range",  ((dst_range == AVCOL_RANGE_JPEG)? 1 : 0),   0);

	coefficients_input = sws_getCoefficients(color_space_input);
	coefficients_output = sws_getCoefficients(AVCOL_SPC_UNSPECIFIED);

	sws_setColorspaceDetails(swsctx,
							 coefficients_input, ((src_range == AVCOL_RANGE_JPEG)? 1 : 0),
							 coefficients_output, ((dst_range == AVCOL_RANGE_JPEG)? 1 : 0),
							 0, 1<<16, 1<<16);

	retcode = sws_init_context(swsctx, NULL, NULL);
	if(retcode < 0)
		goto err_out_init;

	swsframe = av_frame_alloc();
	if(!swsframe)
		goto err_out_init;
	swsframe->width  = output_format->width;
	swsframe->height = output_format->height;
	swsframe->format = dst_pix_fmt;

	if(av_image_alloc(&swsframe->data[0], &swsframe->linesize[0], output_format->width, output_format->height, output_format->pixelformat, DISPQT_VIDEO_PROCESS_BUFFERALIGNMENT) <= 0)
		goto err_out_init;

	video_filter_infos->input_format_cfg.width = input_video_frame->width;
	video_filter_infos->input_format_cfg.height = input_video_frame->height;
	video_filter_infos->input_format_cfg.pixelformat = (AVPixelFormat)input_video_frame->format;
	video_filter_infos->output_format_cfg.width = output_format->width;
	video_filter_infos->output_format_cfg.height = output_format->height;
	video_filter_infos->output_format_cfg.pixelformat = output_format->pixelformat;

	video_filter_infos->sws_ctx = swsctx;
	video_filter_infos->ffmpeg_sws_frame = swsframe;

	return MPXPLAY_ERROR_OK;

err_out_init:
	if(swsctx)
		sws_freeContext(swsctx);
	if(swsframe)
		av_frame_free(&swsframe);
	return -1;
}

void mpxplay_dispqt_videotrans_ffmpeg_swsctx_clean(struct dispqt_video_filter_info_s *video_filter_infos)
{
	video_filter_infos->sws_ctx = NULL;
	video_filter_infos->ffmpeg_sws_frame = NULL;
	video_filter_infos->ffmpeg_swsframe_hasdata = false;
	video_filter_infos->input_format_cfg.width = 0;
	video_filter_infos->input_format_cfg.height = 0;
	video_filter_infos->input_format_cfg.pixelformat = AV_PIX_FMT_NONE;
	video_filter_infos->output_format_cfg.width = 0;
	video_filter_infos->output_format_cfg.height = 0;
	video_filter_infos->output_format_cfg.pixelformat = AV_PIX_FMT_NONE;
}

void mpxplay_dispqt_videotrans_ffmpeg_swsctx_close(struct dispqt_video_filter_info_s *video_filter_infos)
{
	struct SwsContext *sws_ctx = video_filter_infos->sws_ctx;
	AVFrame *av_frame = video_filter_infos->ffmpeg_sws_frame;
	video_filter_infos->sws_ctx = NULL;
	video_filter_infos->ffmpeg_sws_frame = NULL;
	if(av_frame)
	{
		av_freep(&av_frame->data[0]);
		av_frame_free(&av_frame);
	}
	if(sws_ctx)
	{
		sws_freeContext(sws_ctx);
	}
	mpxplay_dispqt_videotrans_ffmpeg_swsctx_clean(video_filter_infos);
}

bool mpxplay_dispqt_videotrans_ffmpeg_swsctx_conv_frame(struct dispqt_video_filter_info_s *video_filter_infos, AVFrame *ivf, struct dispqt_video_filter_format_s *output_format)
{
	if((output_format->width <= 0) || (output_format->height <= 0) || (output_format->pixelformat <= AV_PIX_FMT_NONE))
	{
		if(video_filter_infos->sws_ctx)
		{
			mpxplay_dispqt_videotrans_ffmpeg_swsctx_close(video_filter_infos);
		}
		return false;
	}

	if(!ivf || (ivf->width <= 0) || (ivf->height <= 0) || (ivf->format <= AV_PIX_FMT_NONE))
	{
		return false;
	}

	if(!video_filter_infos->sws_ctx || !video_filter_infos->ffmpeg_sws_frame
		|| (ivf->width != video_filter_infos->input_format_cfg.width) || (ivf->height != video_filter_infos->input_format_cfg.height) || (ivf->format != video_filter_infos->input_format_cfg.pixelformat)
		|| (video_filter_infos->output_format_cfg.width != output_format->width) || (video_filter_infos->output_format_cfg.height != output_format->height) || (video_filter_infos->output_format_cfg.pixelformat != output_format->pixelformat)
	){
		if(mpxplay_dispqt_videotrans_ffmpeg_swsctx_init(video_filter_infos, ivf, output_format) != MPXPLAY_ERROR_OK)
		{
			return false;
		}
	}

	if(video_filter_infos->sws_ctx && video_filter_infos->ffmpeg_sws_frame)
	{
		if(sws_scale(video_filter_infos->sws_ctx, (uint8_t const * const *)&ivf->data[0], ivf->linesize, 0, ivf->height,
					&video_filter_infos->ffmpeg_sws_frame->data[0], &video_filter_infos->ffmpeg_sws_frame->linesize[0]) == video_filter_infos->ffmpeg_sws_frame->height
		){
			if(video_filter_infos->ffmpeg_sws_frame->data[0] && (video_filter_infos->ffmpeg_sws_frame->linesize[0] > 0))
			{
				av_frame_copy_props(video_filter_infos->ffmpeg_sws_frame, ivf); // to keep rotate info
				return true;
			}
		}
	}
	return false;
}

bool mpxplay_dispqt_videotrans_ffmpeg_swsctx_pixfmt_conv(struct dispqt_video_filter_info_s *video_filter_infos, AVFrame *input_avframe, AVPixelFormat out_pixfmt)
{
	struct dispqt_video_filter_format_s output_format;
	output_format.width = input_avframe->width;
	output_format.height = input_avframe->height;
	output_format.pixelformat = out_pixfmt;

	return mpxplay_dispqt_videotrans_ffmpeg_swsctx_conv_frame(video_filter_infos, input_avframe, &output_format);
}

//===============================================================================================================================================================

int FFMpegVideoSoftwareTransform::videowidget_ffmpeg_swsctx_get_sws_outformat(struct dispqt_video_filter_info_s *video_filter_infos, AVFrame *input_video_frame)
{
	struct mmc_dispqt_config_s *gcfg = this->main_window->gui_config;
	DispQtVideoPlayer *qtvp = this->main_window->qt_video_player;
	enum AVPixelFormat sws_out_pixelformat = AV_PIX_FMT_NONE;

	if(gcfg->video_crop_type == DISPQT_VIDEOSCRCROPTYPE_AUTO)
	{
		int pixform = input_video_frame->format;
		if( (pixform != DISPQT_VIDEO_AUTOCROP_AVFORMAT) && (pixform != this->video_surface_infos->video_render_infos->hwdevice_avpixfmt)
		 && (input_video_frame->height <= gcfg->video_autocrop_limit)
		 && !this->video_surface_infos->video_render_infos->render_function_poolbufelem_validity_check(input_video_frame)
		 && ( (this->video_surface_infos->videoout_surface_pix_fmt == DISPQT_VIDEO_AUTOCROP_AVFORMAT)
		   || (pixform == AV_PIX_FMT_BGR565LE) || (pixform == AV_PIX_FMT_BGR555LE)) // TODO: more pixfmts, unsupported by autocrop
		){
			sws_out_pixelformat = DISPQT_VIDEO_AUTOCROP_AVFORMAT;
		}
	}
	else if((this->video_surface_infos->video_render_infos->renderer_capabilities & (DISPQT_VIDEORENDERER_CAPFLAG_COLORMIXER | DISPQT_VIDEORENDERER_CAPFLAG_ROTATE))
		||	((qtvp->video_frame_filter_eq_process->video_filter_check(this->video_source_infos->ffmpeg_decoded_frame, AV_PIX_FMT_NONE) == AV_PIX_FMT_NONE)
	     && (qtvp->video_frame_filter_transform_process->video_filter_check(this->video_source_infos->ffmpeg_decoded_frame, AV_PIX_FMT_NONE) == AV_PIX_FMT_NONE))
	){
		if(input_video_frame->format != this->video_surface_infos->videoout_surface_pix_fmt)
		{
			sws_out_pixelformat = this->video_surface_infos->videoout_surface_pix_fmt;
		}
	}

	return sws_out_pixelformat; // returns AV_PIX_FMT_NONE if swsctx conversion is not needed
}

bool FFMpegVideoSoftwareTransform::videowidget_ffmpeg_swsctx_conv_frame(struct dispqt_video_filter_info_s *video_filter_infos)
{
	AVFrame *ivf = (video_filter_infos->ffmpeg_transformed_frame)? video_filter_infos->ffmpeg_transformed_frame : this->video_source_infos->ffmpeg_decoded_frame;
	AVPixelFormat output_pixfmt = (AVPixelFormat)videowidget_ffmpeg_swsctx_get_sws_outformat(video_filter_infos, ivf);

	return mpxplay_dispqt_videotrans_ffmpeg_swsctx_pixfmt_conv(video_filter_infos, ivf, output_pixfmt);
}

bool FFMpegVideoSoftwareTransform::videowidget_ffmpeg_swsctx_check_swsframe_for_outsurface(struct dispqt_video_filter_info_s *video_filter_infos)
{   // returns true if swsctx has a valid output frame and its pixelformat matches with the video output surface (sws frame is used if all filters are disabled)
	return (video_filter_infos->ffmpeg_swsframe_hasdata && video_filter_infos->ffmpeg_sws_frame && (video_filter_infos->ffmpeg_sws_frame->format == this->video_surface_infos->videoout_surface_pix_fmt))? true : false;
}

void FFMpegVideoSoftwareTransform::videowidget_ffmpeg_swsctx_close(void)
{
	struct dispqt_video_filter_info_s *video_filter_infos = &this->video_filter_infos;
	mpxplay_dispqt_videotrans_ffmpeg_swsctx_close(video_filter_infos);
	this->videowidget_ffmpeg_swsctx_clear(video_filter_infos);
}

void FFMpegVideoSoftwareTransform::videowidget_ffmpeg_swsctx_clear(struct dispqt_video_filter_info_s *video_filter_infos)
{
	mpxplay_dispqt_videotrans_ffmpeg_swsctx_clean(video_filter_infos);
	this->video_surface_infos->videoout_surface_clearpic = true;
	mpxplay_dispqt_videotrans_autocrop_clear(this->video_source_infos);
}

//===============================================================================================================================================================
#define DISPQT_VIDEO_AR_SCALE 1000
#define DISPQT_VIDEO_ROUND_VALUE 1
#define DISPQT_VIDEO_ROUND_MASK (~1)

int mpxplay_dispqt_videotrans_videooutput_get_custom_ar_value(char *custom_str, int scale_value, int default_value)
{
	int ar_value = default_value;
	char *ar_separator = pds_strchr(custom_str, DISPQT_CONFIG_VIDEOAR_DIVISORCHAR);
	if(ar_separator){
		int ar_custom_x = pds_atol(custom_str);
		int ar_custom_y = pds_atol(ar_separator + 1);
		if((ar_custom_x > 0) && (ar_custom_y > 0)){
			int ar = scale_value * ar_custom_x / ar_custom_y;
			if((ar >= (scale_value / 100)) && (ar <= (scale_value * 100))) // FIXME: / 100
				ar_value = ar;
		}
	}
	return ar_value;
}

unsigned long mpxplay_dispqt_videotrans_rotate_get_type(struct mmc_dispqt_config_s *gcfg, AVFrame *video_avframe)
{
	int rot_type = gcfg->video_rotation_type;
	if(rot_type == DISPQT_VIDEO_ROTATIONTYPE_AUTO && video_avframe && video_avframe->metadata)
	{
		AVDictionaryEntry *t = av_dict_get(video_avframe->metadata, "Orientation", NULL, 0);
		if(t && t->value)
		{
			int exif_orientation_num = atoi(t->value);
			switch(exif_orientation_num) // by https://www.exif.org/Exif2-2.PDF -> Page 18, Orientation (alt: https://www.sno.phy.queensu.ca/~phil/exiftool/TagNames/EXIF.html)
			{
				//case 1: // no transformation is required
				//case 2: // flip horizontally
				case 3: // rotate by 180
				case 4: rot_type = DISPQT_VIDEO_ROTATIONTYPE_ROT180; break; // flip vertically
				case 5: // flip horizontally and rotate left 90
				case 6: rot_type = DISPQT_VIDEO_ROTATIONTYPE_ROTR90; break;
				case 7: // flip horizontally and rotate right 90
				case 8: rot_type = DISPQT_VIDEO_ROTATIONTYPE_ROTL90; break;
			}
		}
	}
	return rot_type;
}

bool mpxplay_dispqt_videotrans_videoframe_hw_rotate_check(struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *video_avframe)
{
	bool hw_transform = false;
	if(video_surface_infos->video_render_infos->renderer_capabilities & DISPQT_VIDEORENDERER_CAPFLAGS_TRANSFORM)
	{
		unsigned long rot_type = mpxplay_dispqt_videotrans_rotate_get_type(video_surface_infos->gui_config, video_avframe);
		if(video_surface_infos->rotation_type != rot_type)
		{
			video_surface_infos->rotation_type = rot_type;
			video_surface_infos->videoout_surface_recalc = true;
		}
		hw_transform = true;
	}
	return hw_transform;
}

//-------------------------------------------------------------------------------------------------------------------------------------------
bool mpxplay_dispqt_videotrans_videooutput_surface_move_calculate(
		struct dispqt_video_source_info_s *video_source_infos,
		struct dispqt_video_surface_info_s *video_surface_infos,
		struct mmc_dispqt_config_s *gcfg,
		int widget_global_pos_x, int widget_global_pos_y,
		int move_x, int move_y
){
	int max_crop_x, max_crop_y, prev_crop_x, prev_crop_y;
	bool do_repaint = false;

	if(video_surface_infos->video_render_infos->renderer_capabilities & DISPQT_VIDEORENDERER_CAPFLAG_ROTATE) // video frame is rotated by the renderer
	{
		switch(video_surface_infos->rotation_type)
		{   // workaround to shift surface into the right direction
			case DISPQT_VIDEO_ROTATIONTYPE_ROT180:
			case DISPQT_VIDEO_ROTATIONTYPE_ROTL90: move_y = -move_y; break;
			case DISPQT_VIDEO_ROTATIONTYPE_ROTR90: move_x = -move_x; break;
		}
	}

	prev_crop_x = video_surface_infos->sws_crop_x;
	prev_crop_y = video_surface_infos->sws_crop_y;

	move_x = move_x * DISPQT_CONFIG_VIDEOZOOM_SCALE / gcfg->video_zoom_value;  // scale moving to zoom rate
	max_crop_x = video_source_infos->sws_srccrop_width - video_surface_infos->sws_crop_w;
	if(max_crop_x > 0)
	{
		video_surface_infos->sws_crop_x += move_x;
		if(video_surface_infos->sws_crop_x < video_source_infos->sws_srccrop_x)
			video_surface_infos->sws_crop_x = video_source_infos->sws_srccrop_x;
		else if(video_surface_infos->sws_crop_x > (video_source_infos->sws_srccrop_x + max_crop_x))
			video_surface_infos->sws_crop_x = video_source_infos->sws_srccrop_x + max_crop_x;
	}

	move_y = move_y * DISPQT_CONFIG_VIDEOZOOM_SCALE / gcfg->video_zoom_value;
	max_crop_y = video_source_infos->sws_srccrop_height - video_surface_infos->sws_crop_h;
	if(max_crop_y > 0)
	{
		video_surface_infos->sws_crop_y += move_y;
		if(video_surface_infos->sws_crop_y < video_source_infos->sws_srccrop_y)
			video_surface_infos->sws_crop_y = video_source_infos->sws_srccrop_y;
		else if(video_surface_infos->sws_crop_y > (video_source_infos->sws_srccrop_y + max_crop_y))
			video_surface_infos->sws_crop_y = video_source_infos->sws_srccrop_y + max_crop_y;
	}

	// reconfig zoom center after move
	video_surface_infos->video_zoom_globalpos_x = video_surface_infos->sws_dst_width / 2;
	video_surface_infos->video_zoom_globalpos_y = video_surface_infos->sws_dst_height / 2;
	if(!video_surface_infos->is_fullscreen){
		video_surface_infos->video_zoom_globalpos_x += widget_global_pos_x;
		video_surface_infos->video_zoom_globalpos_y += widget_global_pos_y;
	}

	if((prev_crop_x != video_surface_infos->sws_crop_x) || (prev_crop_y = video_surface_infos->sws_crop_y)){
		mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "MOVE cx:%d cy:%d", video_surface_infos->sws_crop_x, video_surface_infos->sws_crop_y);
		do_repaint = true;
	}
	return do_repaint;
}

bool mpxplay_dispqt_videotrans_videooutput_surface_zoom_calculate(
		struct dispqt_video_source_info_s *video_source_infos,
		struct dispqt_video_surface_info_s *video_surface_infos,
		struct mmc_dispqt_config_s *gcfg, int globalpos_x, int globalpos_y, int direction
){
	int i, zoom_val, center_values[2];
	bool do_update = false;

	if((video_surface_infos->last_video_zoom_min_val <= 0) || (video_surface_infos->last_video_zoom_ti_val <= 0) || (video_source_infos->sws_srccrop_width <= 0) || (video_source_infos->sws_srccrop_height <= 0) || (gcfg->video_zoom_value <= 0))
		return do_update;

	zoom_val = gcfg->video_zoom_value + direction * DISPQT_CONFIG_VIDEOZOOM_STEP * gcfg->video_zoom_value / DISPQT_CONFIG_VIDEOZOOM_SCALE;

	center_values[0] = DISPQT_CONFIG_VIDEOZOOM_CENTER;
	center_values[1] = video_surface_infos->last_video_zoom_ti_val;
	mpxplay_debugf(DISPQT_DEBUGOUT_CROP, "swctx_zoom cv:%d zv:%d mv:%d", gcfg->video_zoom_value, zoom_val, video_surface_infos->last_video_zoom_min_val);

	if(zoom_val < video_surface_infos->last_video_zoom_min_val)
		zoom_val = video_surface_infos->last_video_zoom_min_val;
	else if(zoom_val > DISPQT_CONFIG_VIDEOZOOM_MAX)
		zoom_val = DISPQT_CONFIG_VIDEOZOOM_MAX;

	for(i = 0; i < (sizeof(center_values) / sizeof(center_values[0])); i++)
	{
		const int center = center_values[i];
		//mpxplay_debugf(DISPQT_DEBUGOUT_CROP, "swctx_zoom i:%d c:%d", i, center);
		if(((gcfg->video_zoom_value > center) && (zoom_val <= (center + center / 10))) || ((gcfg->video_zoom_value < center) && (zoom_val >= (center - center / 10)))){
			zoom_val = center;
			mpxplay_debugf(DISPQT_DEBUGOUT_CROP, "swctx_zoom i:%d c:%d", i, center);
			break;
		}
	}

	mpxplay_debugf(DISPQT_DEBUGOUT_CROP, "swctx_zoom zt:%d zv:%d mv:%d", gcfg->video_zoom_type, zoom_val, video_surface_infos->last_video_zoom_min_val);

	if(gcfg->video_zoom_value != zoom_val)
	{
		gcfg->video_zoom_type = DISPQT_VIDEOFULLSCREENZOOMTYPE_MANUAL;
		gcfg->video_zoom_value = zoom_val;
		if(globalpos_x >= 0)
			video_surface_infos->video_zoom_globalpos_x = globalpos_x;
		if(globalpos_y >= 0)
			video_surface_infos->video_zoom_globalpos_y = globalpos_y;
		video_surface_infos->videoout_surface_recalc = true;
		do_update = true;
	}
	mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "videowidget_ffmpeg_swctx_zoom END 1");
	return do_update;
}

bool mpxplay_dispqt_videotrans_videooutput_surface_window_calculate(
	struct dispqt_video_source_info_s *video_source_infos,
	struct dispqt_video_surface_info_s *video_surface_infos,
	struct mmc_dispqt_config_s *gcfg
){
	int outframe_width, outframe_height, video_aspect_ration_num, video_aspect_ratio_den, video_ar, video_crop_ar = 0, window_ar;
	int widget_global_pos_x, widget_global_pos_y, window_width, window_height;
	unsigned long ar_type;

	if(!video_surface_infos->ffmpeg_output_frame || (video_surface_infos->ffmpeg_output_frame->width <= 0) || (video_surface_infos->ffmpeg_output_frame->height <= 0))
		return false;

	if(video_surface_infos->is_fullscreen)
	{
		widget_global_pos_x = 0;
		widget_global_pos_y = 0;
		window_width = video_surface_infos->fullscreen_size_x;
		window_height = video_surface_infos->fullscreen_size_y;
	}
	else
	{
		widget_global_pos_x = video_surface_infos->window_pos_x;
		widget_global_pos_y = video_surface_infos->window_pos_y;
		window_width = video_surface_infos->windowed_size_x;
		window_height = video_surface_infos->windowed_size_y;
	}

	video_surface_infos->window_size_x = window_width;
	video_surface_infos->window_size_y = window_height;
	video_surface_infos->sws_dst_width = window_width;
	video_surface_infos->sws_dst_height = window_height;

	if((video_surface_infos->sws_dst_width <= 0) || (video_surface_infos->sws_dst_height <= 0))
		return false;

	outframe_width = video_surface_infos->ffmpeg_output_frame->width;
	outframe_height = video_surface_infos->ffmpeg_output_frame->height;
	video_aspect_ration_num = video_source_infos->video_aspect_ratio.num;
	video_aspect_ratio_den = video_source_infos->video_aspect_ratio.den;
	if(video_surface_infos->video_render_infos->renderer_capabilities & DISPQT_VIDEORENDERER_CAPFLAG_ROTATE) // video frame is rotated by the renderer
	{
		if((video_surface_infos->rotation_type == DISPQT_VIDEO_ROTATIONTYPE_ROTL90) || (video_surface_infos->rotation_type == DISPQT_VIDEO_ROTATIONTYPE_ROTR90))
		{   // rotate input video frame rectangles to get correct output crop and viewport values
			outframe_width = video_surface_infos->ffmpeg_output_frame->height;
			outframe_height = video_surface_infos->ffmpeg_output_frame->width;
			video_aspect_ration_num = video_source_infos->video_aspect_ratio.den;
			video_aspect_ratio_den = video_source_infos->video_aspect_ratio.num;
		}
	}

	video_ar = DISPQT_VIDEO_AR_SCALE;
	video_ar *= outframe_width * ((video_aspect_ration_num > 0)? video_aspect_ration_num : 1);
	video_ar /= outframe_height * ((video_aspect_ratio_den > 0)? video_aspect_ratio_den : 1);

	// --- crop ratio calculate (manual, auto) -------------------------------------------------------------------------------------

	if( (gcfg->video_crop_type != DISPQT_VIDEOSCRCROPTYPE_AUTO) || (outframe_height > gcfg->video_autocrop_limit)
	 || !video_source_infos->sws_srccrop_width || !video_source_infos->sws_srccrop_height
	 || (video_source_infos->sws_srccrop_width > outframe_width) || (video_source_infos->sws_srccrop_height > outframe_height)
	){
		video_source_infos->sws_srccrop_x = video_source_infos->sws_srccrop_y = 0;
		video_source_infos->sws_srccrop_width = outframe_width;
		video_source_infos->sws_srccrop_height = outframe_height;
	}

	switch(gcfg->video_crop_type)
	{
		case DISPQT_VIDEOSCRCROPTYPE_AUTO:  video_ar = video_ar * video_source_infos->sws_srccrop_width / outframe_width * outframe_height / video_source_infos->sws_srccrop_height; break;
		case DISPQT_VIDEOSCRCROPTYPE_4_3:   video_crop_ar = DISPQT_VIDEO_AR_SCALE * 4 / 3; break;
		case DISPQT_VIDEOSCRCROPTYPE_16_9:  video_crop_ar = DISPQT_VIDEO_AR_SCALE * 16 / 9; break;
		case DISPQT_VIDEOSCRCROPTYPE_235_1: video_crop_ar = DISPQT_VIDEO_AR_SCALE * 235 / 100; break;
		case DISPQT_VIDEOSCRCROPTYPE_CUSTOM:video_crop_ar = mpxplay_dispqt_videotrans_videooutput_get_custom_ar_value(gcfg->video_crop_custom, DISPQT_VIDEO_AR_SCALE, 0); break;
	}

	if((video_crop_ar > 0) && (video_crop_ar != video_ar))
	{
		if(video_ar > video_crop_ar)
			video_source_infos->sws_srccrop_width = outframe_width * video_crop_ar / video_ar;
		else
			video_source_infos->sws_srccrop_height = outframe_height * video_ar / video_crop_ar;
		video_source_infos->sws_srccrop_x = (outframe_width - video_source_infos->sws_srccrop_width) / 2;
		video_source_infos->sws_srccrop_y = (outframe_height - video_source_infos->sws_srccrop_height) / 2;
		mpxplay_debugf(DISPQT_DEBUGOUT_CROP, "CROP var:%d car:%d sw:%d sh:%d cw:%d ch:%d", video_ar, video_crop_ar, outframe_width, outframe_height, video_source_infos->sws_srccrop_width, video_source_infos->sws_srccrop_height);
		video_ar = video_crop_ar;
	}

	// --- aspect ratio calculate -------------------------------------------------------------------------------------

	if(video_source_infos->ffmpeg_decoded_frame && (video_source_infos->ffmpeg_decoded_frame->flags & DISPQT_AVFRAME_FLAG_AR_WINDOW)) // window (fill) aspect ratio by decoder (eg. visualization output)
	{
		ar_type = DISPQT_VIDEOFULLSCREENARTYPE_FILL;
	}
	else
	{
		ar_type = (video_surface_infos->is_fullscreen || (gcfg->video_windowed_ar_type == DISPQT_VIDEOWINDOWEDARTYPE_EQFULLSCREEN))? gcfg->video_fullscreen_ar_type : gcfg->video_windowed_ar_type;
	}

	window_ar = DISPQT_VIDEO_AR_SCALE * window_width / window_height;

	switch(ar_type){
		case DISPQT_VIDEOFULLSCREENARTYPE_FILL:  video_ar = window_ar; break;
		case DISPQT_VIDEOFULLSCREENARTYPE_1_1:   video_ar = DISPQT_VIDEO_AR_SCALE; break;
		case DISPQT_VIDEOFULLSCREENARTYPE_4_3:   video_ar = DISPQT_VIDEO_AR_SCALE * 4 / 3; break;
		case DISPQT_VIDEOFULLSCREENARTYPE_16_9:  video_ar = DISPQT_VIDEO_AR_SCALE * 16 / 9; break;
		case DISPQT_VIDEOFULLSCREENARTYPE_235_1: video_ar = DISPQT_VIDEO_AR_SCALE * 235 / 100; break;
		case DISPQT_VIDEOFULLSCREENARTYPE_CUSTOM:video_ar = mpxplay_dispqt_videotrans_videooutput_get_custom_ar_value(gcfg->video_ar_custom, DISPQT_VIDEO_AR_SCALE, DISPQT_VIDEO_AR_SCALE); break;
		case DISPQT_VIDEOFULLSCREENARTYPE_PIXEL: video_ar = DISPQT_VIDEO_AR_SCALE * outframe_width / outframe_height; break;
	}

	// --- zoom value calculate -----------------------------------------------------------------------------------------

	const int prev_crop_w = video_surface_infos->sws_crop_w;
	const int prev_crop_h = video_surface_infos->sws_crop_h;
	const int pixel_ar = DISPQT_VIDEO_AR_SCALE * video_source_infos->sws_srccrop_width / video_source_infos->sws_srccrop_height;

	window_width = video_source_infos->sws_srccrop_width;
	window_height = video_source_infos->sws_srccrop_height;

	if(video_ar > pixel_ar)
		window_height = window_height * pixel_ar / video_ar;
	else if((video_ar > 0) && (video_ar < pixel_ar))
		window_width = window_width * video_ar / pixel_ar;

	if((window_width <= 0) || (window_height <= 0))
		return false;

	const int zoom_val_ti_w = (DISPQT_CONFIG_VIDEOZOOM_SCALE * video_surface_infos->sws_dst_width + (window_width >> 1)) / window_width;
	const int zoom_val_ti_h = (DISPQT_CONFIG_VIDEOZOOM_SCALE * video_surface_infos->sws_dst_height + (window_height >> 1)) / window_height;
	const int zoom_val_to = max(zoom_val_ti_w , zoom_val_ti_h);
	const int zoom_val_ti = (ar_type == DISPQT_VIDEOFULLSCREENARTYPE_FILL)? zoom_val_to : min(zoom_val_ti_w , zoom_val_ti_h);
	int zoom_value = DISPQT_CONFIG_VIDEOZOOM_CENTER;

	video_surface_infos->last_video_zoom_ti_val = zoom_val_ti;
//	this->last_video_zoom_min_val = ((ar_type == DISPQT_VIDEOFULLSCREENARTYPE_FILL) || (zoom_val_ti < DISPQT_CONFIG_VIDEOZOOM_MIN))? zoom_val_ti : DISPQT_CONFIG_VIDEOZOOM_MIN; // FIXME: if video resolution is larger than window, zoom is incorrect in fill mode below zoom_val_ti
	video_surface_infos->last_video_zoom_min_val = (zoom_val_ti < DISPQT_CONFIG_VIDEOZOOM_MIN)? zoom_val_ti : DISPQT_CONFIG_VIDEOZOOM_MIN;

	switch(gcfg->video_zoom_type)
	{
		case DISPQT_VIDEOFULLSCREENZOOMTYPE_TOUCHINSIDE: zoom_value = zoom_val_ti; break;
		case DISPQT_VIDEOFULLSCREENZOOMTYPE_TOUCHOUTSIDE: zoom_value = zoom_val_to; break;
		case DISPQT_VIDEOFULLSCREENZOOMTYPE_2_1: zoom_value = 2 * DISPQT_CONFIG_VIDEOZOOM_SCALE; break;
		case DISPQT_VIDEOFULLSCREENZOOMTYPE_4_1: zoom_value = 4 * DISPQT_CONFIG_VIDEOZOOM_SCALE; break;
		case DISPQT_VIDEOFULLSCREENZOOMTYPE_CUSTOM: zoom_value = mpxplay_dispqt_videotrans_videooutput_get_custom_ar_value(gcfg->video_zoom_custom, DISPQT_CONFIG_VIDEOZOOM_SCALE, DISPQT_CONFIG_VIDEOZOOM_SCALE); break;
		case DISPQT_VIDEOFULLSCREENZOOMTYPE_MANUAL: zoom_value = gcfg->video_zoom_value; break;
	}

	if(zoom_value < video_surface_infos->last_video_zoom_min_val)
	{
		zoom_value = video_surface_infos->last_video_zoom_min_val;
	}
	else if(zoom_value > DISPQT_CONFIG_VIDEOZOOM_MAX)
	{
		zoom_value = DISPQT_CONFIG_VIDEOZOOM_MAX;
	}

	if(zoom_value != DISPQT_CONFIG_VIDEOZOOM_CENTER){
		window_width = window_width * zoom_value / DISPQT_CONFIG_VIDEOZOOM_SCALE;
		window_height = window_height * zoom_value / DISPQT_CONFIG_VIDEOZOOM_SCALE;
	}

	video_surface_infos->sws_crop_w = video_source_infos->sws_srccrop_width;
	video_surface_infos->sws_crop_h = video_source_infos->sws_srccrop_height;

	if(window_width > video_surface_infos->sws_dst_width){
		video_surface_infos->sws_crop_w = video_source_infos->sws_srccrop_width * video_surface_infos->sws_dst_width / window_width;
		window_width = video_surface_infos->sws_dst_width;
	}
	if(window_height > video_surface_infos->sws_dst_height){
		video_surface_infos->sws_crop_h = video_source_infos->sws_srccrop_height * video_surface_infos->sws_dst_height / window_height;
		window_height = video_surface_infos->sws_dst_height;
	}

	gcfg->video_zoom_value = zoom_value;

	// --- zoom center (source view window) calculate -----------------------------------------------------------------------------------------

	int max_crop_x = video_source_infos->sws_srccrop_width - video_surface_infos->sws_crop_w;
	int max_crop_y = video_source_infos->sws_srccrop_height - video_surface_infos->sws_crop_h;

	if( (gcfg->video_zoom_type == DISPQT_VIDEOFULLSCREENZOOMTYPE_TOUCHINSIDE) || (gcfg->video_zoom_type == DISPQT_VIDEOFULLSCREENZOOMTYPE_TOUCHOUTSIDE)){
		video_surface_infos->video_zoom_globalpos_x = widget_global_pos_x + video_surface_infos->sws_dst_width / 2;
		video_surface_infos->video_zoom_globalpos_y = widget_global_pos_y + video_surface_infos->sws_dst_height / 2;
		video_surface_infos->sws_crop_x = video_source_infos->sws_srccrop_x + max_crop_x / 2;
		video_surface_infos->sws_crop_y = video_source_infos->sws_srccrop_y + max_crop_y / 2;
	}
	else // 1:1, 2:1, 4:1, custom, manual
	{
		int zoom_widget_pos_x = video_surface_infos->video_zoom_globalpos_x - widget_global_pos_x;

		if((zoom_widget_pos_x >= 0) && (zoom_widget_pos_x < video_surface_infos->sws_dst_width))
		{
			if(prev_crop_w > 0){
				int sws_prev_pos_x = video_surface_infos->sws_crop_x + prev_crop_w * zoom_widget_pos_x / video_surface_infos->sws_dst_width;
				mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "ZOOMX px:%d", sws_prev_pos_x);
				video_surface_infos->sws_crop_x = sws_prev_pos_x - video_surface_infos->sws_crop_w * zoom_widget_pos_x / video_surface_infos->sws_dst_width;
			}else{
				video_surface_infos->sws_crop_x = video_source_infos->sws_srccrop_x + max_crop_x * zoom_widget_pos_x / video_surface_infos->sws_dst_width;
			}
			if(video_surface_infos->sws_crop_x < video_source_infos->sws_srccrop_x)
				video_surface_infos->sws_crop_x = video_source_infos->sws_srccrop_x;
			else if(video_surface_infos->sws_crop_x > (video_source_infos->sws_srccrop_x + max_crop_x))
				video_surface_infos->sws_crop_x = video_source_infos->sws_srccrop_x + max_crop_x;
		}else{
			video_surface_infos->sws_crop_x = video_source_infos->sws_srccrop_x + max_crop_x / 2;
		}

		int zoom_widget_pos_y = video_surface_infos->video_zoom_globalpos_y - widget_global_pos_y;

		if((zoom_widget_pos_y >= 0) && (zoom_widget_pos_y < video_surface_infos->sws_dst_height))
		{
			if(prev_crop_h > 0){
				int sws_prev_pos_y = video_surface_infos->sws_crop_y + prev_crop_h * zoom_widget_pos_y / video_surface_infos->sws_dst_height;
				mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "ZOOMX px:%d", sws_prev_pos_y);
				video_surface_infos->sws_crop_y = sws_prev_pos_y - video_surface_infos->sws_crop_h * zoom_widget_pos_y / video_surface_infos->sws_dst_height;
			}else{
				video_surface_infos->sws_crop_y = video_source_infos->sws_srccrop_y + max_crop_y * zoom_widget_pos_y / video_surface_infos->sws_dst_height;
			}
			if(video_surface_infos->sws_crop_y < video_source_infos->sws_srccrop_y)
				video_surface_infos->sws_crop_y = video_source_infos->sws_srccrop_y;
			else if(video_surface_infos->sws_crop_y > (video_source_infos->sws_srccrop_y + max_crop_y))
				video_surface_infos->sws_crop_y = video_source_infos->sws_srccrop_y + max_crop_y;
		}else{
			video_surface_infos->sws_crop_y = video_source_infos->sws_srccrop_y + max_crop_y / 2;
		}

		if(zoom_value == zoom_val_ti)
			gcfg->video_zoom_type = DISPQT_VIDEOFULLSCREENZOOMTYPE_TOUCHINSIDE;
		else if(zoom_value == zoom_val_to)
			gcfg->video_zoom_type = DISPQT_VIDEOFULLSCREENZOOMTYPE_TOUCHOUTSIDE;
		else if(zoom_value == DISPQT_CONFIG_VIDEOZOOM_CENTER)
			gcfg->video_zoom_type = DISPQT_VIDEOFULLSCREENZOOMTYPE_1_1;

		mpxplay_debugf(DISPQT_DEBUGOUT_FFMPEG, "ZOOM cx:%d cy:%d", video_surface_infos->sws_crop_x, video_surface_infos->sws_crop_y);
	}

	mpxplay_debugf(DISPQT_DEBUGOUT_CROP, "sws_calculate war:%d var:%d n:%d d:%d sw:%d sh:%d ww:%d wh:%d z:%d zw:%d zh:%d cx:%d cy:%d cw:%d ch:%d",
			window_ar, video_ar, video_source_infos->video_aspect_ratio.num, video_source_infos->video_aspect_ratio.den, outframe_width, outframe_height,
			video_surface_infos->sws_dst_width, video_surface_infos->sws_dst_height, zoom_value, zoom_val_ti_w, zoom_val_ti_h, video_surface_infos->sws_crop_x, video_surface_infos->sws_crop_y,
			video_surface_infos->sws_crop_w, video_surface_infos->sws_crop_h);

	// --- target window calculate -----------------------------------------------------------------------------------------

	if(window_width < video_surface_infos->sws_dst_width)
	{
		video_surface_infos->sws_dst_pos_x = (video_surface_infos->sws_dst_width - window_width) / 2;
		video_surface_infos->sws_dst_width = window_width;
		video_surface_infos->videoout_surface_clearpic = true;
	}else{
		video_surface_infos->sws_dst_pos_x = 0;
	}
	if(window_height < video_surface_infos->sws_dst_height)
	{
		video_surface_infos->sws_dst_pos_y = (video_surface_infos->sws_dst_height - window_height) / 2;
		video_surface_infos->sws_dst_height = window_height;
		video_surface_infos->videoout_surface_clearpic = true;
	}else{
		video_surface_infos->sws_dst_pos_y = 0;
	}

	video_surface_infos->videoout_surface_recalc = false;

	return true;
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG
