REM
REM  ANSiMat - Graphical viewer of ANSI files for Linux, Windows and DOS.
REM  Copyright (C) Mateusz Viste 2010
REM
REM This program is free software: you can redistribute it and/or modify
REM it under the terms of the GNU General Public License as published by
REM the Free Software Foundation, either version 3 of the License, or
REM (at your option) any later version.
REM
REM This program is distributed in the hope that it will be useful,
REM but WITHOUT ANY WARRANTY; without even the implied warranty of
REM MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
REM GNU General Public License for more details.
REM
REM You should have received a copy of the GNU General Public License
REM along with this program.  If not, see <http://www.gnu.org/licenses/>.
REM

CONST pVer AS STRING = "0.93"
CONST pDate AS STRING = "2010"
CONST pHomepage AS STRING = "gopher://gopher.viste-family.net/1/projects/ansimat/"
CONST DebugMode = 0    ' 0 = Debug OFF    1 = Debug ON
CONST FlcFPS = 5  ' FPS rate for FLC output

DIM SHARED AS STRING ConfigFile

#IFDEF __FB_LINUX__
  CONST DirSeparator = "/"
  ConfigFile = ENVIRON("HOME") + "/.ansimat.cfg"
#ENDIF
#IFDEF __FB_DOS__
  CONST DirSeparator = "\"
  ConfigFile = EXEPATH + "\ansimat.cfg"
#ENDIF
#IFDEF __FB_WIN32__
  CONST DirSeparator = "\"
  ConfigFile = EXEPATH + "\ansimat.cfg"
  #INCLUDE ONCE "windows.bi"  ' provides FreeConsole()
#ENDIF


TYPE CursorProp
  x AS INTEGER = 1
  y AS INTEGER = 1
  FgColor AS INTEGER = 7
  BgColor AS INTEGER = 0
  SavedPosX AS INTEGER = 0
  SavedPosY AS INTEGER = 0
  Bright AS BYTE = 0
  Faint AS BYTE = 0
  FastBlink AS BYTE = 0
  SlowBlink AS BYTE = 0
  Negative AS BYTE = 0
  Invisible AS BYTE = 0
END TYPE

TYPE MouseProp
  X AS INTEGER
  OldX AS INTEGER
  Y AS INTEGER
  OldY AS INTEGER
  Wheel AS INTEGER
  OldWheel AS INTEGER
  Buttons AS INTEGER
  OldButtons AS INTEGER
END TYPE

DIM SHARED Cursor AS CursorProp
DIM SHARED Mouse AS MouseProp
REDIM SHARED BigScreenBuffer(0 TO 16, 0 TO 16) AS UBYTE ' Init the variable, to avoid compile warnings (will be resized later)
DIM SHARED AS INTEGER TextResX, TextResY, BitRate, ColorDepth, DisplayScale, FullScreenResX, FullScreenResY, DisplayOffsetX, DisplayOffsetY, MaxAnsiHeight
DIM SHARED AS BYTE DraftModeFlag = 0
DIM SHARED AS DOUBLE DraftModeStartTime

#INCLUDE ONCE "font.bi"
#INCLUDE ONCE "ansifunctions.bi"
#INCLUDE ONCE "debuglog.bi"
#INCLUDE ONCE "fexists.bi"
#INCLUDE ONCE "readcfg.bi"
#INCLUDE ONCE "checkvideores.bi"
#INCLUDE ONCE "ansipallette.bi"
#INCLUDE ONCE "loadcustompalette.bi"
#INCLUDE ONCE "inttodoublebyte.bi"
#INCLUDE ONCE "inttodword.bi"
#INCLUDE ONCE "savebmpfile.bi"
#INCLUDE ONCE "saveflcfile.bi"
#INCLUDE ONCE "saveppmfile.bi"
#INCLUDE ONCE "savepcxfile.bi"
#INCLUDE ONCE "savetgafile.bi"
#INCLUDE ONCE "savetiffile.bi"
#INCLUDE ONCE "fbgfx.bi" ' Needed for video mode flags (GFX_WINDOWED, GFX_FULLSCREEN, etc)
#INCLUDE ONCE "refreshscreen.bi"
#INCLUDE ONCE "flushkeyb.bi"
#INCLUDE ONCE "enabledraftmode.bi"

#IF __FB_LANG__ = "fb"
  USING FB  ' Screen mode flags are in the FB namespace in lang FB
#ENDIF

DIM AS INTEGER x, y, z, MaxAnsiRow, MaxAnsiColumn, NeedRefresh, ScreenSizeX, ScreenSizeY
DIM AS UINTEGER ByteCounter
DIM AS STRING AnsiCommand, AnsiFile, OutFile, OutFileFormat, LastKey, CustomPalette
DIM AS UBYTE ByteBuff
DIM AS BYTE LineWraping = 1  ' can be changed by ESC[=7h & ESC[=7l
DIM AS DOUBLE StartTime

#IFDEF __FB_WIN32__  ' Use the GDI driver on Windows instead of DirectX
  SCREENCONTROL(fb.SET_DRIVER_NAME, "GDI")
#ENDIF

AnsiFile = COMMAND(1)
OutFile = COMMAND(2)
IF LEN(COMMAND(3)) > 0 THEN AnsiFile = ""
SELECT CASE LCASE(AnsiFile)
  CASE "/help", "-help", "--help", "-help", "-h", "/?", ""
    PRINT
    PRINT "ANSiMat v" & pVer & " Copyright (C) Mateusz Viste " + pDate
    PRINT
    PRINT "ANSiMat is a graphical viewer of ANSI files. It can also be used to"
    PRINT "convert ANSI files into regular (BMP, PCX...) graphic files or FLC"
    PRINT "animation."
    PRINT
    PRINT "Usage: ansimat ansifile.ans [outputfile.ext]"
    PRINT "If no output file is provided, then ANSiMat will simply display the"
    PRINT "ANSI file on screen. The extension of the output file must be one"
    PRINT "of the following: .bmp, .flc, .pcx, .pnm, .ppm, .tga, .tif"
    PRINT
    PRINT "Check out the ansimat.cfg file for configuration tweaks. On this"
    PRINT "system, the ansimat.cfg file should be at the following location:"
    PRINT " " & ConfigFile
    PRINT
    PRINT "This program is free software: you can redistribute it and/or modify"
    PRINT "it under the terms of the GNU General Public License as published by"
    PRINT "the Free Software Foundation, either version 3 of the License, or"
    PRINT "(at your option) any later version."
    PRINT
    PRINT "ANSiMat homepage: "; pHomepage
    PRINT
    END(1)
END SELECT

IF fexists(AnsiFile) = 0 THEN
  PRINT "File """ + AnsiFile + """ does not exist! Please specify a valid (existing) file, or run 'ansimat -h' for help."
  END(1)
END IF

IF LEN(OutFile) > 0 THEN
  IF fexists(OutFile) = 1 THEN
    PRINT "File """ + OutFile + """ already exists! Operation aborted."
    END(1)
  END IF
  OutFileFormat = UCASE(RIGHT(OutFile, 4))
  SELECT CASE OutFileFormat
    CASE ".BMP"
      OutFileFormat = "BMP"
    CASE ".FLC"
      OutFileFormat = "FLC"
    CASE ".PCX"
      OutFileFormat = "PCX"
    CASE ".PPM", ".PNM"
      OutFileFormat = "PPM"
    CASE ".TGA"
      OutFileFormat = "TGA"
    CASE ".TIF", ".TIFF"
      OutFileFormat = "TIF"
    CASE ELSE
      PRINT "The extension of your outfile is unknown. Please specify a supported extension (.bmp, .flc, .ppm, .pnm, .pcx, .tga, .tif)."
      END(1)
  END SELECT
END IF

#IFNDEF __FB_DOS__   ' Set window's title, if not under DOS
  WINDOWTITLE(MID(AnsiFile, INSTRREV(AnsiFile, DirSeparator) + 1) + " (ANSiMat)")
#ENDIF

DebugLog("Looking after config file at """ + ConfigFile + """...")
IF fexists(ConfigFile) = 1 THEN
    DebugLog("Config file found")
    TextResX = VAL(ReadCFG(ConfigFile, "TextResolutionX"))
    TextResY = VAL(ReadCFG(ConfigFile, "TextResolutionY"))
    BitRate = VAL(ReadCFG(ConfigFile, "DisplayRate"))
    ColorDepth = VAL(ReadCFG(ConfigFile, "ColorDepth"))
    DisplayScale = VAL(ReadCFG(ConfigFile, "GraphicScale"))
    FullScreenResX = VAL(ReadCFG(ConfigFile, "FullScreenResX"))
    FullScreenResY = VAL(ReadCFG(ConfigFile, "FullScreenResY"))
    MaxAnsiHeight = VAL(ReadCFG(ConfigFile, "MaxAnsiHeight"))
    CustomPalette = ReadCFG(ConfigFile, "CustomPalette")
  ELSE
    DebugLog("Config file not found")
    PRINT "No configuration file has been found. Using default values."
    TextResX = 80
    TextResY = 25
    BitRate = 0
    ColorDepth = 16
    DisplayScale = 1
    MaxAnsiHeight = 256
    #IFDEF __FB_DOS__
      FullScreenResX = 640
      FullScreenResY = 480
    #ENDIF
    SLEEP 600, 1
END IF

IF LEN(CustomPalette) > 0 THEN
  SELECT CASE LoadCustomPalette(EXEPATH + DirSeparator + CustomPalette)
    CASE 0 ' palette loaded ok
      DebugLog("External palette loaded ok: " + EXEPATH + DirSeparator + CustomPalette)
    CASE 1 ' file could not be opened
      DebugLog("Palette file could not be opened!")
      PRINT "Error: The custom palette file could not be read (" + EXEPATH + DirSeparator + CustomPalette + ")"
      SLEEP 500, 1
    CASE 2 ' not a valid palette file (must be in JASC format)
      DebugLog("Palette is not a valid format (must be in JASC format)")
      PRINT "The custom palette is not a valid JASC palette! The custom palette will be ignored."
      SLEEP 500, 1
    CASE 3 ' not a 256 color palette
      DebugLog("The palette is not a 256 color palette")
      PRINT "The custom palette is not a 256 color palette! The custom palette will be ignored."
      SLEEP 500, 1
    CASE ELSE
      DebugLog("Unexpected error occured while loading palette!")
  END SELECT
END IF

IF TextResX < 1 OR TextResX > 256 THEN
  TextResX = 80
  PRINT "Invalid TextResolutionX config. Fallback to default (80)."
  SLEEP 500, 1
END IF
IF TextResY < 1 OR TextResY > 256 THEN
  TextResY = 25
  PRINT "Invalid TextResolutionY config. Fallback to default (25)."
  SLEEP 500, 1
END IF
IF MaxAnsiHeight < 25 THEN
  MaxAnsiHeight = 256
  PRINT "Invalid MaxAnsiHeight config. Fallback to default (256)."
  SLEEP 500, 1
END IF

REDIM SHARED BigScreenBuffer(0 TO (TextResX * 8) - 1, 0 TO (MaxAnsiHeight * 16) - 1) AS UBYTE  ' Allocate screen buffer
DebugLog("Allocated BigScreenBuffer OK")

IF LEN(OutFile) = 0 THEN
  IF BitRate < 0 THEN
    BitRate = 0
    PRINT "Invalid BitRate config. Fallback to default (0)."
    SLEEP 500, 1
  END IF
  IF ColorDepth <> 8 AND ColorDepth <> 16 AND ColorDepth <> 24 AND ColorDepth <> 32 THEN
    ColorDepth = 16
    PRINT "Invalid ColorDepth config. Fallback to default (16)."
    SLEEP 500, 1
  END IF

  #IFDEF __FB_DOS__
    IF FullScreenResX < 1 OR FullScreenResY < 1 THEN
      PRINT "Fullscreen resolution is not set properly. Using default resolution (640x480)."
      FullScreenResX = 640
      FullScreenResY = 480
      SLEEP 500, 1
    END IF
  #ENDIF

  IF FullScreenResX < 0 OR FullScreenResY < 0 THEN
    PRINT "Invalid fullscreen configuration. Will run in windowed mode."
    FullScreenResX = 0
    FullScreenResY = 0
    SLEEP 500, 1
  END IF

  IF FullScreenResX > 0 AND FullScreenResY > 0 THEN
      DebugLog("Trying to run full-screen, at " & FullScreenResX & "x" & FullScreenResY & "x" & ColorDepth & "...")
      IF CheckVideoRes(FullScreenResX, FullScreenResY, ColorDepth) = 0 THEN
          DebugLog("Resolution not supported at this color depth!")
          PRINT "The configured fullscreen video mode (" & FullScreenResX & "x" & FullScreenResY & "x" & ColorDepth & ") is not supported by your video adapter. Use another one."
          END(1)
        ELSE
          REM init fullscreen, with 1 video page, do not allow user to switch to windowed mode
          #IFDEF __FB_WIN32__
            FreeConsole()    ' On Windows, get rid of the black console window
          #ENDIF
          SCREENRES FullScreenResX, FullScreenResY, ColorDepth, 1, (GFX_FULLSCREEN OR GFX_NO_SWITCH)
          DebugLog("Fullscreen mode initiated.")
          ScreenSizeX = FullScreenResX
          ScreenSizeY = FullScreenResY
      END IF
    ELSE
      DebugLog("Trying to open a window of " & TextResX * 8 & "x" & TextResY * 16 & "x" & ColorDepth & "...")
      REM init a window with one video page, windowed and do not allow user to switch to fullscreen
      #IFDEF __FB_WIN32__
        FreeConsole()    ' On Windows, get rid of the black console window
      #ENDIF
      SCREENRES TextResX * 8, TextResY * 16, ColorDepth, 1, (GFX_WINDOWED OR GFX_NO_SWITCH)
      DebugLog("Window open.")
      ScreenSizeX = TextResX * 8
      ScreenSizeY = TextResY * 16
  END IF
  IF ColorDepth = 8 THEN
    FOR x = 0 TO 255                                                                                           ' load the
      PALETTE x, (AnsiPallette(x) SHR 16 AND 255), (AnsiPallette(x) SHR 8 AND 255), (AnsiPallette(x) AND 255)  ' ANSI palette
    NEXT x                                                                                                     '
  END IF
END IF
PRINT "Loading the ANSI file..."

StartTime = TIMER

OPEN AnsiFile FOR BINARY AS #1

IF OutFileFormat = "FLC" THEN
  IF BitRate = 0 THEN
    PRINT "Warning: Your BitRate configuration is set to 0. To get a real FLC animation, you should set it to a higher value. See the configuration file for details. Using a fallback value of 14400 now."
    BitRate = 14400
  END IF
  IF FlcFileOpen(OutFile, TextResX * 8, TextResY * 16) > 0 THEN
    PRINT "Unknown error occured when trying to open the destination file."
    CLOSE #1
    END(1)
  END IF
  FlcFileAddBlackFrame()   ' Add black frames
  FOR x = 1 TO FlcFPS \ 2  ' for the first
    FlcFileAddEmptyFrame() ' half second
  NEXT x                   '
END IF

WHILE NOT EOF(1) AND ByteBuff <> 26 AND LastKey <> CHR(255) + "k" AND LastKey <> CHR(27)
  GET #1,, ByteBuff
  IF BitRate > 0 THEN
    ByteCounter += 1
    SELECT CASE OutFileFormat
      CASE ""   '  Slow down read speed (modem emulation)
        WHILE TIMER - StartTime < (8/BitRate * ByteCounter)
          SLEEP 1, 1
        WEND
      CASE "FLC"
        IF FlcFileFramesCounter <= INT(((ByteCounter * 8) / BitRate) * FlcFPS) THEN
          IF MaxAnsiRow > TextResY THEN
              FlcFileAddFrame(0, (MaxAnsiRow - TextResY) * 16, TextResX * 8, TextResY * 16)
            ELSE
              FlcFileAddFrame(0, 0, TextResX * 8, TextResY * 16)
          END IF
        END IF
    END SELECT
  END IF

  IF LEN(AnsiCommand) > 0 OR ByteBuff = 27 THEN ' This is an ANSI command
      AnsiCommand += CHR(ByteBuff)
      SELECT CASE CHR(ByteBuff)
        CASE "H", "f"   ' Move cursor to line x, column y
          x = VAL(GetAnsiParam(AnsiCommand, 2))
          y = VAL(GetAnsiParam(AnsiCommand, 1))
          IF x > 0 THEN Cursor.x = x ELSE Cursor.x = 1
          IF y > 0 THEN Cursor.y = y ELSE Cursor.y = 1
          DebugLog("Got command: H or f / x=" & Cursor.x & " y=" & Cursor.y)
          AnsiCommand = ""
        CASE "A"        ' Move cursor up x lines
          x = VAL(GetAnsiParam(AnsiCommand, 1))
          IF x = 0 THEN x = 1
          Cursor.y -= x
          IF Cursor.y < 1 THEN Cursor.y = 1
          DebugLog("Got command: A / x=" & Cursor.y)
          AnsiCommand = ""
        CASE "B"        ' Move cursor down x lines
          x = VAL(GetAnsiParam(AnsiCommand, 1))
          IF x = 0 THEN x = 1
          Cursor.y += x
          DebugLog("Got command: B / x=" & Cursor.y)
          AnsiCommand = ""
        CASE "C"        ' Move cursor forward x spaces
          x = VAL(GetAnsiParam(AnsiCommand, 1))
          IF x = 0 THEN x = 1
          Cursor.x += x
          DebugLog("Got command: C / x=" & Cursor.x)
          AnsiCommand = ""
        CASE "D"        ' Move cursor back x spaces
          x = VAL(GetAnsiParam(AnsiCommand, 1))
          IF x = 0 THEN x = 1
          Cursor.x -= x
          IF Cursor.x < 1 THEN Cursor.x = 1
          DebugLog("Got command: D / x=" & Cursor.x)
          AnsiCommand = ""
        CASE "h"        ' Sets line wrapping ON
          SELECT CASE GetAnsiParam(AnsiCommand, 1)
            CASE "=7"
              DebugLog("Got command: ESC[=7h / sets line wrapping ON")
              LineWraping = 1
            CASE ELSE
              DebugLog("Got unkown command: ESC[" + GetAnsiParam(AnsiCommand, 1) + "h")
          END SELECT
          AnsiCommand = ""
        CASE "l"        ' Sets line wraping OFF (truncate lines longer than TextResX)
          IF GetAnsiParam(AnsiCommand, 1) = "=7" THEN
              DebugLog("Got command: ESC[=7l / sets line wrapping OFF")
              LineWraping = 0
            ELSE
              DebugLog("Got unkown command: ESC[" + GetAnsiParam(AnsiCommand, 1) + "l")
          END IF
          AnsiCommand = ""
        CASE "s"        ' Saves cursor position for recall later
          Cursor.SavedPosX = Cursor.x
          Cursor.SavedPosY = Cursor.y
          AnsiCommand = ""
        CASE "u"        ' Return to saved cursor position
          IF Cursor.SavedPosX > 0 AND Cursor.SavedPosY > 0 THEN
            Cursor.x = Cursor.SavedPosX
            Cursor.y = Cursor.SavedPosY
          END IF
          AnsiCommand = ""
        CASE "J"        ' Clear screen and home cursor (don't check the optional parameter, it could be 2)
          FOR y = 0 TO MaxAnsiRow * 16
            FOR x = 0 TO MaxAnsiColumn * 8
              BigScreenBuffer(x, y) = 0
            NEXT x
          NEXT y
          Cursor.x = 1
          Cursor.y = 1
          DebugLog("Got command: Clear screen (J)")
          IF BitRate > 0 AND LEN(OutFile) = 0 THEN CLS
          AnsiCommand = ""
        CASE "K"        ' Clear to end/start of line or whole line
          DebugLog("Got command: K")
          SELECT CASE VAL(GetAnsiParam(AnsiCommand, 1))
            CASE 0
              FOR x = Cursor.x TO TextResX
                PrintChar(32, x, Cursor.y)
              NEXT x
            CASE 1
              FOR x = 1 TO Cursor.x
                PrintChar(32, x, Cursor.y)
              NEXT x
            CASE 2
              FOR x = 1 TO TextResX
                PrintChar(32, x, Cursor.y)
              NEXT x
          END SELECT
          AnsiCommand = ""
        CASE "m"        ' Set some display attributes
          DebugLog("Got command: m (change video attrib)...")
          x = 1
          DO   ' Do not use a "WHILE" here, because an empty "ESC[m" means "ESC[0m"
            SELECT CASE VAL(GetAnsiParam(AnsiCommand, x))
              CASE 0  ' normal display
                DebugLog("   ...Normal display")
                Cursor.Bright = 0
                Cursor.Faint = 0
                Cursor.FastBlink = 0
                Cursor.SlowBlink = 0
                Cursor.Negative = 0
                Cursor.Invisible = 0
                Cursor.FgColor = 7 ' not sure if ESC[0m should reset to
                Cursor.BgColor = 0 ' default foreground/background colors...
              CASE 1  ' bold on (or BRIGHT)
                DebugLog("   ...Bright")
                Cursor.Bright = 1
              CASE 2  ' faint on
                DebugLog("   ...Faint")
                Cursor.Faint = 1
              CASE 5  ' blink on (slow)
                DebugLog("   ...Blink (slow)")
                Cursor.SlowBlink = 1
              CASE 6  ' blink on (fast)
                DebugLog("   ...Blink (fast)")
                Cursor.FastBlink = 1
              CASE 7  ' reverse video on
                DebugLog("   ...Reverse")
                Cursor.Negative = 1
              CASE 8  ' nondisplayed (invisible)
                DebugLog("   ...Invisible")
                Cursor.Invisible = 1
              CASE 22 ' normal intensity
                DebugLog("   ...Normal intensity")
                Cursor.Bright = 0
                Cursor.Faint = 0
              CASE 25 ' no-blink
                DebugLog("   ...No blink")
                Cursor.SlowBlink = 0
                Cursor.FastBlink = 0
              CASE 27 ' no-reverse
                DebugLog("   ...No reverse")
                Cursor.Negative = 0
              CASE 30 ' black foreground
                DebugLog("   ...Black foreground")
                Cursor.FgColor = 0
              CASE 31 ' red foreground
                DebugLog("   ...Red foreground")
                Cursor.FgColor = 1
              CASE 32 ' green foreground
                DebugLog("   ...Green foreground")
                Cursor.FgColor = 2
              CASE 33 ' yellow foreground
                DebugLog("   ...Yellow foreground")
                Cursor.FgColor = 3
              CASE 34 ' blue foreground
                DebugLog("   ...Blue foreground")
                Cursor.FgColor = 4
              CASE 35 ' magenta foreground
                DebugLog("   ...Magenta foreground")
                Cursor.FgColor = 5
              CASE 36 ' cyan foreground
                DebugLog("   ...Cyan foreground")
                Cursor.FgColor = 6
              CASE 37 ' white foreground
                DebugLog("   ...White foreground")
                Cursor.FgColor = 7
              CASE 39 ' default foreground
                DebugLog("   ...Default (white) foreground")
                Cursor.FgColor = 7
              CASE 40 ' black background
                DebugLog("   ...Black background")
                Cursor.BgColor = 0
              CASE 41 ' red background
                DebugLog("   ...Red background")
                Cursor.BgColor = 1
              CASE 42 ' green background
                DebugLog("   ...Green background")
                Cursor.BgColor = 2
              CASE 43 ' yellow background
                DebugLog("   ...Yellow background")
                Cursor.BgColor = 3
              CASE 44 ' blue background
                DebugLog("   ...Blue background")
                Cursor.BgColor = 4
              CASE 45 ' magenta background
                DebugLog("   ...Magenta background")
                Cursor.BgColor = 5
              CASE 46 ' cyan background
                DebugLog("   ...Cyan background")
                Cursor.BgColor = 6
              CASE 47 ' white background
                DebugLog("   ...White background")
                Cursor.BgColor = 7
              CASE 49 ' default background
                DebugLog("   ...Default (black) background")
                Cursor.BgColor = 0
            END SELECT
            x += 1
          LOOP UNTIL LEN(GetAnsiParam(AnsiCommand, x)) = 0
          AnsiCommand = ""
        CASE "0","1","2","3","4","5","6","7","8","9",";","[","?",CHR(27)
          REM Just ignore, and wait for the rest of params.
        CASE ELSE
          DebugLog("Got unknow ANSI command: " + AnsiCommand)
          AnsiCommand = ""
      END SELECT
    ELSE  ' Not an ANSI command
      IF ByteBuff = 10 THEN
          Cursor.x = 1
          Cursor.y += 1
        ELSE
          IF ByteBuff <> 13 AND ByteBuff <> 26 THEN ' ignore CR and EOF
            IF Cursor.x <= TextResX THEN  ' Ignore characters out of the line (may happen if LineWraping=0)
              PrintChar(ByteBuff, Cursor.x, Cursor.y)
              IF Cursor.x > MaxAnsiColumn THEN MaxAnsiColumn = Cursor.x
              IF Cursor.y > MaxAnsiRow THEN MaxAnsiRow = Cursor.y
              Cursor.x += 1
            END IF
            IF BitRate > 0 AND LEN(OutFileFormat) = 0 THEN
              IF MaxAnsiRow > TextResY THEN DisplayOffsetY = (MaxAnsiRow * 16) - ScreenSizeY
              IF (TIMER - StartTime - (8/BitRate * ByteCounter)) < 0.5 THEN
                ' Refresh screen if display is not more than 0.5s late (else skip frame)
                RefreshScreen(DisplayOffsetX, DisplayOffsetY, DisplayOffsetX+ScreenSizeX, DisplayOffsetY+ScreenSizeY)
              END IF
            END IF
          END IF
      END IF
      IF Cursor.x > TextResX AND LineWraping = 1 THEN
        Cursor.x = 1
        Cursor.y += 1
      END IF
  END IF
  IF LEN(OutFileFormat) = 0 THEN LastKey = INKEY
WEND
CLOSE #1

REM  Check if user pressed ESC or the close button (only in interactive mode)
IF (LastKey = CHR(27) OR LastKey = CHR(255) + "k") AND LEN(OutFileFormat) = 0 THEN
    CLS
    PRINT "Program aborted by the user."
    SLEEP 500, 1
    FlushKeyb()
    END(1)
  ELSE
    FlushKeyb()
END IF

IF LEN(OutFile) > 0 THEN
    PRINT "Source file (ANSI): " & AnsiFile
    PRINT "Target file (" & OutFileFormat & "):  " & OutFile
    IF OutFileFormat = "FLC" THEN
        PRINT "Emulated text resolution: " & TextResX & "x" & TextResY
        PRINT "Animation's resolution: " & TextResX * 8 & "x" & TextResY * 16
      ELSE
        PRINT "Source resolution (text):  " & MaxAnsiColumn & "x" & MaxAnsiRow
        PRINT "Target resolution (image): " & MaxAnsiColumn * 8 & "x" & MaxAnsiRow * 16
    END IF
    PRINT "Conversion in progress..."
    SELECT CASE OutFileFormat
      CASE "PPM"
        IF SavePpmFile(OutFile, MaxAnsiColumn * 8, MaxAnsiRow * 16) > 0 THEN
          PRINT "An unknown error occured while saving the file. Operation aborted."
          END(1)
        END IF
      CASE "BMP"
        IF SaveBmpFile(OutFile, MaxAnsiColumn * 8, MaxAnsiRow * 16) > 0 THEN
          PRINT "An unknown error occured while saving the file. Operation aborted."
          END(1)
        END IF
      CASE "FLC"
        IF MaxAnsiRow > TextResY THEN ' Add a last frame (to be sure that everything is displayed).
            FlcFileAddFrame(0, (MaxAnsiRow - TextResY) * 16, TextResX * 8, TextResY * 16)
          ELSE
            FlcFileAddFrame(0, 0, TextResX * 8, TextResY * 16)
        END IF
        FOR x = 1 TO FlcFPS * 4 ' add 4 seconds of empty frames
          FlcFileAddEmptyFrame()
        NEXT x
        FlcFileAddBlackFrame()  ' add a black frame (mandatory ring frame!)
        PRINT "Number of frames: " & FlcFileFramesCounter
        PRINT "Animation's duration: " & (FlcFileFramesCounter \ FlcFPS) & " s."
        FlcFileClose()
      CASE "PCX"
        IF SavePcxFile(OutFile, MaxAnsiColumn * 8, MaxAnsiRow * 16) > 0 THEN
          PRINT "An unknown error occured while saving the file. Operation aborted."
          END(1)
        END IF
      CASE "TGA"
        IF SaveTgaFile(OutFile, MaxAnsiColumn * 8, MaxAnsiRow * 16) > 0 THEN
          PRINT "An unknown error occured while saving the file. Operation aborted."
          END(1)
        END IF
      CASE "TIF"
        IF SaveTifFile(OutFile, MaxAnsiColumn * 8, MaxAnsiRow * 16) > 0 THEN
          PRINT "An unknown error occured while saving the file. Operation aborted."
          END(1)
        END IF
      CASE ELSE
        PRINT "UNEXPECTED ERROR: INVALID OUT FORMAT!"
        END(1)
    END SELECT
    PRINT "Job done! (took " & INT((TIMER - StartTime) * 1000) & " ms.)"
  ELSE
    NeedRefresh = 1 ' force a first refresh
    DO
      SLEEP 1,1
      LastKey = INKEY
      SELECT CASE LastKey
        CASE CHR(255) + "H"  ' Up
          IF DisplayOffsetY > 0 THEN
            DisplayOffsetY -= 4
            NeedRefresh = 1
            EnableDraftMode()
          END IF
        CASE CHR(255) + "P"  ' Down
          IF DisplayOffsetY < (MaxAnsiRow * 16) - ScreenSizeY THEN
            DisplayOffsetY += 4
            NeedRefresh = 1
            EnableDraftMode()
          END IF
        CASE CHR(255) + "K"  ' Left
          IF DisplayOffsetX > 0 THEN
            DisplayOffsetX -= 4
            NeedRefresh = 1
            EnableDraftMode()
          END IF
        CASE CHR(255) + "M"  ' Right
          DisplayOffsetX += 4
          NeedRefresh = 1
          EnableDraftMode()
        CASE CHR(255) + "G"  ' Home
          IF DisplayOffsetX > 0 OR DisplayOffsetY > 0 THEN
            DisplayOffsetX = 0
            DisplayOffsetY = 0
            NeedRefresh = 1
            EnableDraftMode()
          END IF
        CASE CHR(255) + "O"  ' End
          DisplayOffsetX = (MaxAnsiColumn * 8) - ScreenSizeX
          DisplayOffsetY = (MaxAnsiRow * 16) - ScreenSizeY
          NeedRefresh = 1
          EnableDraftMode()
        CASE CHR(255) + "I"  ' PgUp
          IF DisplayOffsetY > 0 THEN
            DisplayOffsetY -= ScreenSizeY/1.2
            NeedRefresh = 1
            EnableDraftMode()
          END IF
        CASE CHR(255) + "Q"  ' PgDown
          DisplayOffsetY += ScreenSizeY/1.2
          NeedRefresh = 1
          EnableDraftMode()
      END SELECT

      IF GETMOUSE(Mouse.X, Mouse.Y, Mouse.Wheel, Mouse.Buttons) = 0 THEN ' check Mouse-related stuff
        IF Mouse.Buttons AND 1 = 1 AND Mouse.OldX >= 0 AND Mouse.OldY >= 0 AND Mouse.Y >= 0 AND Mouse.X >= 0 THEN  ' Left click pressed
          IF Mouse.OldY - Mouse.Y <> 0 THEN
            DisplayOffsetY += (Mouse.OldY - Mouse.Y)
            NeedRefresh = 1
            EnableDraftMode()
          END IF
        END IF
        IF Mouse.Wheel - Mouse.OldWheel > 0 AND DisplayOffsetY > 0 THEN
          DisplayOffsetY -= 32
          NeedRefresh = 1
          EnableDraftMode()
        END IF
        IF Mouse.Wheel - Mouse.OldWheel < 0 AND DisplayOffsetY < (MaxAnsiRow * 16) - ScreenSizeY THEN
          DisplayOffsetY += 32
          NeedRefresh = 1
          EnableDraftMode()
        END IF
        Mouse.OldWheel = Mouse.Wheel
        Mouse.OldX = Mouse.X
        Mouse.OldY = Mouse.Y
        Mouse.OldButtons = Mouse.Buttons
      END IF

      IF DisplayOffsetX > (MaxAnsiColumn * 8) - ScreenSizeX THEN DisplayOffsetX = (MaxAnsiColumn * 8) - ScreenSizeX
      IF DisplayOffsetX < 0 THEN DisplayOffsetX = 0
      IF DisplayOffsetY > (MaxAnsiRow * 16) - ScreenSizeY THEN DisplayOffsetY = (MaxAnsiRow * 16) - ScreenSizeY
      IF DisplayOffsetY < 0 THEN DisplayOffsetY = 0
      IF DraftModeFlag = 1 THEN
        IF ABS(TIMER - DraftModeStartTime) > 1 THEN
          DraftModeFlag = 0
          NeedRefresh = 1
        END IF
      END IF
      IF NeedRefresh = 1 THEN
        RefreshScreen(DisplayOffsetX, DisplayOffsetY, DisplayOffsetX+ScreenSizeX, DisplayOffsetY+ScreenSizeY)
        NeedRefresh = 0
        FlushKeyb()   ' Flush keyboard buffer
      END IF
    LOOP UNTIL LastKey = CHR(27) OR LastKey = CHR(255) + "k"
    FlushKeyb()   ' Flush keyboard buffer
END IF

END
