program apm;

{

    This file is the source for a series of routines to detect the make and
    model of the CPU the program is running on.
    Copyright (C) 1998 by Phil Brutsche

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the
    Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA  02111-1307  USA.

}

uses dos, crt;

const
  apm_pmode_16bit = $1;
  apm_pmode_32bit = $2;
  apm_cpu_idle_slows = $4;
  apm_disabled = $8;
  apm_disengaged = $10;
  apm_bios = $0001;
  apm_ac_offline = $00;
  apm_ac_online = $01;
  apm_ac_backup = $02;
  apm_ac_unknown = $FF;
  apm_battery_high = $00;
  apm_battery_low = $01;
  apm_battery_critical = $02;
  apm_battery_charging = $03;
  apm_battery_unknown = $ff;

type
  apm_info_block = record
    major, minor : byte;
    flags : word;
  end;
  apm_power_status_block = record
    ac_status : byte;
    battery_status : byte;
    battery_life_left : byte;
    battery_time_left : byte;
  end;

function apm_cpu_idle : boolean;
var
  regs : registers;
begin
  regs.ah := $53;
  regs.al := $05;
  intr ($15, regs);
  if (regs.flags and fcarry) = 0 then
    apm_cpu_idle := true
  else
    apm_cpu_idle := false;
end;

function apm_cpu_busy : boolean;
var
  regs : registers;
begin
  regs.ah := $53;
  regs.al := $06;
  intr ($15);
  if (regs.flags and fcarry) = 0 then
    apm_cpu_busy := true
  else
    apm_cpu_busy := false;
end;

function apm_ac_strs (status : byte): string;
begin
  case status of
    apm_ac_offline : apm_ac_strs := 'offline';
    apm_ac_online : apm_ac_strs := 'online';
    apm_ac_backup : apm_ac_strs := 'backup power';
    apm_ac_unknown : apm_ac_strs := 'unknown';
  end;
end;

function apm_battery_strs (status : byte) : string;
begin
  case status of
    apm_battery_high : apm_battery_strs := 'high';
    apm_battery_low : apm_battery_strs := 'low';
    apm_battery_critical : apm_battery_strs := 'critical';
    apm_battery_charging : apm_battery_strs := 'charging';
    apm_battery_unknown : apm_battery_strs := 'unknown';
  end;
end;

function apm_detect : boolean;
var
  flags : word;
begin
  asm
    mov ax, $5300
    mov bx, 0
    int $15
    pushf
    pop flags
  end;
  if (flags and fcarry) = 0 then begin
    apm_detect := true;
  end else begin
    apm_detect := false;
  end;
end;

procedure apm_get_info (var info_block : apm_info_block);
var
  regs : registers;
begin
  if apm_detect = false then exit;
  regs.ax := $5300;
  regs.bx := 0;
  intr ($15, regs);
  info_block.major := regs.ah;
  info_block.minor := regs.al;
  info_block.flags := regs.cx;
end;

procedure apm_get_power_status (    device : word;
                                var info_block : apm_power_status_block);
var
  regs : registers;
begin
  if apm_detect = false then exit;
  regs.ax := $530a;
  regs.bx := device;
  intr ($15, regs);
  if (regs.flags and fcarry) <> 0 then exit;
  info_block.ac_status := regs.bh;
  info_block.battery_status := regs.bl;
  info_block.battery_life_left := regs.cl;
  info_block.battery_time_left := regs.dx;
end;

var
  apm_info : apm_info_block;
  apm_power_status : apm_power_status_block;

begin
  clrscr;
  if apm_detect then begin
    apm_get_info (apm_info);
    writeln ('APM version ', apm_info.major, '.', apm_info.minor);
    if (apm_info.flags and apm_pmode_16bit) <> 0 then
      writeln ('  16-bit protected mode interface supported');
    if (apm_info.flags and apm_pmode_32bit) <> 0 then
      writeln ('  32-bit protected mode interface supported');
    if (apm_info.flags and apm_cpu_idle_slows) <> 0 then
      writeln ('  CPU idle call reduces processor speed');
    if (apm_info.flags and apm_disabled) <> 0 then
      writeln ('  BIOS power management disabled');
    if (apm_info.flags and apm_disengaged) <> 0 then
      writeln ('  BIOS power management disengaged (APM v1.1)');
    apm_get_power_status (apm_bios, apm_power_status);
    writeln ('  AC status: ', apm_ac_strs (apm_power_status.ac_status));
    writeln ('  Battery status: ', apm_battery_strs (apm_power_status.battery_status));
    if apm_power_status.battery_life_left = $ff then
      writeln ('  Unknown battery life left')
    else
      writeln ('  Battery life at ', apm_power_status.battery_life_left, '%');
    if apm_power_status.battery_life_left = $ffff then
      writeln ('  Unknown ammount of time left')
    else begin
      write ('  Time Units:  ');
      if (apm_power_status.battery_life_left and $80) = 0 then
        writeln ('seconds')
      else
        writeln ('minutes');
      writeln ('  ', apm_power_status.battery_life_left and $7f, ' units left');
    end;
  end else
    writeln ('No APM');
end.
