// --------------------------------------------------------------------------
//
//  File:      MAP.C
//  Author:    Sulyok Peter (C) 1992,1994.
//  Compiler:  Borland C++ 3.1 (COMPACT model with byte alignment)
//  Notes:     Memory map.
//
//  Jason Hood, 29 September to 9 October, 2002:
//    Stripped out the XMS/EMS code.
//    Tweaked it to how I want it.
//    Added /i interrupt vector list, with (attempted) chaining.
//    Pause after each screen, unless redirected.
//    Added /a high memory area (HMA) list (DOS 7+, free space otherwise).
//    Added /o to find the owner of an address.
//
// --------------------------------------------------------------------------

// Include files ------------------------------------------------------------
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <bios.h>
#include <mem.h>
#include <string.h>
#include <conio.h>
#include <io.h>

// Constants ----------------------------------------------------------------
#define uchar	     unsigned char
#define ushort	     unsigned int
#define ulong	     unsigned long

#define FALSE	     0
#define TRUE	     1

#define MAX_ITEM     150
#define MAX_VEC      20
#define MAX_CHAIN    20

#define MT_NONE      0
#define MT_SYSCODE   1
#define MT_SYSDATA   2
#define MT_PROGRAM   3
#define MT_DEVICE    4
#define MT_ENV	     5
#define MT_DATA      6
#define MT_FREE      7
#define MT_MAP	     8

#define NORMAL	     1
#define UPPER	     2

#define USAGE\
	"MAP 3.0, memory map utility, Jason Hood & Sulyok Peter.\n" \
	"jadoxa@yahoo.com.au   http://map.adoxa.cjb.net/\n" \
	"\n" \
	"Usage:   MAP [-option ...] [name ...]\n" \
	"Options:\n" \
	"         -c        list of programs in conventional memory\n" \
	"         -u        list of programs in upper memory\n" \
	"         -f[c|u]   full list of memory blocks\n" \
	"         -d        list of device drivers\n" \
	"         -i[list]  list of interrupt vectors (in hexadecimal)\n" \
	"         -a        list of HMA\n" \
	"         -o addr   determine the owner of addr\n" \
	"         -h,?      this text"


// Types --------------------------------------------------------------------

// Structure of device driver header.
typedef struct device_header
{
  struct device_header* next;
  ushort attr;
  ushort strategy_rutin;
  ushort interrupt_rutin;
  uchar  name[8];
} DEVICE_HEADER;

// Structure of device driver information.
typedef struct
{
  struct device_header* addr;
  uchar  devname[9];
  uchar  progname[9];
  ushort stgy;
  ushort intr;
  ushort attr;
  uchar  first_drive;
  uchar  last_drive;
} DINFO;

// Structure of DOS DPB.
typedef struct dpb
{
  uchar  drive_num;
  uchar  unit_number;
  ushort bytes_in_sector;
  uchar  lsec_num;
  uchar  log_base;
  ushort reserved_num;
  uchar  FAT_num;
  ushort rootentry_num;
  ushort first_sector;
  ushort largest_cluster;
  ushort sectors_in_FAT;
  ushort root_firstsector;
  DEVICE_HEADER* device_addr;
  uchar  media_desc;
  uchar  block_flag;
  struct dpb* next_dpb;
} DPB;

// Internal structure of DOS DATA blocks.
typedef struct
{
  uchar  type;
  ushort start;
  ushort size;
  uchar  unused[3];
  uchar  name[8];
} SD_HEADER;

// Stucture of MCB header.
typedef struct
{
  uchar  type;
  ushort owner;
  ushort size;
  uchar  unused[3];
  uchar  name[8];
} MCB;

// Structure of programs, device drivers, memory blocks information.
typedef struct
{
  uchar  type;
  ushort seg;
  ushort owner;
  ushort environment;
  uchar  name[10];
  ulong  size;
  ulong  total;
  uchar  vecnum;
  uchar  vectors[MAX_VEC];
} MINFO;

// Structure of interrupt vector chain.
typedef struct
{
  uchar  segnum;
  int	 segments[MAX_CHAIN];
  uchar* pointers[MAX_CHAIN];
} VINFO;

// Structure of free memory.
typedef struct
{
  ulong total;
  ulong largest;
} FREEMEM;

// Structure of DOS 7+ HMA MCB.
typedef struct
{
  uchar  sig[2];		// "MS"
  ushort id;			// ?
  ushort size;			// Bytes
  ushort next;
  uchar  unused[8];
} HMA;


// Local variables ----------------------------------------------------------

static MINFO  mlist[MAX_ITEM];
static ushort mlistnum;
static DINFO  dlist[MAX_ITEM];
static ushort dlistnum;
static VINFO  vlist[256];

static uchar* typenames[] =
{
  "",
  "System code",
  "System data",
  "Program",
  "Device driver",
  "Environment",
  "Data",
  "Free",
};

static uchar   upper_installed;
static ushort  upper_segment;
static ushort  upper_index;
static FREEMEM free_mem[2];

static uchar  default_vectors[] =
{
  0x08, 0x09, 0x10, 0x16, 0x1c, 0x21, 0x2d, 0x2f, 0x33
};
#define default_vectors_num sizeof(default_vectors)

static uchar  thou_sep	= ',';
static uchar  num_lines = 24;
static int    console;


// Local functions ----------------------------------------------------------

static void   check_upper( void );
static uchar  get_upperlink( void );
static int    set_upperlink( uchar );
static int    find_owner( void* );
static void   search_vectors( void );
static void   search_sd( MCB* );
static void   check_name( uchar*, uchar* );
static void   register_mcb( MCB* );
static int    is_psp( MCB* );
static ushort env_seg( MCB* );
static void   calc_free_memory( int, int, FREEMEM* );
static void   make_mcb_list( void );
static void   make_dev_list( void );
static uchar* numsep( uchar*, ulong );
static void   page_break( uchar );
static void   normal_list( uchar );
static void   full_list( int, char*[] );
static void   device_list( void );
static void   intr_list( uchar* );
static void   hma_list( void );
static void   display_owner( uchar* );


// --------------------------------------------------------------------------
//  Name:      get_upperlink
//  Input:     -
//  Output:    uchar		0   separated upper and normal blocks
//				1   linked upper and normal blocks
//  Notes:     Checks the connection of normal and upper memory blocks.
// --------------------------------------------------------------------------
static uchar get_upperlink( void )
{
  asm	mov	ax,0x5802
  asm	int	0x21
  return _AL;
}

// --------------------------------------------------------------------------
//  Name:      set_upperlink
//  Input:     uchar link        0   separate it
//                               1   link it
//  Output:    int               1   successful
//                               0   there is no upper memory
//                              -1   system memory is trashed
//  Notes:     Set the connection between upper and normal memory blocks.
// --------------------------------------------------------------------------
static int set_upperlink( uchar link )
{
  asm	mov	ax,0x5803
  asm	sub	bh,bh
  asm	mov	bl,link
  asm	int	0x21
  asm	jc	_noumb

  return 1;

_noumb:
  asm	cmp	ax,1
  asm	jne	_trash
  return 0;

_trash:
  return -1;
}

// --------------------------------------------------------------------------
//  Name:      check_upper
//  Input:     -
//  Output:    -
//  Notes:     Checks upper memory.
// --------------------------------------------------------------------------
static void check_upper( void )
{
  uchar origlink;

  origlink = get_upperlink();
  switch (set_upperlink( 1 ))
  {
    case 1:  upper_installed = TRUE;  break;
    case 0:  upper_installed = FALSE; break;

    case -1: puts( "SYSTEM MEMORY TRASHED!" );
	     exit( 1 );
	     break;
  }
  set_upperlink( origlink );
}

// --------------------------------------------------------------------------
//  Name:      is_psp
//  Input:     MCB* mcb 	address of the MCB structure
//  Output:    int		TRUE  it is a program
//				FALSE it is a simple MCB block
//  Notes:     Checks the PSP of given MCB block.
// --------------------------------------------------------------------------
static int is_psp( MCB* mcb )
{
  asm	les	bx,mcb
  asm	mov	ax,TRUE
  asm	cmp	word ptr es:[bx+16],0x20CD
  asm	je	__exit
  asm	dec	ax

__exit:
  return _AX;
}

// --------------------------------------------------------------------------
//  Name:      env_seg
//  Input:     MCB* mcb 	address of MCB block
//  Output:    ushort		MCB of environment
//  Notes:     Returns the environment segment of the given MCB block.
// --------------------------------------------------------------------------
static ushort env_seg( MCB* mcb )
{
  asm	les	bx,mcb
  asm	mov	ax,es
  asm	inc	ax
  asm	mov	es,ax
  asm	mov	bx,ax
  asm	mov	ax,es:[0x2C]
  asm	dec	ax
  asm	mov	es,ax
  asm	cmp	es:[1],bx
  asm	je	__exit
  asm	sub	ax,ax

__exit:
  return _AX;
}

// --------------------------------------------------------------------------
//  Name:      find_owner
//  Input:     void* ptr	pointer to memory
//  Output:    int		index into memory info list
//				-1 if not found
//  Notes:     Searches the MCB list to find the owner of ptr.
// --------------------------------------------------------------------------
static int find_owner( void* ptr )
{
  int	i;
  ulong iv, begin, end;

  iv = ((ulong)(FP_SEG( ptr )) << 4) + FP_OFF( ptr );
  for (i = 0; i < mlistnum; ++i)
  {
    begin = (ulong)mlist[i].seg << 4;
    end   = begin + 16;
    if (mlist[i].type != MT_SYSDATA)
      end += mlist[i].size;
    if (iv >= begin && iv < end)
      return i;
  }
  return -1;
}

// --------------------------------------------------------------------------
//  Name:      trace_vector
//  Input:     uchar* ivp	pointer to current vector
//  Output:    uchar*		pointer to the chained vector
//				NULL if the chain ends
//  Notes:     Attempts to trace the chain of an interrupt vector.
// --------------------------------------------------------------------------
static uchar* trace_vector( uchar* ivp )
{
  ushort i;
  ushort op;
  uchar* iv;

  // Search the first 64 bytes for a JMP/CALL FAR.
  for (i = 0; i < 64; ++ivp, ++i)
  {
    op = *ivp;
    if (op == 0xff)
      op = 0xff00U | *(++ivp);
    if (op == 0xea   || op == 0x9a ||	// JMP/CALL far xxxx:xxxx
	op == 0xff2e || op == 0xff1e)	// JMP/CALL far [xxxx]
    {
      if (op > 0xff)
      {
	iv = MK_FP( FP_SEG( ivp ), *((ushort*)(ivp+1)) );
	iv = *((uchar**)iv);
      }
      else
	iv = *((uchar**)(ivp + 1));
      op = *iv;
      // Simple heuristic to determine if the opcode/vector has
      // been found, rather than another opcode's data.
      if ((op >= 0x50 && op <= 0x57) || // PUSH reg
	   op == 0x1e || op == 0x06  || // PUSH DS/ES
	   op == 0x60 || op == 0x9c  || // PUSHA/F
	  (op >= 0xfa && op <= 0xfc) || // CLI/STI/CLD
	  (op >= 0x08 && op <= 0x0b) || // OR reg,reg
	   op == 0x84 || op == 0x85  || // TEST reg,reg
	   op == 0x3c || op == 0x3d  || // CMP AL/X,imm
	  (op >= 0x80 && op <= 0x83  && // CMP reg,imm
	   (*(iv+1) & 0370) == 0370) ||
	   op == 0xeb || op == 0xe9  || // JMP short/near
	   op == 0x2e || op == 0xea  || // CS:/JMP far
	   op == 0xcf)			// IRET
      {
	return iv;
      }
    }
    else if (op == 0xcf)		// To prevent following another intr.
      return NULL;
  }
  return NULL;
}

// --------------------------------------------------------------------------
//  Name:      search_vectors
//  Input:     -
//  Output:    -
//  Notes:     Searches interrupt vectors for the owning memory block.
// --------------------------------------------------------------------------
static void search_vectors( void )
{
  ushort i;
  uchar* ivp;
  VINFO* v;
  MINFO* m;
  int	 j;

  for (v = vlist, i = 0; i < 256; ++v, ++i)
  {
    ivp = *(uchar**)MK_FP( 0, i * 4 );
    while (ivp != NULL)
    {
      j = find_owner( ivp );
      if (v->segnum < MAX_CHAIN &&
	  (v->segnum == 0 || v->segments[v->segnum-1] != j))
      {
	if (j != -1 || v->segnum == 0)
	{
	  v->segments[v->segnum]   = j;
	  v->pointers[v->segnum++] = ivp;
	}
      }
      if (j != -1)
      {
	m = mlist + j;
	if (m->vecnum < MAX_VEC &&
	    (m->vecnum == 0 || m->vectors[m->vecnum-1] != i))
	  m->vectors[m->vecnum++] = i;
      }
      ivp = trace_vector( ivp );
    }
  }
}

// --------------------------------------------------------------------------
//  Name:      search_sd
//  Input:     MCB* mcb 	address of MCB block
//  Output:    -
//  Notes:     Searches the internal parts of a DOS data block.
// --------------------------------------------------------------------------
static void search_sd( MCB* mcb )
{
  ushort end;
  SD_HEADER* sd;
  uchar* name;

  sd  = MK_FP( FP_SEG( mcb ) + 1, 0 );
  end = FP_SEG( mcb ) + mcb->size;
  while ((FP_SEG( sd ) < end) && (mlistnum < MAX_ITEM))
  {
    mlistnum++;
    mlist[mlistnum].seg   = FP_SEG( sd );
    mlist[mlistnum].size  = (ulong)sd->size << 4;
    mlist[mlistnum].total = mlist[mlistnum].size;
    mlist[mlistnum].type  = MT_DATA;
    mlist[mlistnum].name[0] = ' ';
    name = NULL;
    switch (sd->type)
    {
      case 'D': check_name( mlist[mlistnum].name + 1, sd->name );
		mlist[mlistnum].type = MT_DEVICE;
		break;

      case 'I': switch (sd->unused[2])	// MCB offset 7
		{
		  default:  name = "IFS";      break;
		  case 'G': name = "BlkDevTb"; break; // Block device tables
		  case 'H': name = "SectrBuf"; break; // Sector buffer
		  case 'R': name = "EBIOS";    break; // Relocated EBIOS data
		  case 'Y': name = "DriveMap"; break; // Drive map table
		}
		break;

      case 'E': name = "DRVRAPP";  break;       // MEM: EMS
      case 'F': name = "FILES";    break;
      case 'X': name = "FCBS";     break;
      case 'C':                                 // MEM: nothing
      case 'B': name = "BUFFERS";  break;
      case 'L': name = "LASTDRV";  break;
      case 'S': name = "STACKS";   break;
      case 'T': name = "INSTALL";  break;

    //case 'G': name = "BlkDevTb"; break;
    //case 'H': name = "SectrBuf"; break;
    //case 'R': name = "EBIOS";    break;
    //case 'Y': name = "DriveMap"; break;

      default:	name = "??????";   break;
    }
    if (name != NULL)
      strcpy( mlist[mlistnum].name + 1, name );

    sd = MK_FP( sd->start + sd->size, 0 );
  }
}

// --------------------------------------------------------------------------
//  Name:      check_name
//  Input:     uchar* name	place to copy the name
//	       uchar* mname	name of MCB or device driver
//  Output:    -
//  Notes:     Filters name of MCBs and device drivers.
// --------------------------------------------------------------------------
void check_name( uchar* name, uchar* mname )
{
  ushort i;

  for (i = 0; i < 8; ++i)
  {
    if (*mname < ' ')
      break;
    name[i] = *mname++;
  }
  while (i > 0 && name[i-1] == ' ')
    --i;
  name[i] = '\0';
}

// --------------------------------------------------------------------------
//  Name:      register_mcb
//  Input:     MCB* mcb 	address of MCB block
//  Output:    -
//  Notes:     Registers parameters of the given MCB block.
// --------------------------------------------------------------------------
static void register_mcb( MCB* mcb )
{
  mlist[mlistnum].seg	= FP_SEG( mcb );
  mlist[mlistnum].owner = mcb->owner;
  mlist[mlistnum].size	= (ulong)mcb->size << 4;
  mlist[mlistnum].total = mlist[mlistnum].size;

  if (!mcb->owner || mcb->owner == _psp)
  {
    strcpy( mlist[mlistnum].name, "free" );
    mlist[mlistnum].type = MT_FREE;
    // mlist[mlistnum].type = (mcb->owner) ? MT_MAP : MT_FREE;
  }
  else if (mcb->owner == 0x0008)
  {
    strcpy( mlist[mlistnum].name, "DOS" );
    mlist[mlistnum].type = MT_SYSDATA;
    if (mcb->name[0] == 'S')
    {
      if (mcb->name[1] == 'D')
	search_sd( mcb );
      else if (mcb->name[1] == 'C')
      {
	mlist[mlistnum].type = MT_SYSCODE;
	if (mlist[mlistnum].seg == upper_segment)
	  upper_index = mlistnum;
      }
    }
    /*
    else if (mcb->name[0] == 'H')
    {
      strcpy( mlist[mlistnum].name, "LOADHIGH" );
      mlist[mlistnum].type = MT_FREE;
    }
    */
  }
  else
  {
    check_name( mlist[mlistnum].name, mcb->name );
    if (is_psp( mcb ))
    {
      mlist[mlistnum].environment = env_seg( mcb );
      mlist[mlistnum].type = MT_PROGRAM;
    }
    else if (mcb->owner == FP_SEG( mcb ) + 1)
      mlist[mlistnum].type = MT_PROGRAM;
  }
  ++mlistnum;
}

// --------------------------------------------------------------------------
//  Name:      calc_free_memory
//  Input:     int start	the beginning block
//	       int end		the ending block (not included)
//	       FREEMEM* fmem	where to place the calculations
//  Output:    -
//  Notes:     Determines the amount of free memory and the largest block.
//	       Adjacent free blocks are combined.
// --------------------------------------------------------------------------
static void calc_free_memory( int start, int end, FREEMEM* fmem )
{
  int	i;
  ulong free;

  fmem->total = fmem->largest = 0;
  free = 0;
  for (i = start; i < end; ++i)
  {
    if (mlist[i].type == MT_FREE) // || mlist[i].type == MT_MAP)
    {
      if (mlist[i-1].type == MT_FREE) // || mlist[i-1].type == MT_MAP)
	free += mlist[i].size + 16;
      else
      {
	fmem->total += free;
	free = mlist[i].size;
      }
      if (free > fmem->largest)
	fmem->largest = free;
    }
  }
  fmem->total += free;
}

// --------------------------------------------------------------------------
//  Name:      make_mcb_list
//  Input:     -
//  Output:    -
//  Notes:     Makes the list of MCBs.
// --------------------------------------------------------------------------
static void make_mcb_list( void )
{
  ushort i, j;
  MCB*	 cur_mcb;
  uchar  origlink;

  memset( mlist, 0, sizeof( mlist ) );
  mlistnum = 0;

  check_upper();

  asm	mov	ah,0x52
  asm	int	0x21
  asm	mov	ax,es:[bx-2]
  asm	mov	word ptr cur_mcb[2],ax
  asm	mov	word ptr cur_mcb,0

  if (upper_installed)
  {
    origlink = get_upperlink();
    set_upperlink( 1 );
    upper_segment = biosmemory() * (1024 / 16) - 1;
  }

  while (mlistnum < MAX_ITEM)
  {
    register_mcb( cur_mcb );
    if (cur_mcb->type == 'Z')
      break;
    FP_SEG( cur_mcb ) += cur_mcb->size + 1;
  }

  if (upper_installed)
    set_upperlink( origlink );

  if (upper_index == 0)
    upper_index = mlistnum;

  calc_free_memory( 0, upper_index, free_mem );
  calc_free_memory( upper_index, mlistnum, free_mem + 1 );

  for (i = 0; i < mlistnum; i++)
  {
    if (mlist[i].type == MT_PROGRAM)
    {
      for (j = 0; j < mlistnum; j++)
      {
	if ((mlist[j].owner == mlist[i].seg + 1) && j != i)
	{
	  strcpy( mlist[j].name, mlist[i].name );
	  mlist[j].type = (mlist[i].environment == mlist[j].seg) ? MT_ENV
								 : MT_DATA;
	  mlist[i].total += mlist[j].size;
	}
      }
    }
  }

  search_vectors();
}

// --------------------------------------------------------------------------
//  Name:      make_dev_list
//  Input:     -
//  Output:    -
//  Notes:     Makes list of device drivers.
// --------------------------------------------------------------------------
static void make_dev_list( void )
{
  ushort i;
  DEVICE_HEADER* cur_dev;
  DPB*	 cur_dpb;
  int	 j;

  memset( dlist, 0, sizeof( dlist ) );
  dlistnum = 0;

  make_mcb_list();

  asm	mov	ah,0x52
  asm	int	0x21
  asm	add	bx,0x22
  asm	mov	word ptr cur_dev[2],es
  asm	mov	word ptr cur_dev,bx

  while ((FP_OFF( cur_dev ) != 0xFFFF) && (dlistnum < MAX_ITEM))
  {
    dlist[dlistnum].addr = cur_dev;
    dlist[dlistnum].stgy = cur_dev->strategy_rutin;
    dlist[dlistnum].intr = cur_dev->interrupt_rutin;
    dlist[dlistnum].attr = cur_dev->attr;
    check_name( dlist[dlistnum].devname, cur_dev->name );
    cur_dev = cur_dev->next;
    ++dlistnum;
  }

  for (i = 0; i < dlistnum; i++)
  {
    j = find_owner( dlist[i].addr );
    if (j != -1)
      strcpy( dlist[i].progname, mlist[j].name + (mlist[j].name[0] == ' ') );
  }

  asm	mov	ah,0x52
  asm	int	0x21
  asm	les	bx,es:[bx]
  asm	mov	word ptr cur_dpb[2],es
  asm	mov	word ptr cur_dpb,bx

  while (FP_OFF( cur_dpb ) != 0xFFFF)
  {
    for (i = 0; i < dlistnum; i++)
    {
      if (dlist[i].addr == cur_dpb->device_addr)
      {
	if (dlist[i].first_drive == 0)
	  dlist[i].first_drive = cur_dpb->drive_num + 'A';
	else
	  dlist[i].last_drive = cur_dpb->drive_num + 'A';
	break;
      }
    }
    cur_dpb = cur_dpb->next_dpb;
  }

  for (i = 0; i < dlistnum; i++)
  {
    if ((dlist[i].attr & 0x8000) == 0)
    {
      if (dlist[i].first_drive)
      {
	if (dlist[i].last_drive == 0)
	  sprintf( dlist[i].devname, "%c:", dlist[i].first_drive );
	else
	  sprintf( dlist[i].devname, "%c: - %c:",
		   dlist[i].first_drive, dlist[i].last_drive );
      }
      else
	strcpy( dlist[i].devname, "(unused)" );
    }
  }
}


// --------------------------------------------------------------------------
//  Name:      numsep
//  Input:     uchar* buf	buffer to store separated number
//	       ulong  num	the number to separate
//  Output:    uchar*		pointer to buffer
//  Notes:     Converts the number to a thousands-separated string.
//	       num is assumed to be < 1,000,000;
//	       buf is assumed to be at least 8 bytes ("123,567\0").
// --------------------------------------------------------------------------
static uchar* numsep( uchar* buf, ulong num )
{
  if (num < 1000)
    sprintf( buf, "%lu", num );
  else
    sprintf( buf, "%u%c%03u",
	     (ushort)(num / 1000), thou_sep, (ushort)(num % 1000) );

  return buf;
}

// --------------------------------------------------------------------------
//  Name:      page_break
//  Input:     uchar lines    number of lines to display
//  Output:    -
//  Notes:     If the next lines would cause a scroll, pause the listing.
//	       Enter does the next line(s), anything else the next page.
// --------------------------------------------------------------------------
static void page_break( uchar lines )
{
  static uchar displayed = 0;
  uchar key;

  if (!console)
    return;

  if ((displayed += lines) > num_lines)
  {
    fputs( "-- more --", stdout ); //fflush( stdout );
    key = getch();
    if (key == 27 || key == 'q' || key == 'Q')
    {
      putchar( '\n' );
      exit( 0 );
    }
    if (key != 13)
    {
      displayed = 0;
      if (key == 0) getch();
    }
    fputs( "\r          \r", stdout ); //fflush( stdout );
  }
}

// --------------------------------------------------------------------------
//  Name:      normal_list
//  Input:     uchar type	type of list
//  Output:    -
//  Notes:     Displays the normal list of programs.
// --------------------------------------------------------------------------
static void normal_list( uchar type )
{
  ushort i, begin, end;
  uchar  str[4][81], num[8];
  ushort len[2];
  ulong  total = 0;
  ushort blocks = 0;

  make_mcb_list();

  page_break( 3 );
  puts( "Ŀ\n"
	" mcb   para   size      name         type      \n"
	"Ĵ" );

  if (upper_installed)
  {
    begin = (type == UPPER)  ? (upper_index+1) : 0;
    end   = (type == NORMAL) ? upper_index : mlistnum;
  }
  else
  {
    type  = NORMAL;
    begin = 0;
    end   = mlistnum;
  }
  for (i = begin; i < end; i++)
  {
    // Locked-out inter-UMB region.
    if (mlist[i].type == MT_SYSCODE && i >= upper_index)
    {
      page_break( 1 );
      puts( "                                               " );
    }
    else if (mlist[i].type == MT_PROGRAM ||
	     mlist[i].type == MT_DEVICE) //||
	   //mlist[i].type == MT_SYSCODE ||
	   //mlist[i].type == MT_SYSDATA ||
	   //mlist[i].type == MT_FREE)
    {
      page_break( 1 );
      printf( " %04x  %04x  %7s  %-9s  %-13s \n",
	      mlist[i].seg, (ushort)(mlist[i].total >> 4),
	      numsep( num, mlist[i].total ),
	      mlist[i].name + (mlist[i].type == MT_DEVICE),
	      typenames[mlist[i].type] );
      total += mlist[i].total;
      ++blocks;
    }
  }

  page_break( 2 );
  printf( "Ĵ\n"
	  "       %04x  %7s  %-9u  Total         \n",
	  (ushort)(total >> 4), numsep( num, total ), blocks );

  if (type & NORMAL)
  {
    len[0] = strlen( numsep( num, free_mem[0].total ) );
	     sprintf( str[0], "Bytes free: %s", num );
    len[0] = sprintf( str[1], "   Largest: %*s",
		      len[0], numsep( num, free_mem[0].largest ) );
  }
  if (type & UPPER)
  {
    len[1] = strlen( numsep( num, free_mem[1].total ) );
    sprintf( str[2], "%s free: %s", (type & NORMAL) ? "Upper" : "Bytes", num );
    len[1] = sprintf( str[3], "   Largest: %*s",
		      len[1], numsep( num, free_mem[1].largest ) );
  }
  page_break( 4 );
  if (type == NORMAL + UPPER)
  {
    begin = (21 - len[0]) / 2;
    end   = (25 - len[1]) / 2;
    printf( "Ĵ\n"
	    " %*c%-*s  %*c%-*s \n"
	    " %*c%-*s  %*c%-*s \n"
	    "\n",
	    begin, ' ', 21 - begin, str[0], end, ' ', 25 - end, str[2],
	    begin, ' ', 21 - begin, str[1], end, ' ', 25 - end, str[3] );
  }
  else
  {
    begin = (type == NORMAL) ? 0 : 2;
    end   = (49 - len[begin >> 1]) / 2;
    printf( "Ĵ\n"
	    " %*c%-*s \n"
	    " %*c%-*s \n"
	    "\n",
	    end, ' ', 49 - end, str[begin],
	    end, ' ', 49 - end, str[begin+1] );
  }
}

// --------------------------------------------------------------------------
//  Name:      full_list
//  Input:     int names	0 to list all blocks
//				-NORMAL to list all conventional blocks
//				-UPPER to list all upper memory blocks
//				otherwise length of name[]
//  Output:    char* name[]	list of names to display
//  Notes:     Displays full list of memory blocks.
// --------------------------------------------------------------------------
static void full_list( int names, char* name[] )
{
  ushort i, j, k, pos, begin, end;
  uchar  line[81], num[8];
  int	 display = TRUE;
  uchar* mname;
  ulong  total = 0;
  ushort blocks = 0;

  make_mcb_list();

  begin = 0;
  end	= mlistnum;
  if (names > 0)
  {
    j = FALSE;
    for (i = 0; i < mlistnum; i++)
    {
      mname = mlist[i].name + (mlist[i].name[0] == ' ');
      for (k = 0; k < names; ++k)
      {
	if (stricmp( name[k], mname ) == 0)
	{
	  j = TRUE;
	  goto found_name;
	}
      }
    }
  found_name:
    if (!j)
    {
      if (names > 1)
      {
	for (k = 0; k < names; ++k)
	{
	  fputs( strupr( name[k] ), stdout );
	  if (k+1 < names)
	    fputs( ", ", stdout );
	}
	fputs( " are", stdout );
      }
      else
	printf( "%s is", strupr( name[0] ) );
      puts( " not currently in memory." );
      return;
    }
  }
  else
  {
    if (names == -UPPER && upper_installed)
      begin = upper_index + 1;
    else if (names == -NORMAL)
      end = upper_index;
    names = 0;
  }

  page_break( 3 );
  puts(
"Ŀ\n"
" mcb   para   size      name         type       interrupt vectors \n"
"Ĵ" );

  for (i = begin; i < end; i++)
  {
    //if (mlist[i].type == MT_MAP)
    //	display = FALSE;
    //else
    if (names)
    {
      display = FALSE;
      mname = mlist[i].name + (mlist[i].name[0] == ' ');
      for (k = 0; k < names; ++k)
      {
	if (stricmp( name[k], mname ) == 0)
	{
	  display = TRUE;
	  break;
	}
      }
    }
    if (display)
    {
      if (mlist[i].type == MT_SYSCODE && !names && i >= upper_index)
      {
	page_break( 1 );
	puts(
"                                                                  " );
      }
      else
      {
	sprintf( line, " %04x  %04x  %7s  %-9s  %-13s  %17c ",
		 mlist[i].seg, (ushort)(mlist[i].size >> 4),
		 numsep( num, mlist[i].size ),
		 mlist[i].name + (names && mlist[i].name[0] == ' '),
		 typenames[mlist[i].type], ' ' );
	for (j = 1, pos = 54; j <= mlist[i].vecnum; j++)
	{
	  sprintf( &line[pos], "%02x", (int)mlist[i].vectors[j-1] );
	  for (k = j; k < mlist[i].vecnum &&
		      mlist[i].vectors[k] == mlist[i].vectors[k-1] + 1; ++k) ;
	  if (k <= j + 1)
	    line[pos + 2] = ' ';
	  else
	  {
	    line[pos + 2] = '-';
	    j = k - 1;
	  }
	  if (pos == 69 && j != mlist[i].vecnum)
	  {
	    page_break( 1 );
	    puts( line );
	    strcpy( line,
"                                                                  " );
	    pos = 54;
	  }
	  else
	    pos += 3;
	}
	page_break( 1 );
	puts( line );
	total += mlist[i].size;
	++blocks;
      }
    }
  }

  if (!names || blocks == 1)
  {
    page_break( 1 );
    puts(
"" );
  }
  else
  {
    page_break( 3 );
    printf(
"Ĵ\n"
"       %04x  %7s  %-9u  Total                            \n"
"\n"
	    , (ushort)(total >> 4), numsep( num, total ), blocks );
  }
}

// --------------------------------------------------------------------------
//  Name:      device_list
//  Input:     -
//  Output:    -
//  Notes:     Display the list of device drivers.
// --------------------------------------------------------------------------
static void device_list( void )
{
  ushort i;

  make_dev_list();

  page_break( 3 );
  puts( "Ŀ\n"
	"  address   stgy  intr  attr    name    program  \n"
	"Ĵ" );

  for (i = 0; i < dlistnum; i++)
  {
    page_break( 1 );
    printf( " %Fp  %04x  %04x  %04x  %-8s  %-8s \n",
	  dlist[i].addr, dlist[i].stgy, dlist[i].intr,
	  dlist[i].attr, dlist[i].devname, dlist[i].progname );
  }

  page_break( 1 );
  puts( "" );
}

// --------------------------------------------------------------------------
//  Name:      intr_list
//  Input:     uchar* list	list of interrupts to display
//  Output:    -
//  Notes:     Display the list of interrupt vectors.
// --------------------------------------------------------------------------
static void intr_list( uchar* list )
{
  int	 begin, end;
  uchar* bp;
  ushort i, j;
  uchar  ints[256];
  ushort int_num = 0;
  uchar  line[81];
  uchar  chain_length;

  make_mcb_list();

  if (*list == '\0')
  {
    memcpy( ints, default_vectors, default_vectors_num );
    int_num = default_vectors_num;
  }
  else while (*list)
  {
    begin = (uchar)strtol( list, (char**)&bp, 16 );
    if (bp == list)
      break;
    list = bp + 1;
    if (*bp == '-')
    {
      end = (uchar)strtol( list, (char**)&bp, 16 );
      if (bp == list)
	break;
      list = bp + 1;
      for (i = begin; i <= end; ++i)
	ints[int_num++] = i;
    }
    else
      ints[int_num++] = begin;
  }

  chain_length = 1;
  for (i = 0; i < int_num; ++i)
  {
    if (vlist[ints[i]].segnum > chain_length)
      chain_length = vlist[ints[i]].segnum;
  }
  --chain_length;

  memset( line, '', 80 );
  line[80] = '\0';
  bp = line + 80 - (chain_length + 9);

  page_break( 3 );
  printf( "%s%sĿ\n"
	  " i#  %*c%-*s %*c%-*s\n"
	  "%s%sĴ\n",
	  bp, bp + 1,
	  (chain_length + 9 - 7) / 2, ' ',
	  chain_length + 9 - (chain_length + 9 - 7) / 2, "address",
	  (chain_length + 2+8 - 7) / 2, ' ',
	  chain_length + 2+8 - (chain_length + 2+8 - 7) / 2, "program",
	  bp, bp + 1 );

  for (i = 0; i < int_num; ++i)
  {
    page_break( 1 );
    printf( " %02x ", (int)ints[i] );
    end = vlist[ints[i]].segnum;
    for (j = 1; j <= end; ++j)
    {
      begin = vlist[ints[i]].segments[j-1];
      printf( "%*c%-*Fp %*c%-*s \n",
	      j, ' ', 1+chain_length + 9 - j, vlist[ints[i]].pointers[j-1],
	      j, ' ', 1+chain_length + 8 - j, (begin == -1) ? ""
	      : mlist[begin].name + (mlist[begin].name[0] == ' ') );
      if (j < end)
      {
	page_break( 1 );
	fputs( "    ", stdout );
      }
    }
  }

  page_break( 1 );
  printf( "%s%s\n", bp, bp + 1 );
}

// --------------------------------------------------------------------------
//  Name:      hma_list
//  Input:     -
//  Output:    -
//  Notes:     Display the list of HMA blocks (DOS 7+) or free HMA space.
// --------------------------------------------------------------------------
static void hma_list( void )
{
  HMA*	 block;
  ushort space;
  uchar  num[8];

  asm	mov	ax,0x4a04		// Start of HMA chain
  asm	int	0x2f
  asm	mov	word ptr block,di
  asm	mov	word ptr block+2,es
  asm	test	ax,ax			// AX == 0 if supported
  asm	jz	_blocks

  asm	mov	ax,0x4a01		// Query free HMA space
  asm	int	0x2f
  asm	mov	space,bx
  asm	mov	word ptr block,di
  asm	mov	word ptr block+2,es

  if (space == 0)
    puts( "The HMA is not available." );
  else
    printf( "The HMA has %s free bytes, located at %Fp.\n",
	    numsep( num, space ), block );
  return;

_blocks:
  if (FP_SEG( block ) != 0xFFFF)
  {
    puts( "The HMA is not available." );
    return;
  }

  page_break( 3 );
  puts( "Ŀ\n"
	" addr   id   para   size  \n"
	"Ĵ" );

  while (TRUE)
  {
    page_break( 1 );
    if (block->id == 0)
      printf( " %04x  Free  %04x  %6s \n",
	      FP_OFF( block ), block->size >> 4, numsep( num, block->size ) );
    else
      printf( " %04x  %04x  %04x  %6s \n",
	      FP_OFF( block ), block->id,
	      block->size >> 4, numsep( num, block->size ) );
    if (block->next == 0)
      break;
    FP_OFF( block ) = block->next;
  }

  page_break( 1 );
  puts( "" );
}

// --------------------------------------------------------------------------
//  Name:      display_owner
//  Input:     uchar* addr	desired address
//  Output:    -
//  Notes:     Display the owner of address.
// --------------------------------------------------------------------------
static void display_owner( uchar* addr )
{
  uchar* ptr;
  ushort seg, ofs;
  ulong  adr, mcb;
  int	 j;

  if (addr == NULL)
  {
    puts( "Please specify an address." );
    return;
  }

  adr = strtol( addr, (char**)&ptr, 16 );
  if (ptr == addr || (*ptr != '\0' && *ptr != ':'))
  {
    printf( "\"%s\" is not a valid address.\n", addr );
    return;
  }
  if (*ptr == '\0')
  {
    if (adr > 0xFFFFFuL)		// xxxxyyyy
    {
      if (adr == 0x7FFFFFFFuL)		// strtol() doesn't do unsigned
      {
	uchar tmp = addr[4];
	addr[4] = '\0';
	seg = (ushort)strtol( addr, NULL, 16 );
	addr[4] = tmp;
	ofs = (ushort)strtol( addr+4, NULL, 16 );
      }
      else
      {
	seg = (ushort)(adr >> 16);
	ofs = (ushort)adr;
      }
    }
    else if (adr > 0xFFFF)		// flat address
    {
      seg = (ushort)(adr >> 4);
      ofs = (ushort)(adr & 0xF);
    }
    else				// xxxx
    {
      seg = (ushort)adr;
      ofs = 0;
    }
  }
  else					// xxxx:yyyy
  {
    seg = (ushort)adr;
    ofs = (ushort)strtol( ptr+1, NULL, 16 );
  }
  ptr = MK_FP( seg, ofs );

  make_mcb_list();

  j = find_owner( ptr );
  if (j == -1)
  {
    printf( "%Fp is not part of normal memory.\n", ptr );
    return;
  }

  printf( "%Fp is in ", ptr );
  mcb = (ulong)mlist[j].seg << 4;
  adr = ((ulong)FP_SEG( ptr ) << 4) + FP_OFF( ptr );
  if (adr >= mcb && adr < mcb + 16)
    fputs( "the MCB of ", stdout );

  switch (mlist[j].type)
  {
    case MT_FREE: ptr = "free memory.\n";           break;
    case MT_ENV:  ptr = "the environment of %s.\n"; break;
    case MT_DATA: ptr = "the data of %s.\n";        break;
    default:	  ptr = "%s.\n";                    break;
  }
  printf( ptr, mlist[j].name + (mlist[j].name[0] == ' ') );
}

// Main ---------------------------------------------------------------------
int main( int argc, char* argv[] )
{
  struct COUNTRY ci;
  ushort i;
  uchar  opt;

  if (_osmajor < 5)
  {
    puts( "This program requires DOS 5.x or higher." );
    return 1;
  }

  if (country( 0, &ci ))
    thou_sep = ci.co_thsep[0];

  console = isatty( 1 );
  if (console)
    num_lines = *(uchar*)0x0484;

  if (argc > 1)
  {
    for (i = 1; i < argc; i++)
    {
      if ((argv[i][0] == '-') || (argv[i][0] == '/'))
      {
	opt = argv[i][1];
	if (opt >= 'A' && opt <= 'Z')
	  opt |= 32;
	switch (opt)
	{
	  case 'c': normal_list( NORMAL );    break;
	  case 'u': normal_list( UPPER );     break;

	  case 'f': opt = argv[i][2];
		    if (opt >= 'A' && opt <= 'Z')
		      opt |= 32;
		    full_list( (opt == 'c') ? -NORMAL :
			       (opt == 'u') ? -UPPER  : 0, NULL );
		    break;

	  case 'd': device_list();            break;
	  case 'i': intr_list( argv[i] + 2 ); break;
	  case 'a': hma_list();               break;

	  case 'o': if (argv[i][2])
		      display_owner( argv[i] + 2 );
		    else
		      display_owner( argv[++i] );
		    break;

	  case 'h':
	  case '?': puts( USAGE );
		    return 0;

	  default:  printf( "Invalid option %s (use -h for help).\n", argv[i] );
		    return 1;
	}
      }
      else
      {
	full_list( argc - i, argv + i );
	break;
      }
    }
  }
  else
    normal_list( NORMAL + UPPER );

  return 0;
}

// End ----------------------------------------------------------------------
